import React, { useState, useRef } from "react";
import { Send, Paperclip, X } from "lucide-react";

export default function MessageInput({ onSendMessage, disabled = false }) {
    const [message, setMessage] = useState("");
    const [selectedFile, setSelectedFile] = useState(null);
    const [isTyping, setIsTyping] = useState(false);
    const fileInputRef = useRef(null);

    const handleSubmit = (e) => {
        e.preventDefault();
        if ((!message.trim() && !selectedFile) || disabled) return;

        onSendMessage(message.trim(), selectedFile);
        setMessage("");
        setSelectedFile(null);
        setIsTyping(false);
    };

    const handleFileSelect = (e) => {
        const file = e.target.files[0];
        if (file) {
            // Check file size (max 10MB)
            if (file.size > 10 * 1024 * 1024) {
                alert("File size must be less than 10MB");
                return;
            }

            // Check file type
            const allowedTypes = [
                'image/jpeg',
                'image/png',
                'image/gif',
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'text/plain'
            ];

            if (!allowedTypes.includes(file.type)) {
                alert("File type not supported. Please upload images, PDFs, Word documents, or text files.");
                return;
            }

            setSelectedFile(file);
        }
    };

    const removeFile = () => {
        setSelectedFile(null);
        if (fileInputRef.current) {
            fileInputRef.current.value = "";
        }
    };

    const handleKeyPress = (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            handleSubmit(e);
        }
    };

    return (
        <div className="border-t border-gray-200 dark:border-gray-700 p-4">
            {/* File Preview */}
            {selectedFile && (
                <div className="mb-3 p-2 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-between">
                    <div className="flex items-center space-x-2">
                        <Paperclip className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-700 dark:text-gray-300">
                            {selectedFile.name}
                        </span>
                    </div>
                    <button
                        onClick={removeFile}
                        className="p-1 hover:bg-gray-200 dark:hover:bg-gray-600 rounded"
                    >
                        <X className="w-3 h-3 text-gray-500" />
                    </button>
                </div>
            )}

            <form onSubmit={handleSubmit} className="flex items-end space-x-2">
                {/* File Upload Button */}
                <button
                    type="button"
                    onClick={() => fileInputRef.current?.click()}
                    disabled={disabled}
                    className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors disabled:opacity-50"
                >
                    <Paperclip className="w-5 h-5" />
                </button>

                {/* Hidden File Input */}
                <input
                    ref={fileInputRef}
                    type="file"
                    onChange={handleFileSelect}
                    className="hidden"
                    accept="image/*,.pdf,.doc,.docx,.txt"
                />

                {/* Message Input */}
                <div className="flex-1 relative">
                    <textarea
                        value={message}
                        onChange={(e) => {
                            setMessage(e.target.value);
                            setIsTyping(e.target.value.length > 0);
                        }}
                        onKeyPress={handleKeyPress}
                        placeholder="Type your message..."
                        disabled={disabled}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg resize-none focus:ring-2 focus:ring-cyan-500 focus:border-transparent dark:bg-gray-700 dark:text-white disabled:opacity-50"
                        rows="1"
                        style={{ minHeight: '40px', maxHeight: '120px' }}
                    />
                </div>

                {/* Send Button */}
                <button
                    type="submit"
                    disabled={(!message.trim() && !selectedFile) || disabled}
                    className="p-2 bg-gradient-to-r from-cyan-600 to-indigo-800 hover:from-cyan-700 hover:to-indigo-900 text-white rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                    <Send className="w-5 h-5" />
                </button>
            </form>
        </div>
    );
} 