import React from "react";
import { User, FileText, Download } from "lucide-react";

export default function MessageList({ messages, currentUser, messagesEndRef }) {
    const formatTime = (timestamp) => {
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    };

    // Get display name for message sender based on user roles
    const getDisplayName = (message) => {
        // If current user is client or writer and message is from admin/super_admin, show "Support"
        if ((currentUser.user_type === 'client' || currentUser.user_type === 'writer') && 
            (message.sender_type === 'admin' || message.sender_type === 'super_admin')) {
            return 'Support';
        }
        
        // Otherwise show the actual name
        return message.sender_name || 'Support';
    };

    const isOwnMessage = (message) => {
        return message.sender_id === currentUser.id;
    };

    const renderMessageContent = (message) => {
        if (message.message_type === 'file') {
            return (
                <div className="flex items-center space-x-2 p-2 bg-gray-100 dark:bg-gray-700 rounded">
                    <FileText className="w-4 h-4 text-gray-500" />
                    <span className="text-sm text-gray-700 dark:text-gray-300">
                        {message.file_name || 'File'}
                    </span>
                    <button
                        onClick={() => window.open(message.file_path, '_blank')}
                        className="p-1 hover:bg-gray-200 dark:hover:bg-gray-600 rounded"
                    >
                        <Download className="w-3 h-3" />
                    </button>
                </div>
            );
        }

        return (
            <p className="text-sm text-gray-900 dark:text-gray-100 whitespace-pre-wrap">
                {message.content}
            </p>
        );
    };

    if (messages.length === 0) {
        return (
            <div className="flex-1 flex flex-col items-center justify-center text-gray-500 p-4">
                <div className="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mb-4">
                    <User className="w-8 h-8 text-gray-400" />
                </div>
                <p className="text-center">No messages yet</p>
                <p className="text-sm text-center mt-2">
                    Start the conversation by sending a message
                </p>
            </div>
        );
    }

    return (
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
            {messages.map((message) => (
                <div
                    key={message.id}
                    className={`flex ${isOwnMessage(message) ? 'justify-end' : 'justify-start'}`}
                >
                    <div
                        className={`max-w-xs lg:max-w-md ${
                            isOwnMessage(message)
                                ? 'bg-gradient-to-r from-cyan-600 to-indigo-800 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-900 dark:text-gray-100'
                        } rounded-lg px-4 py-2 shadow-sm`}
                    >
                        {!isOwnMessage(message) && (
                            <div className="flex items-center space-x-2 mb-1">
                                <div className="w-6 h-6 bg-gradient-to-r from-cyan-600 to-indigo-800 rounded-full flex items-center justify-center">
                                    <User className="w-3 h-3 text-white" />
                                </div>
                                <span className="text-xs font-medium opacity-75">
                                    {getDisplayName(message)}
                                </span>
                            </div>
                        )}
                        
                        {renderMessageContent(message)}
                        
                        <div className={`text-xs mt-1 ${
                            isOwnMessage(message) ? 'text-white/70' : 'text-gray-500 dark:text-gray-400'
                        }`}>
                            {formatTime(message.created_at)}
                            {message.read_at && isOwnMessage(message) && (
                                <span className="ml-2">✓ Read</span>
                            )}
                        </div>
                    </div>
                </div>
            ))}
            <div ref={messagesEndRef} />
        </div>
    );
} 