import { useState, useEffect } from "react";
import { Link, usePage } from "@inertiajs/react";
import { Menu, X, ChevronDown, Sun, Moon, ChevronRight } from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";
import { useTheme } from "@/Contexts/ThemeContext";

export default function NavBar() {
    const { auth } = usePage().props;
    const [isOpen, setIsOpen] = useState(false);
    const [navbarBg, setNavbarBg] = useState(false);
    const [isDropdownOpen, setIsDropdownOpen] = useState(false);
    const [isMobileServicesOpen, setIsMobileServicesOpen] = useState(false);
    const [isMobileAboutOpen, setIsMobileAboutOpen] = useState(false);

    const { darkMode, toggleDarkMode } = useTheme();

    // Add scroll listener for navbar background
    useEffect(() => {
        const handleScroll = () => {
            if (window.scrollY > 10) {
                setNavbarBg(true);
            } else {
                setNavbarBg(false);
            }
        };

        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, []);

    // Dropdown animation variants
    const dropdownVariants = {
        hidden: { opacity: 0, y: -5, height: 0 },
        visible: {
            opacity: 1,
            y: 0,
            height: "auto",
            transition: {
                duration: 0.2,
                ease: "easeOut",
            },
        },
        exit: {
            opacity: 0,
            y: -5,
            height: 0,
            transition: {
                duration: 0.2,
                ease: "easeIn",
            },
        },
    };

    return (
        <div>
            <nav
                className={`fixed w-full z-50 transition-all duration-300 ${
                    navbarBg
                        ? "bg-gray-200 dark:bg-gray-900 shadow-md py-2"
                        : "bg-gradient-to-b from-blue-100 to-white dark:bg-none dark:bg-gray-900 py-4"
                }`}
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex justify-between items-center">
                        {/* ✅ UPDATED: Logo Section - PNG Logo Implementation */}
                        <div className="flex-shrink-0">
                            <Link
                                href="/"
                                className="flex items-center"
                                aria-label="AcademicScribe Home"
                            >
                                {/* Logo Image with WebP Support and Dark Mode */}
                                <img
                                    src="/images/logo.webp"
                                    alt="AcademicScribe - Professional Academic Writing Service"
                                    width="140"
                                    height="32"
                                    className={`h-6 sm:h-7 md:h-8 w-auto transition-all duration-300 ${
                                        navbarBg
                                            ? "h-6 sm:h-6 md:h-7"
                                            : "h-6 sm:h-7 md:h-8"
                                    }`}
                                    style={{
                                        filter: darkMode
                                            ? "brightness(0) saturate(100%) invert(42%) sepia(93%) saturate(1352%) hue-rotate(204deg) brightness(102%) contrast(97%)"
                                            : "none",
                                    }}
                                    onError={(e) => {
                                        // Fallback to PNG if WebP fails
                                        e.target.src = "/images/logo.png";
                                        e.target.onerror = () => {
                                            // Final fallback to text logo
                                            e.target.style.display = "none";
                                            e.target.nextElementSibling.style.display = "block";
                                        };
                                    }}
                                />

                                {/* Fallback Text Logo (hidden by default) */}
                                <h1
                                    className="text-xl sm:text-xl md:text-2xl font-bold text-blue-800 dark:text-blue-400 transition-all duration-300 hidden"
                                    style={{ display: "none" }}
                                >
                                    AcademicScribe
                                </h1>
                            </Link>
                        </div>

                        {/* Desktop Menu Items (Center) - Now only visible on lg screens and up */}
                        <div className="hidden lg:flex items-center justify-center space-x-8">
                            <Link
                                href="/"
                                className="text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                aria-label="Home"
                            >
                                Home
                            </Link>
                            <Link
                                href="/services/essay-writing"
                                className="text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                aria-label="Essays"
                            >
                                Essay Writing
                            </Link>

                            {/* Services Dropdown */}
                            <div className="relative group">
                                <button
                                    className="flex items-center space-x-1 text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors group-hover:text-blue-600 dark:group-hover:text-blue-400"
                                    aria-label="Services"
                                    aria-expanded="false"
                                    aria-haspopup="true"
                                >
                                    <span>Services</span>
                                    <ChevronDown className="h-4 w-4 transition-transform group-hover:rotate-180" />
                                </button>

                                {/* Mega Menu - 3 Column Layout */}
                                <div className="absolute left-1/2 transform -translate-x-1/2 mt-2 w-[800px] rounded-lg shadow-xl bg-white dark:bg-gray-800 ring-1 ring-black ring-opacity-5 dark:ring-gray-700 z-50 transform opacity-0 scale-95 invisible group-hover:opacity-100 group-hover:scale-100 group-hover:visible transition-all duration-200 origin-top">
                                    <div className="p-6" role="menu">
                                        {/* Header */}
                                        <div className="mb-4 pb-4 border-b border-gray-200 dark:border-gray-700">
                                            <Link
                                                href="/services"
                                                className="text-base font-bold text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 transition-colors inline-flex items-center"
                                                role="menuitem"
                                                aria-label="View All Services"
                                            >
                                                📋 View All Services
                                                <ChevronRight className="ml-2 h-4 w-4" />
                                            </Link>
                                        </div>

                                        {/* 4 Column Grid - Balanced Layout */}
                                        <div className="grid grid-cols-4 gap-6">
                                            {/* Column 1: Academic Writing */}
                                            <div>
                                                <h3 className="text-sm font-bold text-gray-900 dark:text-gray-100 uppercase tracking-wide mb-4 pb-2 border-b border-gray-200 dark:border-gray-700">
                                                    Academic Writing
                                                </h3>
                                                <ul className="space-y-2">
                                                    <li>
                                                        <Link
                                                            href="/services/essay-writing"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Essay Writing"
                                                        >
                                                            <span className="mr-2 text-lg">✍️</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Essay Writing</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/write-essays-for-money"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Write Essays for Money"
                                                        >
                                                            <span className="mr-2 text-lg">💵</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Write Essays for Money</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/dissertation-writing"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Dissertations"
                                                        >
                                                            <span className="mr-2 text-lg">📚</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Dissertations</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/research-papers"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Research Papers"
                                                        >
                                                            <span className="mr-2 text-lg">🔬</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Research Papers</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/pay-for-research-paper"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Pay for Research Paper"
                                                        >
                                                            <span className="mr-2 text-lg">📄</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Pay for Research Paper</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/thesis-proposals"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Thesis Proposals"
                                                        >
                                                            <span className="mr-2 text-lg">🎯</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Thesis Proposals</span>
                                                        </Link>
                                                    </li>
                                                </ul>
                                            </div>

                                            {/* Column 2: Specialized Help */}
                                            <div>
                                                <h3 className="text-sm font-bold text-gray-900 dark:text-gray-100 uppercase tracking-wide mb-4 pb-2 border-b border-gray-200 dark:border-gray-700">
                                                    Specialized Help
                                                </h3>
                                                <ul className="space-y-2">
                                                    <li>
                                                        <Link
                                                            href="/services/nursing-paper-writing-service"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Nursing Papers"
                                                        >
                                                            <span className="mr-2 text-lg">🏥</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Nursing Papers</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/case-studies"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Case Studies"
                                                        >
                                                            <span className="mr-2 text-lg">📊</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Case Studies</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/literature-reviews"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Literature Reviews"
                                                        >
                                                            <span className="mr-2 text-lg">📖</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Literature Reviews</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/personal-statements"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Personal Statements"
                                                        >
                                                            <span className="mr-2 text-lg">📝</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Personal Statements</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/powerpoint-presentation"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="PowerPoint Presentations"
                                                        >
                                                            <span className="mr-2 text-lg">📊</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">PowerPoint</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/powerpoint-presentations-writing-service"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="PPT Writing Service"
                                                        >
                                                            <span className="mr-2 text-lg">🎨</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">PPT Writing Service</span>
                                                        </Link>
                                                    </li>
                                                </ul>
                                            </div>

                                            {/* Column 3: Homework & Assignments */}
                                            <div>
                                                <h3 className="text-sm font-bold text-gray-900 dark:text-gray-100 uppercase tracking-wide mb-4 pb-2 border-b border-gray-200 dark:border-gray-700">
                                                    Homework & Classes
                                                </h3>
                                                <ul className="space-y-2">
                                                    <li>
                                                        <Link
                                                            href="/services/assignments"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Assignments"
                                                        >
                                                            <span className="mr-2 text-lg">📝</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Assignments</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/pay-for-homework"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Pay for Homework"
                                                        >
                                                            <span className="mr-2 text-lg">📚</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Pay for Homework</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/do-my-math-homework"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Math Homework Help"
                                                        >
                                                            <span className="mr-2 text-lg">🔢</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Math Homework</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/finance-homework-help"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Finance Homework"
                                                        >
                                                            <span className="mr-2 text-lg">💰</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Finance Homework</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/excel-help"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Excel Help"
                                                        >
                                                            <span className="mr-2 text-lg">📈</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Excel Help</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/spss-assignment-help"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="SPSS Help"
                                                        >
                                                            <span className="mr-2 text-lg">📊</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">SPSS Help</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/pay-someone-to-do-my-online-class"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Do My Online Class"
                                                        >
                                                            <span className="mr-2 text-lg">💻</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Do My Online Class</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/programming-assignments"
                                                            className="flex items-center px-3 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400 rounded-md transition-all duration-150 group"
                                                            role="menuitem"
                                                            aria-label="Programming"
                                                        >
                                                            <span className="mr-2 text-lg">💻</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Programming</span>
                                                        </Link>
                                                    </li>
                                                </ul>
                                            </div>

                                            {/* Column 4: Popular / Featured */}
                                            <div>
                                                <h3 className="text-sm font-bold text-gray-900 dark:text-gray-100 uppercase tracking-wide mb-4 pb-2 border-b border-gray-200 dark:border-gray-700">
                                                    Popular Services
                                                </h3>
                                                <ul className="space-y-2">
                                                    <li>
                                                        <Link
                                                            href="/services/affordable-essay-writing"
                                                            className="flex items-center px-3 py-2.5 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-700 dark:hover:text-blue-300 rounded-md transition-all duration-150 group bg-blue-50/50 dark:bg-blue-900/20"
                                                            role="menuitem"
                                                            aria-label="Affordable Essays"
                                                        >
                                                            <span className="mr-2 text-lg">💰</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Affordable Essays</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/buy-essays-online"
                                                            className="flex items-center px-3 py-2.5 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-700 dark:hover:text-blue-300 rounded-md transition-all duration-150 group bg-blue-50/50 dark:bg-blue-900/20"
                                                            role="menuitem"
                                                            aria-label="Buy Essays Online"
                                                        >
                                                            <span className="mr-2 text-lg">🛒</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Buy Essays Online</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/cheap-essays-online"
                                                            className="flex items-center px-3 py-2.5 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-700 dark:hover:text-blue-300 rounded-md transition-all duration-150 group bg-blue-50/50 dark:bg-blue-900/20"
                                                            role="menuitem"
                                                            aria-label="Cheap Essays"
                                                        >
                                                            <span className="mr-2 text-lg">💵</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Cheap Essays</span>
                                                        </Link>
                                                    </li>
                                                    <li>
                                                        <Link
                                                            href="/services/buy-argumentative-essays"
                                                            className="flex items-center px-3 py-2.5 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-700 dark:hover:text-blue-300 rounded-md transition-all duration-150 group bg-blue-50/50 dark:bg-blue-900/20"
                                                            role="menuitem"
                                                            aria-label="Argumentative Essays"
                                                        >
                                                            <span className="mr-2 text-lg">💭</span>
                                                            <span className="group-hover:translate-x-1 transition-transform inline-block">Argumentative Essays</span>
                                                        </Link>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* About Us Dropdown */}
                            <div className="relative group">
                                <button
                                    className="flex items-center space-x-1 text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors group-hover:text-blue-600 dark:group-hover:text-blue-400"
                                    aria-label="About Us"
                                    aria-expanded="false"
                                    aria-haspopup="true"
                                >
                                    <span>About Us</span>
                                    <ChevronDown className="h-4 w-4 transition-transform group-hover:rotate-180" />
                                </button>

                                {/* About Us Dropdown Menu */}
                                <div className="absolute left-0 mt-2 w-56 rounded-md shadow-lg bg-white dark:bg-gray-800 ring-1 ring-black ring-opacity-5 dark:ring-gray-700 z-50 transform opacity-0 scale-95 invisible group-hover:opacity-100 group-hover:scale-100 group-hover:visible transition-all duration-200 origin-top-left">
                                    <div className="py-2" role="menu">
                                        <Link
                                            href="/about"
                                            className="block px-4 py-3 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 border-b border-gray-100 dark:border-gray-700"
                                            role="menuitem"
                                            aria-label="About Us"
                                        >
                                            ℹ️ About Us
                                        </Link>

                                        <Link
                                            href="/writers"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Our Writers"
                                        >
                                            ✍️ Our Writers
                                        </Link>

                                        <Link
                                            href="/guarantees"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Guarantees"
                                        >
                                            🛡️ Guarantees
                                        </Link>

                                        <Link
                                            href="/reviews"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Reviews"
                                        >
                                            ⭐ Reviews
                                        </Link>

                                        <Link
                                            href="/pricing-page"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Pricing"
                                        >
                                            💰 Pricing
                                        </Link>

                                        <div className="border-t border-gray-100 dark:border-gray-700 my-1"></div>

                                        <Link
                                            href="/privacy-policy"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Privacy Policy"
                                        >
                                            🔒 Privacy Policy
                                        </Link>

                                        <Link
                                            href="/terms-of-service"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Terms of Service"
                                        >
                                            📋 Terms of Service
                                        </Link>
                                        <Link
                                            href="/cookie-policy"
                                            className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 hover:text-blue-600 dark:hover:text-blue-400"
                                            role="menuitem"
                                            aria-label="Cookie Policy"
                                        >
                                            🍪 Cookie Policy
                                        </Link>
                                    </div>
                                </div>
                            </div>

                            <Link
                                href="/blog"
                                className="text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                aria-label="Blog"
                            >
                                Blog
                            </Link>

                            <Link
                                href="/contact"
                                className="text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                aria-label="Contact"
                            >
                                Contact
                            </Link>
                            <Link
                                href="/place-order"
                                className="text-white bg-orange-500 hover:bg-orange-600 dark:bg-orange-600 dark:hover:bg-orange-700 px-4 py-2 rounded-lg transition-colors font-medium shadow-sm"
                                aria-label="Place Order"
                            >
                                Place Order
                            </Link>
                        </div>

                        {/* Login/Register/Profile Buttons and Dark Mode Toggle (Right) - Now only visible on lg screens and up */}
                        <div className="hidden lg:flex items-center space-x-4">
                            {/* Dark Mode Toggle */}
                            <button
                                onClick={toggleDarkMode}
                                className="p-2 rounded-full text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors"
                                aria-label="Toggle dark mode"
                            >
                                {darkMode ? (
                                    <Sun className="h-5 w-5" />
                                ) : (
                                    <Moon className="h-5 w-5" />
                                )}
                            </button>

                            {!auth.user ? (
                                <>
                                    <Link
                                        href="/login"
                                        className="px-4 py-2 rounded-md text-blue-600 dark:text-blue-400 border border-blue-600 dark:border-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 transition-colors"
                                        aria-label="Login"
                                    >
                                        Login
                                    </Link>
                                    <Link
                                        href="/register"
                                        className="px-4 py-2 rounded-md bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-800 transition-colors"
                                        aria-label="Register"
                                    >
                                        Register
                                    </Link>
                                </>
                            ) : (
                                <div className="relative">
                                    <button
                                        onClick={() =>
                                            setIsDropdownOpen(!isDropdownOpen)
                                        }
                                        className="flex items-center space-x-2 px-4 py-2 rounded-md text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30 transition-colors"
                                        aria-label="Profile"
                                    >
                                        <span>{auth.user.name}</span>
                                        <ChevronDown className="h-4 w-4" />
                                    </button>

                                    <AnimatePresence>
                                        {isDropdownOpen && (
                                            <motion.div
                                                className="absolute right-0 mt-2 w-48 rounded-md shadow-lg bg-white dark:bg-gray-800 ring-1 ring-black ring-opacity-5 dark:ring-gray-700"
                                                initial="hidden"
                                                animate="visible"
                                                exit="exit"
                                                variants={dropdownVariants}
                                            >
                                                <div
                                                    className="py-1"
                                                    role="menu"
                                                >
                                                    <Link
                                                        href="/dashboard"
                                                        className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30"
                                                        role="menuitem"
                                                    >
                                                        Dashboard
                                                    </Link>
                                                    <Link
                                                        href="/profile"
                                                        className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/30"
                                                        role="menuitem"
                                                        aria-label="Profile"
                                                    >
                                                        Profile
                                                    </Link>
                                                    <Link
                                                        href="/logout"
                                                        method="post"
                                                        as="button"
                                                        className="block w-full text-left px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
                                                        role="menuitem"
                                                        aria-label="Logout"
                                                    >
                                                        Logout
                                                    </Link>
                                                </div>
                                            </motion.div>
                                        )}
                                    </AnimatePresence>
                                </div>
                            )}
                        </div>

                        {/* Mobile menu button and Dark Mode Toggle - Now visible on all screens below lg */}
                        <div className="lg:hidden flex items-center space-x-2">
                            {/* Mobile Dark Mode Toggle */}
                            <button
                                onClick={toggleDarkMode}
                                className="p-2 rounded-full text-gray-500 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 focus:outline-none"
                                aria-label="Toggle dark mode"
                            >
                                {darkMode ? (
                                    <Sun className="h-5 w-5" />
                                ) : (
                                    <Moon className="h-5 w-5" />
                                )}
                            </button>

                            <button
                                onClick={() => setIsOpen(!isOpen)}
                                className="text-gray-500 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 focus:outline-none"
                                aria-label="Toggle mobile menu"
                            >
                                {isOpen ? (
                                    <X className="h-6 w-6" />
                                ) : (
                                    <Menu className="h-6 w-6" />
                                )}
                            </button>
                        </div>
                    </div>
                </div>

                {/* Enhanced Mobile Menu with improved UI */}
                <AnimatePresence>
                    {isOpen && (
                        <motion.div
                            initial={{ opacity: 0, height: 0 }}
                            animate={{ opacity: 1, height: "auto" }}
                            exit={{ opacity: 0, height: 0 }}
                            transition={{ duration: 0.3 }}
                            className="lg:hidden bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 shadow-lg"
                        >
                            <div className="px-4 pt-4 pb-6 space-y-1">
                                {/* Mobile Navigation Links */}
                                <div className="space-y-3">
                                    <Link
                                        href="/"
                                        className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                        onClick={() => setIsOpen(false)}
                                        aria-label="Home"
                                    >
                                        🏠 Home
                                    </Link>

                                    <Link
                                        href="/services/essay-writing"
                                        className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                        onClick={() => setIsOpen(false)}
                                        aria-label="Essay Writing"
                                    >
                                        ✍️ Essay Writing
                                    </Link>

                                    {/* Mobile Services Section with improved UI */}
                                    <div className="space-y-2">
                                        <button
                                            onClick={() =>
                                                setIsMobileServicesOpen(
                                                    !isMobileServicesOpen
                                                )
                                            }
                                            className="flex items-center justify-between w-full px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                            aria-expanded={isMobileServicesOpen}
                                            aria-label="Services menu"
                                        >
                                            <span className="flex items-center">
                                                📋 All Services
                                            </span>
                                            <ChevronRight
                                                className={`h-5 w-5 transition-transform duration-200 ${
                                                    isMobileServicesOpen
                                                        ? "rotate-90"
                                                        : ""
                                                }`}
                                            />
                                        </button>

                                        {/* Mobile Services Submenu with enhanced styling */}
                                        <AnimatePresence>
                                            {isMobileServicesOpen && (
                                                <motion.div
                                                    initial={{
                                                        opacity: 0,
                                                        height: 0,
                                                    }}
                                                    animate={{
                                                        opacity: 1,
                                                        height: "auto",
                                                    }}
                                                    exit={{
                                                        opacity: 0,
                                                        height: 0,
                                                    }}
                                                    transition={{
                                                        duration: 0.2,
                                                    }}
                                                    className="ml-4 space-y-2 border-l-2 border-blue-100 dark:border-blue-800 pl-4"
                                                >
                                                    {/* All Services Link */}
                                                    <Link
                                                        href="/services"
                                                        className="flex items-center px-3 py-2 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="View All Services"
                                                    >
                                                        📋 View All Services
                                                    </Link>

                                                    {/* Academic Writing Category */}
                                                    <div className="px-3 py-1">
                                                        <h4 className="text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                            Academic Writing
                                                        </h4>
                                                    </div>
                                                    
                                                    <Link
                                                        href="/services/essay-writing"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Essay Writing"
                                                    >
                                                        ✍️ Essay Writing
                                                    </Link>

                                                    <Link
                                                        href="/services/write-essays-for-money"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Write Essays for Money"
                                                    >
                                                        💵 Write Essays for Money
                                                    </Link>

                                                    <Link
                                                        href="/services/dissertation-writing"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Dissertations"
                                                    >
                                                        📚 Dissertations
                                                    </Link>

                                                    <Link
                                                        href="/services/research-papers"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Research Papers"
                                                    >
                                                        🔬 Research Papers
                                                    </Link>

                                                    <Link
                                                        href="/services/pay-for-research-paper"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Pay for Research Paper"
                                                    >
                                                        📄 Pay for Research Paper
                                                    </Link>

                                                    <Link
                                                        href="/services/thesis-proposals"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Thesis Proposals"
                                                    >
                                                        🎯 Thesis Proposals
                                                    </Link>

                                                    {/* Divider */}
                                                    <div className="border-t border-gray-200 dark:border-gray-600 my-2"></div>

                                                    {/* Specialized Help Category */}
                                                    <div className="px-3 py-1">
                                                        <h4 className="text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                            Specialized Help
                                                        </h4>
                                                    </div>

                                                    <Link
                                                        href="/services/nursing-paper-writing-service"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Nursing Papers"
                                                    >
                                                        🏥 Nursing Papers
                                                    </Link>

                                                    <Link
                                                        href="/services/case-studies"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Case Studies"
                                                    >
                                                        📊 Case Studies
                                                    </Link>

                                                    <Link
                                                        href="/services/literature-reviews"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Literature Reviews"
                                                    >
                                                        📖 Literature Reviews
                                                    </Link>

                                                    <Link
                                                        href="/services/personal-statements"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Personal Statements"
                                                    >
                                                        📝 Personal Statements
                                                    </Link>

                                                    <Link
                                                        href="/services/powerpoint-presentation"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="PowerPoint Presentations"
                                                    >
                                                        📊 PowerPoint
                                                    </Link>

                                                    <Link
                                                        href="/services/powerpoint-presentations-writing-service"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="PPT Writing Service"
                                                    >
                                                        🎨 PPT Writing Service
                                                    </Link>

                                                    {/* Divider */}
                                                    <div className="border-t border-gray-200 dark:border-gray-600 my-2"></div>

                                                    {/* Homework & Classes Category */}
                                                    <div className="px-3 py-1">
                                                        <h4 className="text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                            Homework & Classes
                                                        </h4>
                                                    </div>

                                                    <Link
                                                        href="/services/assignments"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Assignments"
                                                    >
                                                        📝 Assignments
                                                    </Link>

                                                    <Link
                                                        href="/services/pay-for-homework"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Pay for Homework"
                                                    >
                                                        📚 Pay for Homework
                                                    </Link>

                                                    <Link
                                                        href="/services/do-my-math-homework"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Math Homework"
                                                    >
                                                        🔢 Math Homework
                                                    </Link>

                                                    <Link
                                                        href="/services/finance-homework-help"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Finance Homework"
                                                    >
                                                        💰 Finance Homework
                                                    </Link>

                                                    <Link
                                                        href="/services/excel-help"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Excel Help"
                                                    >
                                                        📈 Excel Help
                                                    </Link>

                                                    <Link
                                                        href="/services/spss-assignment-help"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="SPSS Help"
                                                    >
                                                        📊 SPSS Help
                                                    </Link>

                                                    <Link
                                                        href="/services/pay-someone-to-do-my-online-class"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Do My Online Class"
                                                    >
                                                        💻 Do My Online Class
                                                    </Link>

                                                    <Link
                                                        href="/services/programming-assignments"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Programming"
                                                    >
                                                        💻 Programming
                                                    </Link>

                                                    {/* Divider */}
                                                    <div className="border-t border-gray-200 dark:border-gray-600 my-2"></div>

                                                    {/* Popular Services Category */}
                                                    <div className="px-3 py-1">
                                                        <h4 className="text-xs font-semibold text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                            Popular Services
                                                        </h4>
                                                    </div>

                                                    <Link
                                                        href="/services/affordable-essay-writing"
                                                        className="flex items-center px-3 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Affordable Essays"
                                                    >
                                                        💰 Affordable Essays
                                                    </Link>

                                                    <Link
                                                        href="/services/buy-essays-online"
                                                        className="flex items-center px-3 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Buy Essays Online"
                                                    >
                                                        🛒 Buy Essays Online
                                                    </Link>

                                                    <Link
                                                        href="/services/cheap-essays-online"
                                                        className="flex items-center px-3 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Cheap Essays"
                                                    >
                                                        💵 Cheap Essays
                                                    </Link>

                                                    <Link
                                                        href="/services/buy-argumentative-essays"
                                                        className="flex items-center px-3 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileServicesOpen(false);
                                                        }}
                                                        aria-label="Argumentative Essays"
                                                    >
                                                        💭 Argumentative Essays
                                                    </Link>
                                                </motion.div>
                                            )}
                                        </AnimatePresence>
                                    </div>

                                    {/* Mobile About Us Section */}
                                    <div className="space-y-2">
                                        <button
                                            onClick={() =>
                                                setIsMobileAboutOpen(
                                                    !isMobileAboutOpen
                                                )
                                            }
                                            className="flex items-center justify-between w-full px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                            aria-expanded={isMobileAboutOpen}
                                            aria-label="About Us menu"
                                        >
                                            <span className="flex items-center">
                                                ℹ️ About Us
                                            </span>
                                            <ChevronRight
                                                className={`h-5 w-5 transition-transform duration-200 ${
                                                    isMobileAboutOpen
                                                        ? "rotate-90"
                                                        : ""
                                                }`}
                                            />
                                        </button>

                                        {/* Mobile About Us Submenu */}
                                        <AnimatePresence>
                                            {isMobileAboutOpen && (
                                                <motion.div
                                                    initial={{
                                                        opacity: 0,
                                                        height: 0,
                                                    }}
                                                    animate={{
                                                        opacity: 1,
                                                        height: "auto",
                                                    }}
                                                    exit={{
                                                        opacity: 0,
                                                        height: 0,
                                                    }}
                                                    transition={{
                                                        duration: 0.2,
                                                    }}
                                                    className="ml-4 space-y-2 border-l-2 border-blue-100 dark:border-blue-800 pl-4"
                                                >
                                                    <Link
                                                        href="/about"
                                                        className="flex items-center px-3 py-2 text-sm font-semibold text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="About Us"
                                                    >
                                                        ℹ️ About Us
                                                    </Link>

                                                    <Link
                                                        href="/writers"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Our Writers"
                                                    >
                                                        ✍️ Our Writers
                                                    </Link>

                                                    <Link
                                                        href="/guarantees"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Guarantees"
                                                    >
                                                        🛡️ Guarantees
                                                    </Link>

                                                    <Link
                                                        href="/reviews"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Reviews"
                                                    >
                                                        ⭐ Reviews
                                                    </Link>

                                                    <Link
                                                        href="/pricing-page"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Pricing"
                                                    >
                                                        💰 Pricing
                                                    </Link>

                                                    {/* Divider */}
                                                    <div className="border-t border-gray-200 dark:border-gray-600 my-2"></div>

                                                    <Link
                                                        href="/privacy-policy"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Privacy Policy"
                                                    >
                                                        🔒 Privacy Policy
                                                    </Link>

                                                    <Link
                                                        href="/terms-of-service"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Terms of Service"
                                                    >
                                                        📋 Terms of Service
                                                    </Link>

                                                    <Link
                                                        href="/cookie-policy"
                                                        className="flex items-center px-3 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-md transition-all duration-200"
                                                        onClick={() => {
                                                            setIsOpen(false);
                                                            setIsMobileAboutOpen(
                                                                false
                                                            );
                                                        }}
                                                        aria-label="Cookie Policy"
                                                    >
                                                        🍪 Cookie Policy
                                                    </Link>
                                                </motion.div>
                                            )}
                                        </AnimatePresence>
                                    </div>

                                    <Link
                                        href="/blog"
                                        className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                        onClick={() => setIsOpen(false)}
                                        aria-label="Blog"
                                    >
                                        📝 Blog
                                    </Link>

                                    <Link
                                        href="/contact"
                                        className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                        onClick={() => setIsOpen(false)}
                                        aria-label="Contact"
                                    >
                                        📞 Contact
                                    </Link>

                                    <Link
                                        href="/place-order"
                                        className="flex items-center px-4 py-3 text-base font-medium text-white bg-orange-500 hover:bg-orange-600 dark:bg-orange-600 dark:hover:bg-orange-700 rounded-lg transition-all duration-200 shadow-md"
                                        onClick={() => setIsOpen(false)}
                                        aria-label="Place Order"
                                    >
                                        🛒 Place Order
                                    </Link>
                                </div>

                                {/* Mobile Authentication Section with improved styling */}
                                <div className="pt-4 mt-4 border-t border-gray-200 dark:border-gray-700">
                                    {!auth.user ? (
                                        <div className="space-y-3">
                                            <Link
                                                href="/login"
                                                className="flex items-center justify-center px-4 py-3 text-base font-medium text-blue-600 dark:text-blue-400 border-2 border-blue-600 dark:border-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                                onClick={() => setIsOpen(false)}
                                                aria-label="Login"
                                            >
                                                🔐 Login
                                            </Link>
                                            <Link
                                                href="/register"
                                                className="flex items-center justify-center px-4 py-3 text-base font-medium text-white bg-blue-600 hover:bg-blue-700 dark:bg-blue-700 dark:hover:bg-blue-800 rounded-lg transition-all duration-200 shadow-md"
                                                onClick={() => setIsOpen(false)}
                                                aria-label="Register"
                                            >
                                                📝 Register
                                            </Link>
                                        </div>
                                    ) : (
                                        <div className="space-y-3">
                                            {/* Mobile User Profile Section */}
                                            <div className="px-4 py-3 bg-blue-50 dark:bg-blue-900/30 rounded-lg">
                                                <div className="flex items-center space-x-3">
                                                    <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center">
                                                        <span className="text-white font-semibold text-sm">
                                                            {auth.user.name
                                                                .charAt(0)
                                                                .toUpperCase()}
                                                        </span>
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                            {auth.user.name}
                                                        </p>
                                                        <p className="text-xs text-gray-500 dark:text-gray-400">
                                                            {auth.user.email}
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Mobile User Menu Items */}
                                            <div className="space-y-2">
                                                <Link
                                                    href="/dashboard"
                                                    className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                                    onClick={() =>
                                                        setIsOpen(false)
                                                    }
                                                    aria-label="Dashboard"
                                                >
                                                    📊 Dashboard
                                                </Link>
                                                <Link
                                                    href="/profile"
                                                    className="flex items-center px-4 py-3 text-base font-medium text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/30 rounded-lg transition-all duration-200"
                                                    onClick={() =>
                                                        setIsOpen(false)
                                                    }
                                                    aria-label="Profile"
                                                >
                                                    👤 Profile
                                                </Link>
                                                <Link
                                                    href="/logout"
                                                    method="post"
                                                    as="button"
                                                    className="flex items-center w-full px-4 py-3 text-base font-medium text-red-600 dark:text-red-400 hover:bg-red-50 dark:hover:bg-red-900/30 rounded-lg transition-all duration-200"
                                                    onClick={() =>
                                                        setIsOpen(false)
                                                    }
                                                    aria-label="Logout"
                                                >
                                                    🚪 Logout
                                                </Link>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </nav>
        </div>
    );
}
