import React, { useState, useEffect, useRef, useCallback } from "react";
import {
    Bell,
    X,
    Check,
    Settings,
    ChevronRight,
    Loader2,
    RefreshCw,
    AlertCircle,
} from "lucide-react";
import { Link } from "@inertiajs/react";
import axios from "axios";
import NotificationList from "./NotificationList";
import { debounce } from "lodash";
import NotificationItem from "./NotificationItem";

export default function NotificationBell() {
    const [isOpen, setIsOpen] = useState(false);
    const [unreadCount, setUnreadCount] = useState(0);
    const [prevUnreadCount, setPrevUnreadCount] = useState(0);
    const [notifications, setNotifications] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const [isRefreshing, setIsRefreshing] = useState(false);
    const [error, setError] = useState(null);
    const [hasMore, setHasMore] = useState(false);
    const [page, setPage] = useState(1);
    const [isCountBadgeAnimating, setIsCountBadgeAnimating] = useState(false);
    const [dropdownPosition, setDropdownPosition] = useState({
        top: 0,
        left: 0,
    });

    const dropdownRef = useRef(null);
    const bellRef = useRef(null);
    const countRef = useRef(null);

    // Helper function to safely get route URLs
    // Update the getRouteUrl function to ensure consistent hostname
    const getRouteUrl = (routeName, params = {}) => {
        try {
            // Check if route exists in Ziggy routes
            if (route().has(routeName)) {
                // Get the base URL from the current window location
                const baseUrl = window.location.origin;
                // Get the route path without the domain
                const routePath = route(routeName, params).replace(
                    /^https?:\/\/[^/]+/,
                    ""
                );
                // Combine them to ensure consistent hostname
                return `${baseUrl}${routePath}`;
            }

            // Rest of the function remains the same...
        } catch (error) {
            console.error(`Error getting route URL for ${routeName}:`, error);
            return "#";
        }
    };

    // Fetch unread count when component mounts and periodically
    useEffect(() => {
        fetchUnreadCount();

        // Poll for new notifications every 30 seconds
        const intervalId = setInterval(fetchUnreadCount, 30000);

        // Add event listener for clicks outside dropdown
        document.addEventListener("mousedown", handleClickOutside);

        // Add event listener for escape key
        document.addEventListener("keydown", handleEscapeKey);

        return () => {
            clearInterval(intervalId);
            document.removeEventListener("mousedown", handleClickOutside);
            document.removeEventListener("keydown", handleEscapeKey);
        };
    }, []);

    // Trigger animation when unread count changes
    useEffect(() => {
        if (unreadCount > prevUnreadCount && prevUnreadCount !== 0) {
            // Only animate if this isn't the initial load
            triggerCountAnimation();
        }
        setPrevUnreadCount(unreadCount);
    }, [unreadCount, prevUnreadCount]);

    // Fetch notifications when dropdown is opened
    useEffect(() => {
        if (isOpen) {
            setPage(1);
            fetchNotifications(1, true);
            // Position the dropdown after opening
            setTimeout(positionDropdown, 0);
        }
    }, [isOpen]);

    // Calculate and update dropdown position
    useEffect(() => {
        if (isOpen && dropdownRef.current && bellRef.current) {
            positionDropdown();
            // Add window resize listener to reposition dropdown
            window.addEventListener("resize", handleResize);
            window.addEventListener("scroll", handleScroll, true);
        }

        return () => {
            window.removeEventListener("resize", handleResize);
            window.removeEventListener("scroll", handleScroll, true);
        };
    }, [isOpen]);

    // Debounced resize handler
    const handleResize = debounce(() => {
        if (isOpen && dropdownRef.current) {
            positionDropdown();
        }
    }, 150);

    // Debounced scroll handler
    const handleScroll = debounce(() => {
        if (isOpen && dropdownRef.current) {
            positionDropdown();
        }
    }, 50);

    // Calculate the best position for the dropdown
    const positionDropdown = () => {
        if (!dropdownRef.current || !bellRef.current) return;

        const bellRect = bellRef.current.getBoundingClientRect();
        const dropdownWidth = Math.min(window.innerWidth - 20, 380); // Cap width on small screens

        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;

        // Calculate optimal left position (default to right-aligned with bell)
        let leftPos = Math.min(
            bellRect.right - dropdownWidth,
            viewportWidth - dropdownWidth - 10
        );

        // Ensure dropdown doesn't overflow left edge
        leftPos = Math.max(10, leftPos);

        // Calculate top position (below the bell) - FIXED: Don't add window.scrollY for fixed positioning
        let topPos = bellRect.bottom + 5; // Add 5px gap

        // Check if dropdown would overflow bottom of viewport
        const dropdownHeight = Math.min(viewportHeight * 0.7, 500); // Estimate max height

        if (topPos + dropdownHeight > viewportHeight) {
            // Position above the bell if there's enough space
            if (bellRect.top > dropdownHeight) {
                topPos = bellRect.top - dropdownHeight - 5;
            }
        }

        // Update state with new position
        setDropdownPosition({
            top: topPos,
            left: leftPos,
            width: dropdownWidth,
        });
    };

    // Animate the badge when new notifications arrive
    const triggerCountAnimation = () => {
        setIsCountBadgeAnimating(true);
        setTimeout(() => {
            setIsCountBadgeAnimating(false);
        }, 1000);
    };

    // Handle clicks outside of dropdown
    const handleClickOutside = (event) => {
        if (
            dropdownRef.current &&
            !dropdownRef.current.contains(event.target) &&
            bellRef.current &&
            !bellRef.current.contains(event.target)
        ) {
            setIsOpen(false);
        }
    };

    // Handle escape key to close dropdown
    const handleEscapeKey = (event) => {
        if (event.key === "Escape" && isOpen) {
            setIsOpen(false);
        }
    };

    // Fetch unread notification count
    const fetchUnreadCount = async () => {
        try {
            const response = await axios.get(
                getRouteUrl("notifications.count")
            );
            if (response.data.success) {
                setUnreadCount(response.data.count);
            }
        } catch (err) {
            console.error("Error fetching notification count:", err);
            // Don't set error state here to avoid UI disruption
        }
    };

    // Fetch recent notifications
    // Fetch recent notifications
    const fetchNotifications = async (pageNumber = 1, resetList = false) => {
        if (resetList) {
            setIsLoading(true);
        } else {
            setIsRefreshing(true);
        }
        setError(null);

        try {
            // The issue is here - we need to explicitly set read: false as a boolean, not a string
            const response = await axios.get(
                getRouteUrl("notifications.unread"),
                {
                    params: {
                        page: pageNumber,
                        per_page: 5, // Limit to 5 per page
                        // The problem is probably that this is getting converted to a string 'false'
                        // Let's not send it at all since the controller adds it anyway
                    },
                }
            );

            if (response.data.success) {
                const newNotifications = response.data.data.data || [];

                if (resetList) {
                    setNotifications(newNotifications);
                } else {
                    setNotifications((prev) => [...prev, ...newNotifications]);
                }

                // Check if there are more pages
                const currentPage = response.data.data.current_page || 1;
                const lastPage = response.data.data.last_page || 1;
                setHasMore(currentPage < lastPage);
                setPage(currentPage);
            } else {
                setError("Failed to load notifications");
            }
        } catch (err) {
            console.error("Error fetching notifications:", err);
            setError(
                "Failed to load notifications: " +
                    (err.response?.data?.message || err.message)
            );
        } finally {
            setIsLoading(false);
            setIsRefreshing(false);
        }
    };

    // Load more notifications
    const loadMoreNotifications = async () => {
        if (hasMore && !isRefreshing) {
            await fetchNotifications(page + 1, false);
        }
    };

    // Refresh notifications
    const refreshNotifications = async () => {
        setPage(1);
        await fetchNotifications(1, true);
    };

    // Mark all notifications as read
    const markAllAsRead = async (e) => {
        // Prevent event propagation to avoid closing dropdown
        if (e) {
            e.preventDefault();
            e.stopPropagation();
        }

        try {
            const response = await axios.post(
                getRouteUrl("notifications.mark-all-read")
            );

            if (response.data.success) {
                // Update local state
                setUnreadCount(0);

                // Update notification items to show as read
                setNotifications((prevNotifications) =>
                    prevNotifications.map((notification) => ({
                        ...notification,
                        read_at: new Date().toISOString(),
                    }))
                );

                // Show success message if toast is available
                if (window.toast?.success) {
                    window.toast.success("All notifications marked as read");
                }
            }
        } catch (err) {
            console.error("Error marking notifications as read:", err);

            // Show error message if toast is available
            if (window.toast?.error) {
                window.toast.error("Failed to mark notifications as read");
            }
        }
    };

    // Handle individual notification update
    const handleNotificationUpdate = (updatedNotification) => {
        // Update the notification in the list
        setNotifications((prev) =>
            prev.map((notification) =>
                notification.id === updatedNotification.id
                    ? updatedNotification
                    : notification
            )
        );

        // If notification was marked as read, update unread count
        if (updatedNotification.read_at && unreadCount > 0) {
            setUnreadCount((prev) => prev - 1);
        }
    };

    // Toggle dropdown
    const toggleDropdown = () => {
        setIsOpen(!isOpen);
    };

    return (
        <div className="relative">
            {/* Bell button with notification count */}
            <button
                ref={bellRef}
                onClick={toggleDropdown}
                className={`p-1.5 sm:p-2 rounded-full transition-colors duration-200 relative ${
                    isOpen
                        ? "bg-primary/20 text-primary dark:bg-primary/30 dark:text-primary-foreground"
                        : "hover:bg-gray-200/20 dark:hover:bg-gray-700 text-white dark:text-gray-200"
                }`}
                aria-label={`Notifications ${
                    unreadCount > 0 ? `(${unreadCount} unread)` : ""
                }`}
                aria-expanded={isOpen}
                aria-haspopup="true"
            >
                <Bell className="h-4 w-4 sm:h-5 sm:w-5" />

                {/* Notification count badge */}
                {unreadCount > 0 && (
                    <span
                        ref={countRef}
                        className={`absolute top-0 right-0 h-3.5 w-3.5 sm:h-4 sm:w-4 flex items-center justify-center bg-red-500 rounded-full text-xs text-white font-medium transform transition-all ${
                            isCountBadgeAnimating
                                ? "scale-125 animate-pulse"
                                : "scale-100"
                        }`}
                        aria-hidden="true"
                    >
                        <span className="text-xs sm:text-xs">
                            {unreadCount > 9 ? "9+" : unreadCount}
                        </span>
                    </span>
                )}
            </button>

            {/* Notification dropdown - positioned with fixed position */}
            {isOpen && (
                <div
                    ref={dropdownRef}
                    className="fixed bg-white dark:bg-gray-800 rounded-md shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden animate-in fade-in zoom-in-95 duration-200 z-50"
                    role="dialog"
                    aria-label="Notifications"
                    style={{
                        top: `${dropdownPosition.top}px`,
                        left: `${dropdownPosition.left}px`,
                        width: `${dropdownPosition.width}px`,
                        maxHeight: "70vh",
                    }}
                >
                    {/* Dropdown header */}
                    <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-gray-900 ">
                        <h3 className="font-medium text-gray-900 dark:text-white flex items-center gap-2">
                            <Bell className="h-4 w-4" />
                            <span>Notifications</span>
                            {unreadCount > 0 && (
                                <span className="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                                    {unreadCount}
                                </span>
                            )}
                        </h3>

                        <div className="flex space-x-1">
                            {/* Refresh button */}
                            <button
                                onClick={(e) => {
                                    e.preventDefault();
                                    e.stopPropagation();
                                    refreshNotifications();
                                }}
                                disabled={isRefreshing || isLoading}
                                className="p-1 rounded-full hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-500 dark:text-gray-400"
                                title="Refresh notifications"
                            >
                                {isRefreshing ? (
                                    <Loader2 className="h-4 w-4 animate-spin" />
                                ) : (
                                    <RefreshCw className="h-4 w-4" />
                                )}
                            </button>

                            {/* Mark all as read button */}
                            {unreadCount > 0 && (
                                <button
                                    onClick={markAllAsRead}
                                    className="p-1 rounded-full hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-500 dark:text-gray-400"
                                    title="Mark all as read"
                                >
                                    <Check className="h-4 w-4" />
                                </button>
                            )}

                            {/* Close dropdown button */}
                            <button
                                onClick={(e) => {
                                    e.preventDefault();
                                    e.stopPropagation();
                                    setIsOpen(false);
                                }}
                                className="p-1 rounded-full hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-500 dark:text-gray-400"
                                title="Close"
                            >
                                <X className="h-4 w-4" />
                            </button>
                        </div>
                    </div>

                    {/* Notification list */}
                    <div
                        className="overflow-y-auto"
                        style={{ maxHeight: "calc(70vh - 110px)" }}
                    >
                        {isLoading ? (
                            <div className="flex justify-center items-center p-6">
                                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
                            </div>
                        ) : error ? (
                            <div className="p-6 text-center text-red-500 dark:text-red-400">
                                {error}
                                <button
                                    onClick={refreshNotifications}
                                    className="block mx-auto mt-2 text-sm text-blue-500 hover:text-blue-700 dark:text-blue-400 dark:hover:text-blue-300"
                                >
                                    Try again
                                </button>
                            </div>
                        ) : notifications.length === 0 ? (
                            <div className="p-6 text-center text-gray-500 dark:text-gray-400">
                                No new notifications
                            </div>
                        ) : (
                            <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                {notifications.map((notification) => (
                                    <NotificationItem
                                        key={notification.id}
                                        notification={notification}
                                        onUpdate={handleNotificationUpdate}
                                        getRouteUrl={getRouteUrl}
                                    />
                                ))}

                                {hasMore && (
                                    <div className="p-2 text-center">
                                        <button
                                            onClick={loadMoreNotifications}
                                            disabled={isRefreshing}
                                            className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                        >
                                            {isRefreshing
                                                ? "Loading..."
                                                : "Load more"}
                                        </button>
                                    </div>
                                )}
                            </div>
                        )}
                    </div>

                    {/* Dropdown footer */}
                    <div className="border-t border-gray-200 dark:border-gray-700 p-2 bg-gray-50 dark:bg-gray-900">
                        <div className="grid grid-cols-2 gap-2">
                            <Link
                                href={getRouteUrl("notifications.index")}
                                className="text-center text-sm py-2 px-3 rounded-md hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 transition-colors flex items-center justify-center gap-1"
                                onClick={() => setIsOpen(false)}
                            >
                                <span>View All</span>
                                <ChevronRight className="h-3.5 w-3.5" />
                            </Link>
                            {/* <Link
                                href={getRouteUrl("notifications.preferences")}
                                className="text-center text-sm py-2 px-3 rounded-md hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 transition-colors flex items-center justify-center gap-1"
                                onClick={() => setIsOpen(false)}
                            >
                                <Settings className="h-3.5 w-3.5" />
                                <span>Settings</span>
                            </Link> */}
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
