import React, { useState } from "react";
import { Link } from "@inertiajs/react";
import { formatDistanceToNow } from "date-fns";
import axios from "axios";
import {
    Bell,
    CheckCircle,
    AlertCircle,
    XCircle,
    Info,
    FileText,
    DollarSign,
    Gift,
    AlertTriangle,
    Clock,
    Zap,
    User,
    MessageSquare,
    Loader2,
} from "lucide-react";

export default function NotificationItem({
    notification,
    onUpdate,
    getRouteUrl,
}) {
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState(null);
    const [isRead, setIsRead] = useState(notification?.read_at !== null);

    // Safety check for null or undefined notification
    if (!notification) return null;

    // Parse the data from the notification
    const data = (() => {
        try {
            if (!notification.data) return {};
            return typeof notification.data === "string"
                ? JSON.parse(notification.data)
                : notification.data;
        } catch (e) {
            console.error("Error parsing notification data:", e);
            return {};
        }
    })();

    // Get icon based on notification type
    const getIcon = () => {
        const iconName = data?.icon || "bell";
        const iconSize = 18;
        const iconProps = {
            size: iconSize,
            className: `${getIconColorClass()} ${
                isLoading ? "opacity-50" : ""
            }`,
        };

        switch (iconName) {
            case "check-circle":
                return <CheckCircle {...iconProps} />;
            case "alert-circle":
                return <AlertCircle {...iconProps} />;
            case "x-circle":
                return <XCircle {...iconProps} />;
            case "info":
                return <Info {...iconProps} />;
            case "file-text":
                return <FileText {...iconProps} />;
            case "dollar-sign":
                return <DollarSign {...iconProps} />;
            case "gift":
                return <Gift {...iconProps} />;
            case "alert-triangle":
                return <AlertTriangle {...iconProps} />;
            case "clock":
                return <Clock {...iconProps} />;
            case "zap":
                return <Zap {...iconProps} />;
            case "user":
                return <User {...iconProps} />;
            case "message-square":
                return <MessageSquare {...iconProps} />;
            default:
                return <Bell {...iconProps} />;
        }
    };

    // Get color class based on notification type
    const getIconColorClass = () => {
        const color = data?.color || "default";

        switch (color) {
            case "success":
                return "text-green-500 dark:text-green-400";
            case "danger":
                return "text-red-500 dark:text-red-400";
            case "warning":
                return "text-amber-500 dark:text-amber-400";
            case "info":
                return "text-blue-500 dark:text-blue-400";
            default:
                return "text-gray-500 dark:text-gray-400";
        }
    };

    // Get background color class based on read status
    const getBackgroundClass = () => {
        return isRead
            ? "bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-750"
            : "bg-blue-50 dark:bg-blue-900/20 hover:bg-blue-100 dark:hover:bg-blue-900/30";
    };

    // Format the notification time
    const getFormattedTime = () => {
        try {
            if (!notification.created_at) return "recently";
            return formatDistanceToNow(new Date(notification.created_at), {
                addSuffix: true,
            });
        } catch (error) {
            console.error("Error formatting date:", error);
            return "recently";
        }
    };

    // Mark notification as read
    const markAsRead = async (e) => {
        // If it's already read or currently loading, do nothing
        if (isRead || isLoading) return;

        // If there's a link and it's a regular click (not middle-click or ctrl+click), let the navigation happen
        if (
            e.currentTarget.href &&
            e.button !== 1 &&
            !e.ctrlKey &&
            !e.metaKey &&
            !e.shiftKey
        ) {
            return;
        }

        // Otherwise prevent default and mark as read
        e.preventDefault();
        setError(null);
        setIsLoading(true);

        try {
            // Get the URL for marking as read
            const url = getRouteUrl
                ? getRouteUrl("notifications.mark-read", {
                      id: notification.id,
                  })
                : `/notifications/api/${notification.id}/read`;

            const response = await axios.post(url);

            if (response.data.success) {
                // Update the local state
                setIsRead(true);

                // Update the notification in the parent component
                if (typeof onUpdate === "function") {
                    if (response.data.notification) {
                        // If the API returns the updated notification, use it
                        onUpdate(response.data.notification);
                    } else {
                        // Otherwise update the read_at field manually
                        onUpdate({
                            ...notification,
                            read_at: new Date().toISOString(),
                        });
                    }
                }
            } else {
                // Handle API success: false case
                console.error(
                    "API returned success: false when marking notification as read:",
                    response.data
                );
                setError(response.data.message || "Failed to mark as read");

                // Show toast if available
                if (window.toast?.error) {
                    window.toast.error(
                        response.data.message ||
                            "Failed to mark notification as read"
                    );
                }
            }
        } catch (error) {
            console.error("Error marking notification as read:", error);
            setError(error.response?.data?.message || "Failed to mark as read");

            // Show toast if available
            if (window.toast?.error) {
                window.toast.error(
                    error.response?.data?.message ||
                        "Failed to mark notification as read"
                );
            }
        } finally {
            setIsLoading(false);
        }
    };

    // Get the action URL from the notification data
    // Get the action URL from the notification data
    const getActionUrl = () => {
        // Always prioritize going to the notification detail page first
        if (notification && notification.id) {
            return getRouteUrl
                ? getRouteUrl("notifications.show", { id: notification.id })
                : `/notifications/${notification.id}`;
        }

        // Fallback to action URL if notification show route isn't available
        if (!data.actions || !data.actions.length) return "#";
        return data.actions[0].url || "#";
    };

    return (
        <div
            className={`block p-4 transition-colors duration-200 relative ${getBackgroundClass()}`}
        >
            {/* Loading indicator overlay */}
            {isLoading && (
                <div className="absolute inset-0 bg-white/50 dark:bg-gray-800/50 flex items-center justify-center z-10">
                    <Loader2 className="h-5 w-5 animate-spin text-primary" />
                </div>
            )}

            {/* Error indicator */}
            {error && (
                <div className="absolute top-1 right-1 text-xs text-red-500 dark:text-red-400 bg-red-50 dark:bg-red-900/20 px-2 py-0.5 rounded-full">
                    Error
                </div>
            )}

            <div className="flex items-start gap-3">
                <div className="flex-shrink-0 mt-0.5">{getIcon()}</div>

                <div className="flex-1 min-w-0">
                    <Link
                        href={getActionUrl()}
                        onClick={markAsRead}
                        className={`text-sm font-medium hover:underline ${
                            isRead
                                ? "text-gray-700 dark:text-gray-300"
                                : "text-gray-900 dark:text-white"
                        }`}
                    >
                        {data?.title || "Notification"}
                    </Link>

                    <p className="text-sm text-gray-500 dark:text-gray-400 mt-1 line-clamp-2">
                        {data?.message || "You have a new notification"}
                    </p>

                    <p className="text-xs text-gray-400 dark:text-gray-500 mt-1">
                        {getFormattedTime()}
                    </p>
                </div>
            </div>
        </div>
    );
}
