import React from "react";
import {
    Check,
    Calendar,
    ClipboardCheck,
    BookOpen,
    FileText,
    DollarSign,
    Globe,
    FileUp,
    ShoppingCart,
    Tag,
} from "lucide-react";

const OrderSummary = ({
    data,
    orderCalculation,
    academicLevels,
    subjects,
    paperTypes,
    processing,
}) => {
    // Helper function to get name from value string
    const getNameFromValue = (valueString, options) => {
        if (!valueString || !options) return "Not specified";

        const parts = valueString.split("#");
        if (parts.length > 1) return parts[1];

        // If not found in the format, look up in options
        const option = options.find((opt) => opt.value === valueString);
        return option ? option.name : valueString;
    };

    // Format date from deadline string
    const formatDeadline = () => {
        if (!data.deadline) return "Not specified";

        const parts = data.deadline.split("#");
        if (parts.length < 3) return "Not specified";

        return `${parts[1]} ${parts[2]}`;
    };

    const formatPrice = (amount) => {
        return `${orderCalculation.currencySymbol}${parseFloat(amount).toFixed(
            2
        )}`;
    };

    // Get citation style name
    const getCitationStyle = () => {
        const styles = {
            1: "APA",
            2: "MLA",
            3: "Chicago",
            4: "Harvard",
            5: "Turabian",
            6: "Oxford",
            7: "Vancouver",
        };

        return styles[data.styles] || "Not specified";
    };

    // Get language preference name
    const getLanguagePreference = () => {
        const languages = {
            1: "American English",
            2: "British English",
            3: "Australian English",
            4: "Canadian English",
        };

        return languages[data.language] || "Not specified";
    };

    // Calculate discount amount
    const calculateDiscount = () => {
        if (!data.couponCode || !data.discountAmount) return 0;

        if (data.discountType === "percentage") {
            return (
                (parseFloat(orderCalculation.totalAmount) *
                    data.discountAmount) /
                100
            );
        } else {
            return parseFloat(data.discountAmount);
        }
    };

    // Calculate final amount after discount
    const calculateFinalAmount = () => {
        const discount = calculateDiscount();
        return parseFloat(orderCalculation.totalAmount) - discount;
    };

    return (
        <div className="space-y-8" aria-labelledby="step4-heading">
            <div className="bg-green-50 dark:bg-green-900/20 border border-green-100 dark:border-green-800 rounded-lg p-4 mb-6">
                <div className="flex items-start">
                    <div className="flex-shrink-0">
                        <Check
                            className="h-5 w-5 text-green-600 dark:text-green-400"
                            aria-hidden="true"
                        />
                    </div>
                    <div className="ml-3">
                        <h3 className="text-sm font-medium text-green-800 dark:text-green-300">
                            Final Step
                        </h3>
                        <p className="mt-1 text-sm text-green-700 dark:text-green-400">
                            Please review your order details below. If
                            everything looks correct, click the "Place Order"
                            button to submit.
                        </p>
                    </div>
                </div>
            </div>

            {/* Order Summary Card */}
            <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden shadow-sm">
                <div className="px-6 py-4 bg-gradient-to-r from-blue-100 to-blue-50 dark:from-gray-700 dark:to-gray-750">
                    <h2 className="text-lg font-semibold text-gray-900 dark:text-white">
                        Order Summary
                    </h2>
                </div>

                {/* Basic Information Section */}
                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                    <h3 className="text-md font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                        <ClipboardCheck className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                        Basic Information
                    </h3>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Paper Title
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {data.title || "Not specified"}
                            </p>
                        </div>

                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Academic Level
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getNameFromValue(data.aclevel, academicLevels)}
                            </p>
                        </div>

                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Subject
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getNameFromValue(data.subject, subjects)}
                            </p>
                        </div>

                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Type of Paper
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getNameFromValue(data.papertype, paperTypes)}
                            </p>
                        </div>
                    </div>
                </div>

                {/* Project Specifications Section */}
                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                    <h3 className="text-md font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                        <Calendar className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                        Project Specifications
                    </h3>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Deadline
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {formatDeadline()}
                            </p>
                        </div>

                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Spacing
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getNameFromValue(data.spacing, [
                                    {
                                        value: "1#Double Spaced",
                                        name: "Double Spaced",
                                    },
                                    {
                                        value: "2#Single Spaced",
                                        name: "Single Spaced",
                                    },
                                ])}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Number of Pages
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {data.pages || "0"}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                PowerPoint Slides
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {data.ppslides || "0"}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Number of Sources
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {data.sources || "0"}
                            </p>
                        </div>
                    </div>
                </div>

                {/* Writing Preferences Section */}
                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                    <h3 className="text-md font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                        <BookOpen className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                        Writing Preferences
                    </h3>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Citation Style
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getCitationStyle()}
                            </p>
                        </div>

                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Language Preference
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getLanguagePreference()}
                            </p>
                        </div>
                    </div>

                    {/* Instructions */}
                    <div className="mt-4">
                        <p className="text-sm text-gray-500 dark:text-gray-400">
                            Instructions
                        </p>
                        <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <p className="text-sm text-gray-800 dark:text-gray-200 whitespace-pre-wrap">
                                {data.instructions ||
                                    "No instructions provided."}
                            </p>
                        </div>
                    </div>

                    {/* Uploaded Files */}
                    {data.files?.length > 0 && (
                        <div className="mt-4">
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Uploaded Files
                            </p>
                            <div className="mt-2 p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                <ul className="space-y-2">
                                    {Array.from(data.files).map(
                                        (file, index) => (
                                            <li
                                                key={index}
                                                className="flex items-center"
                                            >
                                                <FileUp className="w-4 h-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm text-gray-800 dark:text-gray-200">
                                                    {file.name} (
                                                    {(file.size / 1024).toFixed(
                                                        2
                                                    )}{" "}
                                                    KB)
                                                </span>
                                            </li>
                                        )
                                    )}
                                </ul>
                            </div>
                        </div>
                    )}
                </div>

                {/* Price Summary Section */}
                <div className="p-6">
                    <h3 className="text-md font-medium text-gray-900 dark:text-white mb-4 flex items-center">
                        <ShoppingCart className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                        Price Summary
                    </h3>

                    <div className="space-y-3">
                        <div className="flex justify-between">
                            <span className="text-sm text-gray-500 dark:text-gray-400">
                                Pages: {data.pages} ×{" "}
                                {orderCalculation.currencySymbol}
                                {orderCalculation.costPerPage}
                            </span>
                            <span className="text-sm font-medium text-gray-800 dark:text-gray-200">
                                {formatPrice(orderCalculation.basePrice)}
                            </span>
                        </div>

                        {data.ppslides > 0 && (
                            <div className="flex justify-between">
                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                    PowerPoint Slides: {data.ppslides} ×{" "}
                                    {orderCalculation.currencySymbol}
                                    {data.currency && data.currency !== "USD"
                                        ? (
                                              4 * (data.currencyRate || 1)
                                          ).toFixed(2)
                                        : "4.00"}
                                </span>
                                <span className="text-sm font-medium text-gray-800 dark:text-gray-200">
                                    {orderCalculation.currencySymbol}
                                    {parseFloat(
                                        orderCalculation.costPerSlide
                                    ).toFixed(2)}
                                </span>
                            </div>
                        )}

                        {/* Show coupon discount if applied */}
                        {data.couponCode && data.discountAmount > 0 && (
                            <div className="flex justify-between text-green-600 dark:text-green-400">
                                <span className="text-sm flex items-center">
                                    <Tag className="w-4 h-4 mr-1" />
                                    Coupon Discount ({data.couponCode}):
                                </span>
                                <span className="text-sm font-medium">
                                    - {orderCalculation.currencySymbol}
                                    {data.discountType === "percentage"
                                        ? (
                                              (parseFloat(
                                                  orderCalculation.totalAmount
                                              ) *
                                                  data.discountAmount) /
                                              100
                                          ).toFixed(2)
                                        : parseFloat(
                                              data.discountAmount
                                          ).toFixed(2)}
                                </span>
                            </div>
                        )}

                        <div className="pt-2 border-t border-gray-200 dark:border-gray-700 flex justify-between">
                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                Total:
                            </span>
                            <span className="text-xl font-bold text-blue-600 dark:text-blue-300">
                                {orderCalculation.currencySymbol}{orderCalculation.netAmount}
                            </span>
                        </div>

                        {data.currency !== "USD" && (
                            <div className="text-right">
                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                    ($
                                    {(
                                        parseFloat(
                                            data.couponCode &&
                                                data.discountAmount > 0
                                                ? data.discountType ===
                                                  "percentage"
                                                    ? parseFloat(
                                                          orderCalculation.totalAmount
                                                      ) -
                                                      (parseFloat(
                                                          orderCalculation.totalAmount
                                                      ) *
                                                          data.discountAmount) /
                                                          100
                                                    : parseFloat(
                                                          orderCalculation.totalAmount
                                                      ) -
                                                      parseFloat(
                                                          data.discountAmount
                                                      )
                                                : orderCalculation.totalAmount
                                        ) / (data.currencyRate || 1)
                                    ).toFixed(2)}{" "}
                                    USD)
                                </span>
                            </div>
                        )}

                        {/* Add savings message if any discount is applied */}
                        {orderCalculation.discount > 0 && (
                            <div className="mt-2 p-2 bg-green-50 dark:bg-green-900/20 rounded-md text-sm text-green-700 dark:text-green-300">
                                <p className="flex items-center">
                                    <DollarSign className="w-4 h-4 mr-1 flex-shrink-0" />
                                    You saved {orderCalculation.currencySymbol}{orderCalculation.discount} total
                                    ({Math.round(((parseFloat(orderCalculation.totalAmount) - parseFloat(orderCalculation.netAmount)) / parseFloat(orderCalculation.totalAmount)) * 100)}% off)!
                                </p>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Terms and Conditions */}
            <div className="p-4 bg-gray-50 dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700">
                <div className="flex items-start">
                    <div className="flex-shrink-0">
                        <FileText
                            className="h-5 w-5 text-gray-400 dark:text-gray-500"
                            aria-hidden="true"
                        />
                    </div>
                    <div className="ml-3 text-sm">
                        <p className="text-gray-500 dark:text-gray-400">
                            By placing this order, you agree to our{" "}
                            <a
                                href="/terms"
                                className="text-blue-600 dark:text-blue-400 hover:underline"
                            >
                                Terms and Conditions
                            </a>{" "}
                            and{" "}
                            <a
                                href="/privacy"
                                className="text-blue-600 dark:text-blue-400 hover:underline"
                            >
                                Privacy Policy
                            </a>
                            . Your order will enter our bidding system, where
                            qualified writers can view and bid on your project.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default OrderSummary;
