import React, { useState, useEffect } from "react";
import {
    Calendar,
    AlignLeft,
    AlignCenter,
    Plus,
    Minus,
    Calendar as CalendarIcon,
} from "lucide-react";
import { format, addHours } from "date-fns";

const ProjectSpecifications = ({
    data,
    setData,
    errors,
    deadlines,
    orderCalculation,
    singleSlidePrice,
}) => {
    // Preview deadline date/time based on selected urgency
    const [deadlinePreview, setDeadlinePreview] = useState("");

    // Calculate deadline preview when urgency changes
    useEffect(() => {
        if (data.deadline) {
            const parts = data.deadline.split("#");
            if (parts.length >= 3) {
                const value = parseInt(parts[1]);
                const unit = parts[2];

                let deadlineDate;
                if (unit.toLowerCase() === "days") {
                    deadlineDate = addHours(new Date(), value * 24);
                } else {
                    deadlineDate = addHours(new Date(), value);
                }

                const formattedDeadline = format(deadlineDate, "PPP 'at' p");
                setDeadlinePreview(formattedDeadline);
            }
        }
    }, [data.deadline]);

    // Handle spacing selection
    const handleSpacingChange = (value) => {
        setData("spacing", value);
    };

    // Handle deadline selection
    const handleDeadlineChange = (value) => {
        setData("deadline", value);
    };

    // Increment/decrement number input fields
    const incrementField = (field, min = 1, max = 100) => {
        const currentValue = parseInt(data[field]) || min;
        setData(field, Math.min(currentValue + 1, max));
    };

    const decrementField = (field, min = 1) => {
        const currentValue = parseInt(data[field]) || min + 1;
        setData(field, Math.max(currentValue - 1, min));
    };

    // Handle number field input
    const handleNumberInput = (field, value, min = 1, max = 100) => {
        const numValue = parseInt(value) || min;
        setData(field, Math.max(Math.min(numValue, max), min));
    };

    return (
        <div className="space-y-8" aria-labelledby="step2-heading">
            {/* Deadline selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Deadline <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select deadline"
                >
                    {deadlines?.map((deadline) => (
                        <div
                            key={deadline.id}
                            onClick={() =>
                                handleDeadlineChange(
                                    `${deadline.value}#${deadline.urgency}#${deadline.duration}`
                                )
                            }
                            className={`
                                relative flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                ${
                                    data.deadline?.includes(
                                        `${deadline.urgency}#${deadline.duration}`
                                    )
                                        ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                        : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                }
                            `}
                            role="radio"
                            aria-checked={data.deadline?.includes(
                                `${deadline.urgency}#${deadline.duration}`
                            )}
                            tabIndex={0}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" || e.key === " ") {
                                    e.preventDefault();
                                    handleDeadlineChange(
                                        `${deadline.value}#${deadline.urgency}#${deadline.duration}`
                                    );
                                }
                            }}
                        >
                            <CalendarIcon
                                className={`w-5 h-5 mr-3 ${
                                    data.deadline?.includes(
                                        `${deadline.urgency}#${deadline.duration}`
                                    )
                                        ? "text-blue-600 dark:text-blue-400"
                                        : "text-gray-500 dark:text-gray-400"
                                }`}
                            />
                            <div>
                                <span
                                    className={`text-sm font-medium ${
                                        data.deadline?.includes(
                                            `${deadline.urgency}#${deadline.duration}`
                                        )
                                            ? "text-blue-700 dark:text-blue-300"
                                            : "text-gray-700 dark:text-gray-300"
                                    }`}
                                >
                                    {deadline.urgency} {deadline.duration}
                                </span>
                            </div>
                            {data.deadline?.includes(
                                `${deadline.urgency}#${deadline.duration}`
                            ) && (
                                <div className="absolute -top-2 -right-2 w-5 h-5 bg-blue-500 dark:bg-blue-400 rounded-full flex items-center justify-center">
                                    <svg
                                        className="w-3 h-3 text-white"
                                        fill="none"
                                        viewBox="0 0 24 24"
                                        stroke="currentColor"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M5 13l4 4L19 7"
                                        />
                                    </svg>
                                </div>
                            )}
                        </div>
                    ))}
                </div>

                {/* Deadline preview */}
                {deadlinePreview && (
                    <div className="mt-3 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-100 dark:border-blue-800">
                        <p className="text-sm text-gray-700 dark:text-gray-300">
                            <Calendar className="inline-block w-4 h-4 mr-1 text-blue-600 dark:text-blue-400" />
                            Your deadline:{" "}
                            <span className="font-semibold">
                                {deadlinePreview}
                            </span>
                            <span className="block text-xs mt-1 text-gray-500 dark:text-gray-400">
                                (in your local timezone:{" "}
                                {data.clientTimezone ||
                                    Intl.DateTimeFormat().resolvedOptions()
                                        .timeZone}
                                )
                            </span>
                        </p>
                    </div>
                )}

                {errors.deadline && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.deadline}
                    </p>
                )}
            </div>

            {/* Spacing selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Document Spacing <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select spacing"
                >
                    <div
                        onClick={() => handleSpacingChange("1#Double Spaced")}
                        className={`
                            flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                            ${
                                data.spacing?.includes("Double Spaced")
                                    ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                    : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                            }
                        `}
                        role="radio"
                        aria-checked={data.spacing?.includes("Double Spaced")}
                        tabIndex={0}
                        onKeyDown={(e) => {
                            if (e.key === "Enter" || e.key === " ") {
                                e.preventDefault();
                                handleSpacingChange("1#Double Spaced");
                            }
                        }}
                    >
                        <AlignCenter
                            className={`w-5 h-5 mr-3 ${
                                data.spacing?.includes("Double Spaced")
                                    ? "text-blue-600 dark:text-blue-400"
                                    : "text-gray-500 dark:text-gray-400"
                            }`}
                        />
                        <div className="flex flex-col">
                            <span
                                className={`text-sm font-medium ${
                                    data.spacing?.includes("Double Spaced")
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                Double Spaced
                            </span>
                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                Standard academic format
                            </span>
                        </div>
                    </div>

                    <div
                        onClick={() => handleSpacingChange("2#Single Spaced")}
                        className={`
                            flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                            ${
                                data.spacing?.includes("Single Spaced")
                                    ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                    : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                            }
                        `}
                        role="radio"
                        aria-checked={data.spacing?.includes("Single Spaced")}
                        tabIndex={0}
                        onKeyDown={(e) => {
                            if (e.key === "Enter" || e.key === " ") {
                                e.preventDefault();
                                handleSpacingChange("2#Single Spaced");
                            }
                        }}
                    >
                        <AlignLeft
                            className={`w-5 h-5 mr-3 ${
                                data.spacing?.includes("Single Spaced")
                                    ? "text-blue-600 dark:text-blue-400"
                                    : "text-gray-500 dark:text-gray-400"
                            }`}
                        />
                        <div className="flex flex-col">
                            <span
                                className={`text-sm font-medium ${
                                    data.spacing?.includes("Single Spaced")
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                Single Spaced
                            </span>
                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                More content per page
                            </span>
                        </div>
                    </div>
                </div>
                {errors.spacing && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.spacing}
                    </p>
                )}
            </div>

            {/* Number inputs section */}
            <div className="flex flex-col space-y-6">
                {/* Pages input */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Number of Pages <span className="text-red-500">*</span>
                    </label>
                    <div className="flex">
                        <button
                            type="button"
                            onClick={() => decrementField("pages")}
                            className="px-3 py-2 rounded-l-lg border border-r-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Decrease pages"
                        >
                            <Minus className="w-4 h-4" />
                        </button>
                        <input
                            type="number"
                            id="pages"
                            value={data.pages || 1}
                            onChange={(e) =>
                                handleNumberInput("pages", e.target.value)
                            }
                            min="1"
                            max="100"
                            className="w-full text-center py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white focus:ring-blue-500 focus:border-blue-500"
                            required
                            aria-required="true"
                        />
                        <button
                            type="button"
                            onClick={() => incrementField("pages")}
                            className="px-3 py-2 rounded-r-lg border border-l-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Increase pages"
                        >
                            <Plus className="w-4 h-4" />
                        </button>
                    </div>
                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                        275 words per page, double spaced
                    </p>
                    {errors.pages && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.pages}
                        </p>
                    )}
                </div>

                {/* PowerPoint Slides input */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        PowerPoint Slides (Optional)
                    </label>
                    <div className="flex">
                        <button
                            type="button"
                            onClick={() => decrementField("ppslides", 0)}
                            className="px-3 py-2 rounded-l-lg border border-r-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Decrease slides"
                        >
                            <Minus className="w-4 h-4" />
                        </button>
                        <input
                            type="number"
                            id="ppslides"
                            value={data.ppslides || 0}
                            onChange={(e) =>
                                handleNumberInput("ppslides", e.target.value, 0)
                            }
                            min="0"
                            max="100"
                            className="w-full text-center py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white focus:ring-blue-500 focus:border-blue-500"
                            aria-label="Number of PowerPoint slides"
                        />
                        <button
                            type="button"
                            onClick={() => incrementField("ppslides", 0)}
                            className="px-3 py-2 rounded-r-lg border border-l-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Increase slides"
                        >
                            <Plus className="w-4 h-4" />
                        </button>
                    </div>
                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                        {orderCalculation.currencySymbol}
                        {orderCalculation.currencySymbol === "$"
                            ? 4
                            : (4 * (data.currencyRate || 1)).toFixed(2)}{" "}
                        per slide
                    </p>
                    {errors.ppslides && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.ppslides}
                        </p>
                    )}
                </div>

                {/* Sources input */}
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Number of Sources{" "}
                        <span className="text-red-500">*</span>
                    </label>
                    <div className="flex">
                        <button
                            type="button"
                            onClick={() => decrementField("sources", 0)}
                            className="px-3 py-2 rounded-l-lg border border-r-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Decrease sources"
                        >
                            <Minus className="w-4 h-4" />
                        </button>
                        <input
                            type="number"
                            id="sources"
                            value={data.sources || 0}
                            onChange={(e) =>
                                handleNumberInput("sources", e.target.value, 0)
                            }
                            min="0"
                            max="50"
                            className="w-full text-center py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white focus:ring-blue-500 focus:border-blue-500"
                            required
                            aria-required="true"
                        />
                        <button
                            type="button"
                            onClick={() => incrementField("sources", 0)}
                            className="px-3 py-2 rounded-r-lg border border-l-0 border-gray-300 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:border-gray-600 dark:hover:bg-gray-600 dark:text-gray-300"
                            aria-label="Increase sources"
                        >
                            <Plus className="w-4 h-4" />
                        </button>
                    </div>
                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                        Number of references required
                    </p>
                    {errors.sources && (
                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                            {errors.sources}
                        </p>
                    )}
                </div>
            </div>

            {/* Price calculation summary */}
            {data.deadline && data.spacing && data.pages > 0 && (
                <div className="mt-8 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-100 dark:border-blue-800">
                    <h3 className="text-sm font-semibold text-blue-800 dark:text-blue-300 mb-2">
                        Price Calculation
                    </h3>
                    <div className="space-y-2">
                        <div className="flex justify-between">
                            <span className="text-sm text-gray-600 dark:text-gray-300">
                                Pages: {data.pages} ×{" "}
                                {orderCalculation.currencySymbol}
                                {orderCalculation.costPerPage}
                            </span>
                            <span className="text-sm font-medium text-gray-800 dark:text-gray-200">
                                {orderCalculation.currencySymbol}
                                {parseFloat(orderCalculation.basePrice).toFixed(
                                    2
                                )}
                            </span>
                        </div>

                        {data.ppslides > 0 && (
                            <div className="flex justify-between">
                                <span className="text-sm text-gray-600 dark:text-gray-300">
                                    PowerPoint Slides: {data.ppslides} ×{" "}
                                    {orderCalculation.currencySymbol}
                                    {orderCalculation.currencySymbol === "$"
                                        ? 4
                                        : (
                                              4 * (data.currencyRate || 1)
                                          ).toFixed(2)}
                                </span>
                                <span className="text-sm font-medium text-gray-800 dark:text-gray-200">
                                    {orderCalculation.currencySymbol}
                                    {parseFloat(
                                        orderCalculation.costPerSlide
                                    ).toFixed(2)}
                                </span>
                            </div>
                        )}

                        <div className="pt-2 border-t border-gray-200 dark:border-gray-700 flex justify-between">
                            <span className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                Total:
                            </span>
                            <span className="text-lg font-bold text-blue-600 dark:text-blue-300">
                                {orderCalculation.currencySymbol}
                                {orderCalculation.totalAmount}
                            </span>
                        </div>

                        {data.currency !== "USD" && (
                            <div className="text-right">
                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                    ($
                                    {(
                                        parseFloat(
                                            orderCalculation.totalAmount
                                        ) / (data.currencyRate || 1)
                                    ).toFixed(2)}{" "}
                                    USD)
                                </span>
                            </div>
                        )}
                    </div>
                </div>
            )}
        </div>
    );
};

export default ProjectSpecifications;
