import React from "react";
import { CheckCircle } from "lucide-react";

const StepIndicator = ({ currentStep, totalSteps = 4 }) => {
    // Ensure valid input values
    const validCurrentStep = currentStep
        ? Math.min(Math.max(1, currentStep), totalSteps)
        : 1;
    const validTotalSteps = totalSteps || 4;

    // Step labels
    const stepLabels = [
        "Basic Information",
        "Project Specifications",
        "Writing Preferences",
        "Review & Confirm",
    ];

    return (
        <nav aria-label="Order form progress" className="w-full mb-8">
            <ol className="relative flex flex-col sm:flex-row justify-between">
                {[...Array(validTotalSteps)].map((_, index) => {
                    // Determine step status
                    const isComplete = index + 1 < validCurrentStep;
                    const isCurrent = index + 1 === validCurrentStep;
                    const isPending = index + 1 > validCurrentStep;

                    return (
                        <li
                            key={index}
                            className={`
                            flex items-center
                            ${
                                index < validTotalSteps - 1
                                    ? "mb-6 sm:mb-0 sm:flex-1"
                                    : ""
                            }
                        `}
                        >
                            {/* Step connector line */}
                            {index < validTotalSteps - 1 && (
                                <div className="hidden sm:block absolute h-0.5 top-5 left-0 right-0 -translate-y-1/2 mx-12 z-0">
                                    <div
                                        className={`
                                        h-full
                                        ${
                                            isComplete
                                                ? "bg-green-500 dark:bg-green-400"
                                                : "bg-gray-200 dark:bg-gray-700"
                                        }
                                    `}
                                        style={{
                                            width: isComplete ? "100%" : "0%",
                                            transition:
                                                "width 0.5s ease-in-out",
                                        }}
                                    ></div>
                                </div>
                            )}

                            {/* Step indicator */}
                            <div className="flex flex-col items-center relative z-10">
                                <div
                                    className={`
                                    flex items-center justify-center w-10 h-10 rounded-full border-2 transition-all duration-300
                                    ${
                                        isComplete
                                            ? "border-green-500 bg-green-500 text-white dark:border-green-400 dark:bg-green-500"
                                            : isCurrent
                                            ? "border-blue-600 bg-blue-100 text-blue-600 dark:border-blue-400 dark:bg-blue-900 dark:text-blue-300"
                                            : "border-gray-300 bg-white text-gray-400 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-400"
                                    }
                                `}
                                >
                                    {isComplete ? (
                                        <CheckCircle
                                            className="w-5 h-5"
                                            aria-hidden="true"
                                        />
                                    ) : (
                                        <span
                                            className="text-sm font-medium"
                                            aria-hidden="true"
                                        >
                                            {index + 1}
                                        </span>
                                    )}
                                </div>
                                <span
                                    className={`
                                    mt-2 text-xs font-medium hidden sm:block
                                    ${
                                        isComplete
                                            ? "text-green-600 dark:text-green-400"
                                            : isCurrent
                                            ? "text-blue-600 dark:text-blue-400"
                                            : "text-gray-500 dark:text-gray-400"
                                    }
                                `}
                                >
                                    {stepLabels[index] || `Step ${index + 1}`}
                                </span>
                            </div>

                            {/* Mobile view step label */}
                            <span
                                className={`
                                ml-3 text-sm font-medium sm:hidden
                                ${
                                    isComplete
                                        ? "text-green-600 dark:text-green-400"
                                        : isCurrent
                                        ? "text-blue-600 dark:text-blue-400"
                                        : "text-gray-500 dark:text-gray-400"
                                }
                            `}
                            >
                                {stepLabels[index] || `Step ${index + 1}`}
                            </span>
                        </li>
                    );
                })}
            </ol>
        </nav>
    );
};

export default StepIndicator;
