import React, { useState, useEffect } from "react";
import {
    BookOpen,
    Book,
    FileText,
    Globe,
    Upload,
    FileUp,
    X,
    AlertCircle,
    Tag,
    Check,
    RefreshCw,
    Gift,
    TrendingDown,
} from "lucide-react";
import { FormTextarea } from "@/Components/ui/FormTextarea";
import axios from "axios";

const WritingPreferences = ({
    data,
    setData,
    errors,
    orderCalculation,
    validateCoupon,
    discountInfo,
    couponEnabled,
    suggestedCoupons,
}) => {
    const [couponCode, setCouponCode] = useState(data.couponCode || "");
    const [couponLoading, setCouponLoading] = useState(false);
    const [couponMessage, setCouponMessage] = useState(null);
    const [couponStatus, setCouponStatus] = useState(null); // 'success', 'error', null
    const [availableCoupons, setAvailableCoupons] = useState([]);
    const [showAvailableCoupons, setShowAvailableCoupons] = useState(false);

    // Format file size for display
    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
    };

    // Process suggested coupons on component mount
    useEffect(() => {
        if (suggestedCoupons && Array.isArray(suggestedCoupons)) {
            console.log("Suggested coupons received:", suggestedCoupons);
            // Parse any coupons that might be strings
            const parsedCoupons = suggestedCoupons.map((coupon) => {
                if (typeof coupon === "string") {
                    try {
                        return JSON.parse(coupon);
                    } catch (e) {
                        // If it's not valid JSON, return the original string
                        return coupon;
                    }
                }
                return coupon;
            });

            setAvailableCoupons(parsedCoupons);

            // Show available coupons section if we have valid coupons
            if (parsedCoupons.length > 0) {
                setShowAvailableCoupons(true);
            }
        }
    }, [suggestedCoupons]);

    // Handle file upload
    const handleFileUpload = (e) => {
        const files = Array.from(e.target.files);
        console.log('[FILE UPLOAD] Files selected:', {
            count: files.length,
            files: files.map(f => ({
                name: f.name,
                size: f.size,
                type: f.type
            }))
        });
        setData("files", files);
        console.log('[FILE UPLOAD] Files set in form data');
    };

    // Remove file from upload list
    const removeFile = (index) => {
        const newFiles = [...data.files];
        newFiles.splice(index, 1);
        setData("files", newFiles);
    };

    // Handle citation style selection
    const handleStyleChange = (value) => {
        setData("styles", value);
    };

    // Handle language preference selection
    const handleLanguageChange = (value) => {
        setData("language", value);
    };

    // Handle coupon code validation
    const handleValidateCoupon = async () => {
        if (!couponCode.trim()) {
            setCouponMessage("Please enter a coupon code");
            setCouponStatus("error");
            return;
        }

        setCouponLoading(true);
        setCouponMessage(null);
        setCouponStatus(null);

        try {
            const response = await axios.post("/validate-coupon", {
                code: couponCode,
                order_total: orderCalculation.totalAmount,
                pages: data.pages || 1,
            });

            console.log("Coupon API response:", response.data);

            if (response.data.valid) {
                setCouponStatus("success");
                setCouponMessage(
                    `Coupon applied: ${
                        response.data.description ||
                        "Discount will be applied to your order"
                    }`
                );

                // Use the correct field names from API response
                const discountType = response.data.type; // 'percentage' or 'fixed'
                const discountAmount = parseFloat(response.data.amount);
                const calculatedDiscount = parseFloat(
                    response.data.calculated_discount || 0
                );

                console.log("Setting discount values:", {
                    discountType,
                    discountAmount,
                    calculatedDiscount,
                });

                // Update all state fields with the correct values
                setData({
                    ...data,
                    couponCode: couponCode,
                    discountType: discountType,
                    discountAmount: discountAmount,
                    discount:
                        calculatedDiscount ||
                        (discountType === "percentage"
                            ? (parseFloat(orderCalculation.totalAmount) *
                                  discountAmount) /
                              100
                            : discountAmount),
                });
            } else {
                setCouponStatus("error");
                setCouponMessage(
                    response.data.message || "Invalid coupon code"
                );
                // Reset coupon data
                setData({
                    ...data,
                    couponCode: "",
                    discountType: null,
                    discountAmount: 0,
                    discount: 0,
                });
            }
        } catch (error) {
            setCouponStatus("error");
            setCouponMessage("Error validating coupon. Please try again.");
            console.error("Coupon validation error:", error);
        } finally {
            setCouponLoading(false);
        }
    };

    // Handle selecting a coupon from available ones
    const handleSelectCoupon = (coupon) => {
        setCouponCode(coupon.code);
        setData("couponCode", coupon.code);
        handleValidateCoupon();
    };

    return (
        <div className="space-y-8" aria-labelledby="step3-heading">
            {/* Citation Style selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Citation Style <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select citation style"
                >
                    {/* ... existing citation style options ... */}
                    {[
                        {
                            id: 1,
                            value: "1",
                            name: "APA",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                        {
                            id: 2,
                            value: "2",
                            name: "MLA",
                            icon: <Book className="w-5 h-5" />,
                        },
                        {
                            id: 3,
                            value: "3",
                            name: "Chicago",
                            icon: <FileText className="w-5 h-5" />,
                        },
                        {
                            id: 4,
                            value: "4",
                            name: "Harvard",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                        {
                            id: 5,
                            value: "5",
                            name: "Turabian",
                            icon: <Book className="w-5 h-5" />,
                        },
                        {
                            id: 6,
                            value: "6",
                            name: "Oxford",
                            icon: <FileText className="w-5 h-5" />,
                        },
                        {
                            id: 7,
                            value: "7",
                            name: "Vancouver",
                            icon: <BookOpen className="w-5 h-5" />,
                        },
                    ].map((style) => (
                        <div
                            key={style.id}
                            onClick={() => handleStyleChange(style.value)}
                            className={`
                                flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                ${
                                    data.styles === style.value
                                        ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                        : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                }
                            `}
                            role="radio"
                            aria-checked={data.styles === style.value}
                            tabIndex={0}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" || e.key === " ") {
                                    e.preventDefault();
                                    handleStyleChange(style.value);
                                }
                            }}
                        >
                            <div
                                className={`mr-3 ${
                                    data.styles === style.value
                                        ? "text-blue-600 dark:text-blue-400"
                                        : "text-gray-500 dark:text-gray-400"
                                }`}
                            >
                                {style.icon}
                            </div>
                            <span
                                className={`text-sm font-medium ${
                                    data.styles === style.value
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                {style.name}
                            </span>
                        </div>
                    ))}
                </div>
                {errors.styles && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.styles}
                    </p>
                )}
            </div>

            {/* Language Preference selection */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Language Preference <span className="text-red-500">*</span>
                </label>
                <div
                    className="grid grid-cols-1 sm:grid-cols-2 gap-3"
                    role="radiogroup"
                    aria-required="true"
                    aria-label="Select language preference"
                >
                    {/* ... existing language options ... */}
                    {[
                        {
                            id: 1,
                            value: "1",
                            name: "American English",
                            flag: "🇺🇸",
                        },
                        {
                            id: 2,
                            value: "2",
                            name: "British English",
                            flag: "🇬🇧",
                        },
                        {
                            id: 3,
                            value: "3",
                            name: "Australian English",
                            flag: "🇦🇺",
                        },
                        {
                            id: 4,
                            value: "4",
                            name: "Canadian English",
                            flag: "🇨🇦",
                        },
                    ].map((lang) => (
                        <div
                            key={lang.id}
                            onClick={() => handleLanguageChange(lang.value)}
                            className={`
                                flex items-center px-4 py-3 rounded-lg border-2 cursor-pointer transition-all
                                ${
                                    data.language === lang.value
                                        ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                        : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                }
                            `}
                            role="radio"
                            aria-checked={data.language === lang.value}
                            tabIndex={0}
                            onKeyDown={(e) => {
                                if (e.key === "Enter" || e.key === " ") {
                                    e.preventDefault();
                                    handleLanguageChange(lang.value);
                                }
                            }}
                        >
                            <span className="text-xl mr-3" aria-hidden="true">
                                {lang.flag}
                            </span>
                            <span
                                className={`text-sm font-medium ${
                                    data.language === lang.value
                                        ? "text-blue-700 dark:text-blue-300"
                                        : "text-gray-700 dark:text-gray-300"
                                }`}
                            >
                                {lang.name}
                            </span>
                        </div>
                    ))}
                </div>
                {errors.language && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.language}
                    </p>
                )}
            </div>

            {/* Special Discounts Section - NEW */}
            {suggestedCoupons && suggestedCoupons.length > 0 && (
                <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 p-4 rounded-lg border border-blue-100 dark:border-blue-800">
                    <div className="flex items-center mb-3">
                        <Gift className="h-5 w-5 text-blue-600 dark:text-blue-400 mr-2" />
                        <h3 className="text-md font-medium text-blue-800 dark:text-blue-300">
                            Special Discounts Available!
                        </h3>
                    </div>

                    <div className="space-y-3">
                        {suggestedCoupons.map((coupon, index) => {
                            // Handle both object and string formats
                            const couponData =
                                typeof coupon === "string"
                                    ? JSON.parse(coupon)
                                    : coupon;
                            return (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 p-3 rounded-md border border-blue-200 dark:border-blue-700 flex justify-between items-center hover:bg-blue-50 dark:hover:bg-blue-900/10 transition-colors cursor-pointer"
                                    onClick={() =>
                                        handleSelectCoupon(couponData)
                                    }
                                >
                                    <div>
                                        <div className="flex items-center">
                                            <TrendingDown className="h-4 w-4 text-green-600 dark:text-green-400 mr-2" />
                                            <span className="font-medium text-blue-700 dark:text-blue-300">
                                                {couponData.code}
                                            </span>
                                        </div>
                                        <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                            {couponData.description}
                                        </p>
                                    </div>
                                    <div className="text-right">
                                        <span className="inline-block px-2 py-1 bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300 text-xs font-semibold rounded">
                                            {couponData.discount_type ===
                                            "percentage"
                                                ? `${couponData.discount_amount}% OFF`
                                                : `${orderCalculation.currencySymbol}${couponData.discount_amount} OFF`}
                                        </span>
                                        <button
                                            className="block text-xs text-blue-600 dark:text-blue-400 mt-1 hover:underline"
                                            onClick={(e) => {
                                                e.stopPropagation();
                                                handleSelectCoupon(couponData);
                                            }}
                                        >
                                            Apply Now
                                        </button>
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                </div>
            )}

            {/* Coupon Code Input */}
            {couponEnabled && (
                <div>
                    <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                        Coupon Code{" "}
                        <span className="text-gray-500">(Optional)</span>
                    </label>
                    <div className="flex">
                        <div className="relative flex-grow">
                            <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                <Tag className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                            </div>
                            <input
                                type="text"
                                className="bg-white border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 p-2.5 dark:bg-gray-700 dark:border-gray-600 dark:placeholder-gray-400 dark:text-white"
                                placeholder="Enter coupon code"
                                value={couponCode}
                                onChange={(e) =>
                                    setCouponCode(e.target.value.toUpperCase())
                                }
                            />
                        </div>
                        <button
                            type="button"
                            onClick={handleValidateCoupon}
                            className="ml-2 px-4 py-2.5 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:ring-4 focus:ring-blue-300 dark:bg-blue-500 dark:hover:bg-blue-600"
                            disabled={couponLoading}
                        >
                            {couponLoading ? (
                                <RefreshCw className="w-5 h-5 animate-spin" />
                            ) : (
                                "Apply"
                            )}
                        </button>
                    </div>

                    {/* Coupon validation message */}
                    {couponStatus && (
                        <div
                            className={`mt-2 p-2 rounded-md flex items-start ${
                                couponStatus === "success"
                                    ? "bg-green-50 dark:bg-green-900/20 text-green-700 dark:text-green-300"
                                    : "bg-red-50 dark:bg-red-900/20 text-red-700 dark:text-red-300"
                            }`}
                        >
                            {couponStatus === "success" ? (
                                <Check className="w-5 h-5 mr-2 flex-shrink-0" />
                            ) : (
                                <AlertCircle className="w-5 h-5 mr-2 flex-shrink-0" />
                            )}
                            <span className="text-sm">{couponMessage}</span>
                        </div>
                    )}
                </div>
            )}

            {/* Instructions textarea */}
            <div>
                <FormTextarea
                    id="instructions"
                    name="instructions"
                    label="Instructions"
                    value={data.instructions || ""}
                    setData={setData}
                    error={errors.instructions}
                    placeholder="Provide detailed instructions for your assignment. "
                    className="dark:text-gray-200 min-h-40"
                    required={true}
                    rows={6}
                />
                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                    Be as detailed as possible to ensure your writer understands
                    your requirements exactly.
                </p>
            </div>

            {/* File Upload */}
            <div>
                <label className="block text-sm font-medium text-gray-900 dark:text-gray-200 mb-3">
                    Upload Files (Optional)
                </label>
                <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-dashed rounded-lg border-gray-300 dark:border-gray-600">
                    <div className="space-y-3 text-center">
                        <FileUp
                            className="mx-auto h-12 w-12 text-gray-400"
                            aria-hidden="true"
                        />
                        <div className="flex text-sm text-gray-600 dark:text-gray-400">
                            <label
                                htmlFor="file-upload"
                                className="relative cursor-pointer  dark:bg-gray-800 rounded-md font-medium text-blue-600 dark:text-blue-400 hover:text-blue-500 dark:hover:text-blue-300 focus:outline-none p-2"
                            >
                                <span className="text-xs bg-white dark:bg-gray-800 rounded-md px-2 py-1 border border-gray-300 dark:border-gray-600">
                                    Click to Upload{" "}
                                </span>
                                <input
                                    id="file-upload"
                                    name="file-upload"
                                    type="file"
                                    className="sr-only"
                                    onChange={handleFileUpload}
                                    multiple
                                    accept=".pdf,.doc,.docx,.txt,.rtf,.jpg,.jpeg,.png,.xlsx,.pptx"
                                />
                            </label>
                        </div>
                        <p className="text-xs text-gray-500 dark:text-gray-400">
                            PDF, Word, Excel, PowerPoint, images up to 10MB
                            each. (For Large files, Upload in your order
                            management page)
                        </p>
                    </div>
                </div>

                {/* Files list */}
                {data.files?.length > 0 && (
                    <div className="mt-4">
                        <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Selected Files ({data.files.length})
                        </h4>
                        <ul className="divide-y divide-gray-200 dark:divide-gray-700 border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden">
                            {Array.from(data.files).map((file, index) => (
                                <li
                                    key={index}
                                    className="px-4 py-3 flex items-center justify-between bg-white dark:bg-gray-800"
                                >
                                    <div className="flex items-center">
                                        <FileText className="w-5 h-5 text-gray-500 dark:text-gray-400 mr-3" />
                                        <div className="flex flex-col">
                                            <span
                                                className="text-sm font-medium text-gray-700 dark:text-gray-300 truncate"
                                                style={{ maxWidth: "200px" }}
                                            >
                                                {file.name}
                                            </span>
                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                {formatFileSize(file.size)}
                                            </span>
                                        </div>
                                    </div>
                                    <button
                                        type="button"
                                        onClick={() => removeFile(index)}
                                        className="text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300"
                                        aria-label={`Remove file ${file.name}`}
                                    >
                                        <X className="w-5 h-5" />
                                    </button>
                                </li>
                            ))}
                        </ul>
                    </div>
                )}

                {errors.files && (
                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                        {errors.files}
                    </p>
                )}
            </div>

            {/* Final price display */}
            <div className="mt-8 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-100 dark:border-blue-800">
                <h3 className="text-sm font-semibold text-blue-800 dark:text-blue-300 mb-2">
                    Order Summary
                </h3>

                <div className="space-y-2">
                    <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-700 dark:text-gray-300">
                            Subtotal:
                        </span>
                        <span className="text-sm font-medium text-gray-900 dark:text-white">
                            {orderCalculation.currencySymbol}
                            {orderCalculation.totalAmount}
                        </span>
                    </div>

                    {/* Show discount if coupon is valid and has been applied */}
                    {couponStatus === "success" && data.discountAmount > 0 && (
                        <div className="flex justify-between items-center text-green-600 dark:text-green-400">
                            <span className="text-sm flex items-center">
                                <Tag className="w-4 h-4 mr-1" />
                                Discount ({data.couponCode}):
                            </span>
                            <span className="text-sm font-medium">
                                - {orderCalculation.currencySymbol}
                                {data.discountType === "percentage"
                                    ? (
                                          (parseFloat(
                                              orderCalculation.totalAmount
                                          ) *
                                              parseFloat(data.discountAmount)) /
                                          100
                                      ).toFixed(2)
                                    : parseFloat(data.discountAmount).toFixed(
                                          2
                                      )}
                            </span>
                        </div>
                    )}

                    <div className="flex justify-between items-center pt-2 border-t border-gray-200 dark:border-gray-700">
                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                            Total Amount:
                        </span>
                        <span className="text-xl font-bold text-blue-600 dark:text-blue-300">
                            {orderCalculation.currencySymbol}
                            {couponStatus === "success" &&
                            data.discountAmount > 0
                                ? data.discountType === "percentage"
                                    ? (
                                          parseFloat(
                                              orderCalculation.totalAmount
                                          ) -
                                          (parseFloat(
                                              orderCalculation.totalAmount
                                          ) *
                                              parseFloat(data.discountAmount)) /
                                              100
                                      ).toFixed(2)
                                    : (
                                          parseFloat(
                                              orderCalculation.totalAmount
                                          ) - parseFloat(data.discountAmount)
                                      ).toFixed(2)
                                : orderCalculation.totalAmount}
                        </span>
                    </div>
                </div>

                {data.currency !== "USD" && (
                    <div className="text-right mt-1">
                        <span className="text-xs text-gray-500 dark:text-gray-400">
                            ($
                            {(
                                parseFloat(orderCalculation.totalAmount) /
                                (data.currencyRate || 1)
                            ).toFixed(2)}{" "}
                            USD)
                        </span>
                    </div>
                )}
                <p className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                    Review your order on the next step before submitting.
                </p>
            </div>
        </div>
    );
};

export default WritingPreferences;
