import React from 'react';
import { 
    Clock, 
    CreditCard, 
    Users, 
    Play, 
    FileText, 
    Eye, 
    CheckCircle, 
    AlertCircle,
    XCircle,
    RotateCcw
} from 'lucide-react';

const OrderTimeline = ({ order, className = '' }) => {
    const timelineSteps = [
        {
            id: 'pending',
            title: 'Order Placed',
            description: 'Order has been created and is awaiting payment',
            icon: Clock,
            status: 'pending',
            color: 'text-gray-500',
            bgColor: 'bg-gray-100',
            borderColor: 'border-gray-300'
        },
        {
            id: 'bidding',
            title: 'Payment Completed',
            description: 'Order is now available for writers to bid',
            icon: CreditCard,
            status: 'bidding',
            color: 'text-blue-500',
            bgColor: 'bg-blue-100',
            borderColor: 'border-blue-300'
        },
        {
            id: 'assigned',
            title: 'Writer Assigned',
            description: 'A writer has been selected and assigned to the order',
            icon: Users,
            status: 'assigned',
            color: 'text-purple-500',
            bgColor: 'bg-purple-100',
            borderColor: 'border-purple-300'
        },
        {
            id: 'in_progress',
            title: 'Work in Progress',
            description: 'Writer is currently working on the assignment',
            icon: Play,
            status: 'in_progress',
            color: 'text-orange-500',
            bgColor: 'bg-orange-100',
            borderColor: 'border-orange-300'
        },
        {
            id: 'submitted',
            title: 'Work Submitted',
            description: 'Writer has submitted the completed work',
            icon: FileText,
            status: 'submitted',
            color: 'text-indigo-500',
            bgColor: 'bg-indigo-100',
            borderColor: 'border-indigo-300'
        },
        {
            id: 'under_review',
            title: 'Under Review',
            description: 'Work is being reviewed by admin and client',
            icon: Eye,
            status: 'under_review',
            color: 'text-yellow-500',
            bgColor: 'bg-yellow-100',
            borderColor: 'border-yellow-300'
        },
        {
            id: 'approved',
            title: 'Completed',
            description: 'Order has been approved and completed',
            icon: CheckCircle,
            status: 'approved',
            color: 'text-green-500',
            bgColor: 'bg-green-100',
            borderColor: 'border-green-300'
        }
    ];

    const getCurrentStepIndex = () => {
        const currentStatus = order.order_status;
        const stepIndex = timelineSteps.findIndex(step => step.status === currentStatus);
        return stepIndex >= 0 ? stepIndex : 0;
    };

    const isStepCompleted = (stepIndex) => {
        const currentIndex = getCurrentStepIndex();
        return stepIndex < currentIndex;
    };

    const isStepCurrent = (stepIndex) => {
        const currentIndex = getCurrentStepIndex();
        return stepIndex === currentIndex;
    };

    const getStepStatus = (stepIndex) => {
        if (isStepCompleted(stepIndex)) return 'completed';
        if (isStepCurrent(stepIndex)) return 'current';
        return 'pending';
    };

    const getStatusHistory = () => {
        if (!order.status_history || !Array.isArray(order.status_history)) {
            return [];
        }
        return order.status_history.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
    };

    const formatDate = (dateString) => {
        if (!dateString) return null;
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getStatusDisplayName = (status) => {
        const statusNames = {
            'pending': 'Pending',
            'bidding': 'Bidding',
            'assigned': 'Assigned',
            'in_progress': 'In Progress',
            'submitted': 'Submitted',
            'under_review': 'Under Review',
            'approved': 'Approved',
            'revision_requested': 'Revision Requested',
            'cancelled': 'Cancelled'
        };
        return statusNames[status] || status;
    };

    return (
        <div className={`bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 ${className}`}>
            <div className="p-6">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-6">
                    Order Timeline
                </h3>

                {/* Timeline Steps */}
                <div className="relative">
                    {timelineSteps.map((step, index) => {
                        const Icon = step.icon;
                        const stepStatus = getStepStatus(index);
                        const isCompleted = stepStatus === 'completed';
                        const isCurrent = stepStatus === 'current';

                        return (
                            <div key={step.id} className="relative flex items-start mb-8 last:mb-0">
                                {/* Timeline Line */}
                                {index < timelineSteps.length - 1 && (
                                    <div className={`absolute left-6 top-12 w-0.5 h-8 ${
                                        isCompleted ? 'bg-green-400' : 'bg-gray-300 dark:bg-gray-600'
                                    }`} />
                                )}

                                {/* Step Icon */}
                                <div className={`flex-shrink-0 w-12 h-12 rounded-full flex items-center justify-center border-2 ${
                                    isCompleted 
                                        ? 'bg-green-500 border-green-500 text-white' 
                                        : isCurrent 
                                            ? `${step.bgColor} ${step.borderColor} ${step.color}`
                                            : 'bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-400'
                                }`}>
                                    <Icon className="w-5 h-5" />
                                </div>

                                {/* Step Content */}
                                <div className="ml-4 flex-1">
                                    <div className="flex items-center justify-between">
                                        <h4 className={`text-sm font-medium ${
                                            isCompleted 
                                                ? 'text-green-600 dark:text-green-400' 
                                                : isCurrent 
                                                    ? step.color 
                                                    : 'text-gray-500 dark:text-gray-400'
                                        }`}>
                                            {step.title}
                                        </h4>
                                        {isCurrent && (
                                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${step.bgColor} ${step.color}`}>
                                                Current
                                            </span>
                                        )}
                                    </div>
                                    <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                        {step.description}
                                    </p>
                                    
                                    {/* Status-specific information */}
                                    {isCurrent && order.order_status === 'under_review' && order.client_review_started_at && (
                                        <div className="mt-2 p-3 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-md">
                                            <p className="text-sm text-gray-700 dark:text-gray-300">
                                                Client review started: {formatDate(order.client_review_started_at)}
                                            </p>
                                            {order.auto_approval_at && (
                                                <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                                    Auto-approval scheduled: {formatDate(order.auto_approval_at)}
                                                </p>
                                            )}
                                        </div>
                                    )}

                                    {isCurrent && order.order_status === 'revision_requested' && (
                                        <div className="mt-2 p-3 bg-gradient-to-r from-orange-100 to-red-100 dark:bg-none border border-orange-300 dark:border-orange-600 rounded-md">
                                            <div className="flex items-start">
                                                <RotateCcw className="w-4 h-4 text-orange-500 mt-0.5 mr-2 flex-shrink-0" />
                                                <div>
                                                    <p className="text-sm font-medium text-orange-800 dark:text-orange-200">
                                                        Revision Requested
                                                    </p>
                                                    {order.revision_notes && (
                                                        <p className="text-xs text-orange-700 dark:text-orange-300 mt-1">
                                                            {order.revision_notes}
                                                        </p>
                                                    )}
                                                    <p className="text-xs text-orange-600 dark:text-orange-400 mt-1">
                                                        Revision count: {order.revision_count || 0}
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>
                        );
                    })}
                </div>

                {/* Status History */}
                {getStatusHistory().length > 0 && (
                    <div className="mt-8 pt-6 border-t border-gray-200 dark:border-gray-700">
                        <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-4">
                            Recent Status Changes
                        </h4>
                        <div className="space-y-3">
                            {getStatusHistory().slice(0, 5).map((history, index) => (
                                <div key={index} className="flex items-center justify-between text-sm">
                                    <div className="flex items-center">
                                        <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                            history.to_status === 'approved' ? 'bg-green-100 text-green-800' :
                                            history.to_status === 'cancelled' ? 'bg-red-100 text-red-800' :
                                            history.to_status === 'revision_requested' ? 'bg-orange-100 text-orange-800' :
                                            'bg-gray-100 text-gray-800'
                                        }`}>
                                            {getStatusDisplayName(history.to_status)}
                                        </span>
                                        {history.reason && (
                                            <span className="ml-2 text-gray-600 dark:text-gray-400">
                                                - {history.reason}
                                            </span>
                                        )}
                                    </div>
                                    <span className="text-gray-500 dark:text-gray-400">
                                        {formatDate(history.created_at)}
                                    </span>
                                </div>
                            ))}
                        </div>
                    </div>
                )}

                {/* Key Dates */}
                <div className="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                    <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-4">
                        Key Dates
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div>
                            <span className="text-gray-500 dark:text-gray-400">Order Created:</span>
                            <span className="ml-2 text-gray-900 dark:text-white">
                                {formatDate(order.dateposted)}
                            </span>
                        </div>
                        <div>
                            <span className="text-gray-500 dark:text-gray-400">Client Deadline:</span>
                            <span className="ml-2 text-gray-900 dark:text-white">
                                {formatDate(order.deadline)}
                            </span>
                        </div>
                        {order.writer_deadline && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Writer Deadline:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.writer_deadline)}
                                </span>
                            </div>
                        )}
                        {order.writer_assigned_at && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Writer Assigned:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.writer_assigned_at)}
                                </span>
                            </div>
                        )}
                        {order.work_started_at && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Work Started:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.work_started_at)}
                                </span>
                            </div>
                        )}
                        {order.work_submitted_at && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Work Submitted:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.work_submitted_at)}
                                </span>
                            </div>
                        )}
                        {order.admin_reviewed_at && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Admin Reviewed:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.admin_reviewed_at)}
                                </span>
                            </div>
                        )}
                        {order.client_approved_at && (
                            <div>
                                <span className="text-gray-500 dark:text-gray-400">Client Approved:</span>
                                <span className="ml-2 text-gray-900 dark:text-white">
                                    {formatDate(order.client_approved_at)}
                                </span>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </div>
    );
};

export default OrderTimeline; 