import { Head } from "@inertiajs/react";
import { useEffect } from "react";

export default function PerformanceOptimization() {
    useEffect(() => {
        // Preload critical resources
        const preloadCriticalResources = () => {
            // Preload hero section background image
            const heroImage = new Image();
            heroImage.src = "/images/academic-writing-hero-bg.jpg";

            // Preload testimonial images
            const testimonialImages = [
                "/images/testimonial-1.jpg",
                "/images/testimonial-2.jpg",
                "/images/testimonial-3.jpg",
            ];

            testimonialImages.forEach((src) => {
                const img = new Image();
                img.src = src;
            });
        };

        // Implement lazy loading for non-critical images
        const implementLazyLoading = () => {
            if ("IntersectionObserver" in window) {
                const imageObserver = new IntersectionObserver(
                    (entries, observer) => {
                        entries.forEach((entry) => {
                            if (entry.isIntersecting) {
                                const img = entry.target;
                                img.src = img.dataset.src;
                                img.classList.remove("lazy");
                                imageObserver.unobserve(img);
                            }
                        });
                    }
                );

                document.querySelectorAll("img[data-src]").forEach((img) => {
                    imageObserver.observe(img);
                });
            }
        };

        // Optimize scroll performance
        const optimizeScrollPerformance = () => {
            let ticking = false;

            const updateScrollPosition = () => {
                // Update scroll-dependent elements
                const scrolled = window.pageYOffset;
                const rate = scrolled * -0.5;

                // Parallax effect for hero section
                const hero = document.querySelector(".hero-parallax");
                if (hero) {
                    hero.style.transform = `translateY(${rate}px)`;
                }

                ticking = false;
            };

            const requestTick = () => {
                if (!ticking) {
                    requestAnimationFrame(updateScrollPosition);
                    ticking = true;
                }
            };

            window.addEventListener("scroll", requestTick, { passive: true });
        };

        // Implement service worker for caching
        const registerServiceWorker = () => {
            if ("serviceWorker" in navigator) {
                navigator.serviceWorker
                    .register("/sw.js")
                    .then((registration) => {
                        console.log("SW registered: ", registration);
                    })
                    .catch((registrationError) => {
                        console.log(
                            "SW registration failed: ",
                            registrationError
                        );
                    });
            }
        };

        // Initialize performance optimizations
        preloadCriticalResources();
        implementLazyLoading();
        optimizeScrollPerformance();
        registerServiceWorker();

        // Cleanup
        return () => {
            window.removeEventListener("scroll", () => {});
        };
    }, []);

    return (
        <Head>
            {/* DNS Prefetch for external domains */}
            <link rel="dns-prefetch" href="//fonts.googleapis.com" />
            <link rel="dns-prefetch" href="//fonts.gstatic.com" />
            <link rel="dns-prefetch" href="//www.google-analytics.com" />
            <link rel="dns-prefetch" href="//www.googletagmanager.com" />
            <link rel="dns-prefetch" href="//connect.facebook.net" />

            {/* Preconnect to critical third-party origins */}
            <link rel="preconnect" href="https://fonts.googleapis.com" />
            <link
                rel="preconnect"
                href="https://fonts.gstatic.com"
                crossOrigin="anonymous"
            />
            <link rel="preconnect" href="https://www.google-analytics.com" />
            <link rel="preconnect" href="https://www.googletagmanager.com" />

            {/* Preload critical resources */}
            <link
                rel="preload"
                href="/fonts/inter-var.woff2"
                as="font"
                type="font/woff2"
                crossOrigin="anonymous"
            />
            <link
                rel="preload"
                href="/images/academic-writing-hero-bg.webp"
                as="image"
                type="image/webp"
            />
            <link rel="preload" href="/css/critical.css" as="style" />

            {/* Prefetch next likely pages */}
            <link rel="prefetch" href="/services" />
            <link rel="prefetch" href="/order" />
            <link rel="prefetch" href="/pricing" />
            <link rel="prefetch" href="/about" />

            {/* Resource hints for academic writing service pages */}
            <link rel="prefetch" href="/services/dissertation-writing" />
            <link rel="prefetch" href="/services/research-papers" />
            <link rel="prefetch" href="/services/essay-writing" />
            <link rel="prefetch" href="/services/custom-papers" />

            {/* Critical CSS inline */}
            <style
                dangerouslySetInnerHTML={{
                    __html: `
                    /* Critical CSS for above-the-fold content */
                    .hero-section {
                        background: linear-gradient(135deg, #1e3a8a 0%, #06b6d4 100%);
                        min-height: 100vh;
                        display: flex;
                        align-items: center;
                        position: relative;
                        overflow: hidden;
                    }

                    .hero-content {
                        position: relative;
                        z-index: 2;
                        max-width: 1200px;
                        margin: 0 auto;
                        padding: 0 1rem;
                    }

                    .hero-title {
                        font-size: clamp(2rem, 5vw, 3.5rem);
                        font-weight: 700;
                        line-height: 1.2;
                        color: white;
                        margin-bottom: 1rem;
                    }

                    .hero-subtitle {
                        font-size: clamp(1rem, 2.5vw, 1.25rem);
                        color: rgba(255, 255, 255, 0.9);
                        margin-bottom: 2rem;
                        max-width: 600px;
                    }

                    .cta-button {
                        display: inline-block;
                        padding: 0.75rem 2rem;
                        background: white;
                        color: #1e3a8a;
                        text-decoration: none;
                        border-radius: 0.5rem;
                        font-weight: 600;
                        transition: transform 0.2s ease, box-shadow 0.2s ease;
                    }

                    .cta-button:hover {
                        transform: translateY(-2px);
                        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
                    }

                    /* Loading states */
                    .lazy {
                        opacity: 0;
                        transition: opacity 0.3s;
                    }

                    .lazy.loaded {
                        opacity: 1;
                    }

                    /* Skeleton loading for testimonials */
                    .testimonial-skeleton {
                        background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
                        background-size: 200% 100%;
                        animation: loading 1.5s infinite;
                    }

                    @keyframes loading {
                        0% { background-position: 200% 0; }
                        100% { background-position: -200% 0; }
                    }

                    /* Optimize font loading */
                    @font-face {
                        font-family: 'Inter';
                        font-style: normal;
                        font-weight: 100 900;
                        font-display: swap;
                        src: url('/fonts/inter-var.woff2') format('woff2');
                        unicode-range: U+0000-00FF, U+0131, U+0152-0153, U+02BB-02BC, U+02C6, U+02DA, U+02DC, U+2000-206F, U+2074, U+20AC, U+2122, U+2191, U+2193, U+2212, U+2215, U+FEFF, U+FFFD;
                    }
                `,
                }}
            />

            {/* Performance monitoring script */}
            <script
                dangerouslySetInnerHTML={{
                    __html: `
                    // Core Web Vitals monitoring
                    function getCLS(onPerfEntry) {
                        if (onPerfEntry && onPerfEntry instanceof Function) {
                            import('web-vitals').then(({getCLS}) => getCLS(onPerfEntry));
                        }
                    }

                    function getFID(onPerfEntry) {
                        if (onPerfEntry && onPerfEntry instanceof Function) {
                            import('web-vitals').then(({getFID}) => getFID(onPerfEntry));
                        }
                    }

                    function getFCP(onPerfEntry) {
                        if (onPerfEntry && onPerfEntry instanceof Function) {
                            import('web-vitals').then(({getFCP}) => getFCP(onPerfEntry));
                        }
                    }

                    function getLCP(onPerfEntry) {
                        if (onPerfEntry && onPerfEntry instanceof Function) {
                            import('web-vitals').then(({getLCP}) => getLCP(onPerfEntry));
                        }
                    }

                    function getTTFB(onPerfEntry) {
                        if (onPerfEntry && onPerfEntry instanceof Function) {
                            import('web-vitals').then(({getTTFB}) => getTTFB(onPerfEntry));
                        }
                    }

                    // Send performance metrics to analytics
                    function sendToAnalytics(metric) {
                        if (typeof gtag !== 'undefined') {
                            gtag('event', metric.name, {
                                event_category: 'Web Vitals',
                                event_label: metric.id,
                                value: Math.round(metric.name === 'CLS' ? metric.value * 1000 : metric.value),
                                non_interaction: true,
                                custom_parameter_page_type: 'academic_writing_homepage'
                            });
                        }
                    }

                    // Initialize Core Web Vitals tracking
                    getCLS(sendToAnalytics);
                    getFID(sendToAnalytics);
                    getFCP(sendToAnalytics);
                    getLCP(sendToAnalytics);
                    getTTFB(sendToAnalytics);

                    // Image optimization
                    function optimizeImages() {
                        const images = document.querySelectorAll('img');
                        images.forEach(img => {
                            // Add loading="lazy" to images below the fold
                            if (img.getBoundingClientRect().top > window.innerHeight) {
                                img.loading = 'lazy';
                            }

                            // Add error handling
                            img.onerror = function() {
                                this.src = '/images/placeholder-academic-writing.jpg';
                            };
                        });
                    }

                    // Optimize third-party scripts loading
                    function loadThirdPartyScripts() {
                        // Delay non-critical scripts
                        setTimeout(() => {
                            // Load chat widget after 3 seconds
                            if (typeof loadChatWidget === 'function') {
                                loadChatWidget();
                            }
                        }, 3000);

                        // Load social media widgets on user interaction
                        ['click', 'scroll', 'keydown'].forEach(event => {
                                                       document.addEventListener(event, function loadSocialWidgets() {
                                // Load Facebook SDK
                                if (!document.getElementById('facebook-jssdk')) {
                                    const fjs = document.getElementsByTagName('script')[0];
                                    const js = document.createElement('script');
                                    js.id = 'facebook-jssdk';
                                    js.src = 'https://connect.facebook.net/en_US/sdk.js#xfbml=1&version=v18.0';
                                    fjs.parentNode.insertBefore(js, fjs);
                                }

                                // Load Twitter widgets
                                if (!window.twttr) {
                                    window.twttr = (function(d, s, id) {
                                        var js, fjs = d.getElementsByTagName(s)[0],
                                            t = window.twttr || {};
                                        if (d.getElementById(id)) return t;
                                        js = d.createElement(s);
                                        js.id = id;
                                        js.src = 'https://platform.twitter.com/widgets.js';
                                        fjs.parentNode.insertBefore(js, fjs);
                                        t._e = [];
                                        t.ready = function(f) {
                                            t._e.push(f);
                                        };
                                        return t;
                                    }(document, 'script', 'twitter-wjs'));
                                }

                                // Remove event listeners after loading
                                ['click', 'scroll', 'keydown'].forEach(evt => {
                                    document.removeEventListener(evt, loadSocialWidgets);
                                });
                            }, { once: true });
                        });
                    }

                    // Initialize performance optimizations when DOM is ready
                    if (document.readyState === 'loading') {
                        document.addEventListener('DOMContentLoaded', function() {
                            optimizeImages();
                            loadThirdPartyScripts();
                        });
                    } else {
                        optimizeImages();
                        loadThirdPartyScripts();
                    }

                    // Preload next page on hover
                    function preloadOnHover() {
                        const links = document.querySelectorAll('a[href^="/"]');
                        links.forEach(link => {
                            link.addEventListener('mouseenter', function() {
                                const href = this.getAttribute('href');
                                if (href && !document.querySelector(\`link[rel="prefetch"][href="\${href}"]\`)) {
                                    const prefetchLink = document.createElement('link');
                                    prefetchLink.rel = 'prefetch';
                                    prefetchLink.href = href;
                                    document.head.appendChild(prefetchLink);
                                }
                            }, { once: true });
                        });
                    }

                    // Initialize hover preloading
                    setTimeout(preloadOnHover, 2000);

                    // Academic writing service specific optimizations
                    window.academicWritingOptimizations = {
                        // Optimize quote form loading
                        optimizeQuoteForm: function() {
                            const quoteForm = document.querySelector('#quote-form');
                            if (quoteForm) {
                                // Lazy load form validation library
                                import('/js/form-validation.js').then(module => {
                                    module.initializeValidation(quoteForm);
                                });
                            }
                        },

                        // Optimize testimonial carousel
                        optimizeTestimonials: function() {
                            const testimonialCarousel = document.querySelector('.testimonial-carousel');
                            if (testimonialCarousel) {
                                // Intersection Observer for testimonial loading
                                const observer = new IntersectionObserver((entries) => {
                                    entries.forEach(entry => {
                                        if (entry.isIntersecting) {
                                            import('/js/testimonial-carousel.js').then(module => {
                                                module.initializeCarousel(testimonialCarousel);
                                            });
                                            observer.unobserve(entry.target);
                                        }
                                    });
                                });
                                observer.observe(testimonialCarousel);
                            }
                        },

                        // Optimize FAQ section
                        optimizeFAQ: function() {
                            const faqSection = document.querySelector('#faq');
                            if (faqSection) {
                                const observer = new IntersectionObserver((entries) => {
                                    entries.forEach(entry => {
                                        if (entry.isIntersecting) {
                                            import('/js/faq-interactions.js').then(module => {
                                                module.initializeFAQ(faqSection);
                                            });
                                            observer.unobserve(entry.target);
                                        }
                                    });
                                });
                                observer.observe(faqSection);
                            }
                        }
                    };

                    // Initialize academic writing optimizations
                    setTimeout(() => {
                        window.academicWritingOptimizations.optimizeQuoteForm();
                        window.academicWritingOptimizations.optimizeTestimonials();
                        window.academicWritingOptimizations.optimizeFAQ();
                    }, 1000);
                `,
                }}
            />

            {/* Service Worker registration */}
            <script
                dangerouslySetInnerHTML={{
                    __html: `
                    // Service Worker for caching academic writing service assets
                    if ('serviceWorker' in navigator) {
                        window.addEventListener('load', function() {
                            navigator.serviceWorker.register('/sw.js', {
                                scope: '/'
                            }).then(function(registration) {
                                console.log('Academic Writing SW registered: ', registration.scope);

                                // Update available
                                registration.addEventListener('updatefound', function() {
                                    const newWorker = registration.installing;
                                    newWorker.addEventListener('statechange', function() {
                                        if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                                            // New content available, refresh page
                                            if (confirm('New content available! Refresh to update?')) {
                                                window.location.reload();
                                            }
                                        }
                                    });
                                });
                            }).catch(function(registrationError) {
                                console.log('Academic Writing SW registration failed: ', registrationError);
                            });
                        });
                    }
                `,
                }}
            />

            {/* Optimize Google Fonts loading */}
            <link
                rel="preload"
                href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap"
                as="style"
                onLoad="this.onload=null;this.rel='stylesheet'"
            />
            <noscript>
                <link
                    rel="stylesheet"
                    href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap"
                />
            </noscript>

            {/* Optimize critical resource loading */}
            <script
                dangerouslySetInnerHTML={{
                    __html: `
                    // Optimize font loading for academic writing content
                    if ('fonts' in document) {
                        Promise.all([
                            document.fonts.load('400 1em Inter'),
                            document.fonts.load('600 1em Inter'),
                            document.fonts.load('700 1em Inter')
                        ]).then(function() {
                            document.documentElement.classList.add('fonts-loaded');
                        });
                    }

                    // Optimize image loading with WebP support
                    function supportsWebP() {
                        return new Promise(resolve => {
                            const webP = new Image();
                            webP.onload = webP.onerror = function () {
                                resolve(webP.height === 2);
                            };
                            webP.src = 'data:image/webp;base64,UklGRjoAAABXRUJQVlA4IC4AAACyAgCdASoCAAIALmk0mk0iIiIiIgBoSygABc6WWgAA/veff/0PP8bA//LwYAAA';
                        });
                    }

                    supportsWebP().then(function(supported) {
                        if (supported) {
                            document.documentElement.classList.add('webp-supported');
                        }
                    });
                `,
                }}
            />
        </Head>
    );
}
