import React, { useState, useEffect } from 'react';
import Modal from '@/Components/Modal';
import { X, Loader2 } from 'lucide-react';
import axios from 'axios';
import PriceCalculatorForm from './PriceCalculatorForm';
import PriceCalculatorResults from './PriceCalculatorResults';

export default function PriceCalculatorPopup({
    isOpen,
    onClose,
    triggerSource = 'button',
    isAuthenticated = false,
    user = null,
    initialData = {},
}) {
    const [pricingData, setPricingData] = useState({
        academicLevels: [],
        paperTypes: [],
        subjects: [],
        deadlines: [],
        singleSlidePrice: 5,
    });
    const [isLoadingData, setIsLoadingData] = useState(false);
    const [dataError, setDataError] = useState(null);
    const [formData, setFormData] = useState({
        academic_level_id: initialData.academic_level_id || null,
        paper_type_id: initialData.paper_type_id || null,
        deadline_id: initialData.deadline_id || null,
        pages: 1,
        subject_id: initialData.subject_id || null,
        slides: 0,
        spacing: 'double',
        coupon_code: '',
    });

    const [pricing, setPricing] = useState(null);
    const [isCalculating, setIsCalculating] = useState(false);
    const [errors, setErrors] = useState({});
    const [showResults, setShowResults] = useState(false);

    // Fetch pricing data when modal opens
    useEffect(() => {
        if (isOpen && pricingData.academicLevels.length === 0 && !isLoadingData && !dataError) {
            fetchPricingData();
        }
    }, [isOpen]);

    const fetchPricingData = async () => {
        setIsLoadingData(true);
        setDataError(null);

        try {
            const response = await axios.get('/api/pricing-data');
            if (response.data.success) {
                setPricingData({
                    academicLevels: response.data.data.academic_levels || [],
                    paperTypes: response.data.data.paper_types || [],
                    subjects: response.data.data.subjects || [],
                    deadlines: response.data.data.deadlines || [],
                    singleSlidePrice: response.data.data.single_slide_price || 5,
                });
            } else {
                setDataError('Unable to load pricing data');
            }
        } catch (error) {
            console.error('Error fetching pricing data:', error);
            setDataError('Unable to load pricing data. Please try again.');
        } finally {
            setIsLoadingData(false);
        }
    };

    const handleCalculate = async (calculatedPricing) => {
        setPricing(calculatedPricing);
        setShowResults(true);
    };

    const handleReset = () => {
        setPricing(null);
        setShowResults(false);
        setErrors({});
        setFormData({
            academic_level_id: initialData.academic_level_id || null,
            paper_type_id: initialData.paper_type_id || null,
            deadline_id: initialData.deadline_id || null,
            pages: 1,
            subject_id: initialData.subject_id || null,
            slides: 0,
            spacing: 'double',
            coupon_code: '',
        });
    };

    const handleClose = () => {
        handleReset();
        onClose();
    };

    const handlePlaceOrder = () => {
        // Store form data for order form pre-population
        if (typeof window !== 'undefined') {
            sessionStorage.setItem('calculator_data', JSON.stringify(formData));
        }
        handleClose();
        // Redirect to order form
        window.location.href = '/place-order?prefill=true';
    };

    return (
        <Modal show={isOpen} onClose={handleClose} maxWidth="2xl">
            <div className="p-6">
                {/* Header */}
                <div className="flex items-center justify-between mb-6">
                    <h2 className="text-2xl font-bold text-gray-900 dark:text-white">
                        Price Calculator
                    </h2>
                    <button
                        onClick={handleClose}
                        className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors"
                        aria-label="Close calculator"
                    >
                        <X className="w-6 h-6" />
                    </button>
                </div>

                {/* Content */}
                {isLoadingData ? (
                    <div className="flex flex-col items-center justify-center py-12">
                        <Loader2 className="w-8 h-8 animate-spin text-blue-600 mb-4" />
                        <p className="text-gray-600 dark:text-gray-400">Loading pricing options...</p>
                    </div>
                ) : dataError ? (
                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
                        <p className="text-sm text-red-600 dark:text-red-400 mb-4">{dataError}</p>
                        <button
                            onClick={fetchPricingData}
                            className="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg text-sm transition-colors"
                        >
                            Retry
                        </button>
                    </div>
                ) : !showResults ? (
                    <PriceCalculatorForm
                        formData={formData}
                        setFormData={setFormData}
                        academicLevels={pricingData.academicLevels}
                        paperTypes={pricingData.paperTypes}
                        subjects={pricingData.subjects}
                        deadlines={pricingData.deadlines}
                        singleSlidePrice={pricingData.singleSlidePrice}
                        isAuthenticated={isAuthenticated}
                        onCalculate={handleCalculate}
                        onClose={handleClose}
                        errors={errors}
                        setErrors={setErrors}
                        isCalculating={isCalculating}
                        setIsCalculating={setIsCalculating}
                    />
                ) : (
                    <PriceCalculatorResults
                        pricing={pricing}
                        isAuthenticated={isAuthenticated}
                        onCalculateAgain={handleReset}
                        onPlaceOrder={handlePlaceOrder}
                        onClose={handleClose}
                    />
                )}
            </div>
        </Modal>
    );
}

