import React from 'react';
import { DollarSign, RefreshCw, ShoppingCart, Info } from 'lucide-react';

export default function PriceCalculatorResults({
    pricing,
    isAuthenticated = false,
    onCalculateAgain,
    onPlaceOrder,
    onClose,
}) {
    // Handle both anonymous and authenticated pricing responses
    const basePrice = pricing?.base_price || pricing?.pricing?.base_price || 0;
    const finalPrice = pricing?.pricing?.final_price || pricing?.estimated_savings?.min_final_price || basePrice;
    const discountAmount = pricing?.pricing?.discount_amount || 0;
    const estimatedSavings = pricing?.estimated_savings;
    const discountInfo = pricing?.discount_info;
    const breakdown = pricing?.pricing?.breakdown;

    return (
        <div className="space-y-6">
            {/* Price Display */}
            <div className="bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 rounded-lg p-6 border border-blue-200 dark:border-blue-800">
                <div className="text-center">
                    <div className="flex items-center justify-center gap-2 text-gray-600 dark:text-gray-400 mb-2">
                        <DollarSign className="w-5 h-5" />
                        <span className="text-sm font-medium">
                            {isAuthenticated ? 'Your Price' : 'Estimated Price'}
                        </span>
                    </div>
                    <div className="text-4xl font-bold text-gray-900 dark:text-white mb-1">
                        ${finalPrice.toFixed(2)}
                    </div>
                    {!isAuthenticated && estimatedSavings && (
                        <div className="text-sm text-gray-600 dark:text-gray-400">
                            After {discountInfo?.minimum_discount || 10}% - {discountInfo?.newcomer_discount || 20}% discount
                        </div>
                    )}
                </div>
            </div>

            {/* Breakdown (for authenticated users) */}
            {isAuthenticated && breakdown && (
                <div className="bg-gray-50 dark:bg-gray-800/50 rounded-lg p-4 space-y-2">
                    <h3 className="font-semibold text-gray-900 dark:text-white mb-3">
                        Pricing Breakdown
                    </h3>
                    <div className="space-y-1.5 text-sm">
                        <div className="flex justify-between text-gray-600 dark:text-gray-400">
                            <span>Cost per page:</span>
                            <span className="font-medium">${breakdown.cost_per_page?.toFixed(2) || '0.00'}</span>
                        </div>
                        {breakdown.slides_cost > 0 && (
                            <div className="flex justify-between text-gray-600 dark:text-gray-400">
                                <span>Slides:</span>
                                <span className="font-medium">${breakdown.slides_cost.toFixed(2)}</span>
                            </div>
                        )}
                        <div className="flex justify-between text-gray-700 dark:text-gray-300 pt-2 border-t border-gray-200 dark:border-gray-700">
                            <span>Subtotal:</span>
                            <span className="font-medium">${(breakdown.pages_cost + (breakdown.slides_cost || 0)).toFixed(2)}</span>
                        </div>
                        {discountAmount > 0 && (
                            <>
                                <div className="flex justify-between text-green-600 dark:text-green-400">
                                    <span>Discount:</span>
                                    <span className="font-medium">-${discountAmount.toFixed(2)}</span>
                                </div>
                                <div className="flex justify-between text-gray-900 dark:text-white pt-2 border-t border-gray-200 dark:border-gray-700 font-semibold">
                                    <span>Total:</span>
                                    <span>${finalPrice.toFixed(2)}</span>
                                </div>
                            </>
                        )}
                    </div>
                </div>
            )}

            {/* Discount Info for Anonymous Users */}
            {!isAuthenticated && estimatedSavings && (
                <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
                    <div className="flex items-start gap-3">
                        <Info className="w-5 h-5 text-green-600 dark:text-green-400 mt-0.5 flex-shrink-0" />
                        <div className="flex-1">
                            <h4 className="font-semibold text-green-900 dark:text-green-100 mb-1">
                                Estimated Savings
                            </h4>
                            <p className="text-sm text-green-800 dark:text-green-200">
                                {discountInfo?.guarantee_message || 
                                    'All customers save with our discount system - minimum 10% guaranteed!'}
                            </p>
                            <div className="mt-2 text-sm text-green-700 dark:text-green-300">
                                <div>Base Price: ${basePrice.toFixed(2)}</div>
                                <div>Estimated Final Price: ${estimatedSavings.min_final_price?.toFixed(2)} - ${estimatedSavings.max_final_price?.toFixed(2)}</div>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Deadline Pricing Info */}
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                <div className="flex items-start gap-3">
                    <Info className="w-5 h-5 text-blue-600 dark:text-blue-400 mt-0.5 flex-shrink-0" />
                    <div className="flex-1">
                        <h4 className="font-semibold text-blue-900 dark:text-blue-100 mb-1">
                            💡 Pricing Tip
                        </h4>
                        <p className="text-sm text-blue-800 dark:text-blue-200">
                            Shorter deadlines (6-12 hours) are more expensive due to urgency, while longer deadlines (5+ days) offer the best value. Choose a flexible deadline to save more!
                        </p>
                    </div>
                </div>
            </div>

            {/* Action Buttons */}
            <div className="flex flex-col sm:flex-row gap-3 pt-4">
                <button
                    onClick={onCalculateAgain}
                    className="flex-1 px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors flex items-center justify-center gap-2 font-medium"
                >
                    <RefreshCw className="w-4 h-4" />
                    Calculate Again
                </button>
                <button
                    onClick={onPlaceOrder}
                    className="flex-1 px-4 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-medium transition-colors flex items-center justify-center gap-2"
                >
                    <ShoppingCart className="w-4 h-4" />
                    Place Order
                </button>
            </div>

            {/* Note */}
            <p className="text-xs text-center text-gray-500 dark:text-gray-400">
                Final price may vary based on specific requirements. Proceed to order form for exact pricing.
            </p>
        </div>
    );
}

