import React from "react";
import SectionHeader from "@/Components/SectionHeader";

function parseTable(desc) {
  const lines = (desc || "").split(/\n/).map((l) => l.trim()).filter(Boolean);
  if (!lines.length) return null;
  // Detect header row with pipes
  if (!lines[0].includes("|")) return null;
  const rows = lines.map((ln) => ln.split("|").map((c) => c.trim()).filter((c) => c.length));
  // Require at least 2 rows, 2+ cols
  if (rows.length < 2 || rows[0].length < 2) return null;
  return rows;
}

function Prose({ text }) {
  const lines = (text || "").split(/\n/);
  const out = [];
  lines.forEach((ln, i) => {
    const t = ln.trim();
    if (!t) {
      out.push(<div key={`sp-${i}`} className="h-2" />);
      return;
    }
    // Headings: #, ##, ###
    const heading = t.match(/^(#{1,3})\s+(.*)$/);
    if (heading) {
      const level = heading[1].length;
      const textContent = heading[2];
      const cls = level === 1
        ? "text-2xl sm:text-3xl font-bold text-gray-900 mt-6"
        : level === 2
        ? "text-xl sm:text-2xl font-semibold text-gray-900 mt-5"
        : "text-lg font-semibold text-gray-900 mt-4";
      out.push(<h3 key={`h-${i}`} className={cls}>{textContent}</h3>);
      return;
    }
    // Unordered list: lines starting with '- '
    if (t.startsWith("- ")) {
      const items = [t.slice(2)];
      let j = i + 1;
      while (j < lines.length && lines[j].trim().startsWith("- ")) {
        items.push(lines[j].trim().slice(2));
        lines[j] = ""; // consume
        j++;
      }
      out.push(
        <ul key={`ul-${i}`} className="list-disc pl-6 space-y-1 text-gray-700">
          {items.map((it, k) => (
            <li key={k}>{it}</li>
          ))}
        </ul>
      );
      return;
    }
    // Ordered list: lines like '1) ' or '1. '
    const olMatch = t.match(/^\d+[\.)]\s+(.*)$/);
    if (olMatch) {
      const items = [olMatch[1]];
      let j = i + 1;
      while (j < lines.length) {
        const next = lines[j].trim();
        const m = next.match(/^\d+[\.)]\s+(.*)$/);
        if (!m) break;
        items.push(m[1]);
        lines[j] = ""; // consume
        j++;
      }
      out.push(
        <ol key={`ol-${i}`} className="list-decimal pl-6 space-y-1 text-gray-700">
          {items.map((it, k) => (
            <li key={k}>{it}</li>
          ))}
        </ol>
      );
      return;
    }
    out.push(<p key={`p-${i}`} className="leading-7 text-gray-700">{t}</p>);
  });
  return <div className="space-y-3">{out}</div>;
}

export default function ProRichSection({ eyebrow, title, subtitle, description, variant = "default", className = "" }) {
  const rows = parseTable(description);
  const wrapperCls = variant === "muted"
    ? "bg-gray-50 border border-gray-100"
    : variant === "brand"
    ? "bg-gradient-to-br from-blue-50 to-indigo-50 border border-blue-100"
    : "bg-white border border-gray-100";

  return (
    <section className={`py-12 ${className}`}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className={`rounded-2xl ${wrapperCls} shadow-sm p-6 sm:p-10`}>        
          <SectionHeader eyebrow={eyebrow} title={title} subtitle={subtitle} align="left" />
          <div className="mt-6">
            {rows ? (
              <div className="overflow-x-auto">
                <table className="min-w-full text-left border-separate border-spacing-0">
                  <thead>
                    <tr>
                      {rows[0].map((h, i) => (
                        <th key={i} className="bg-white sticky top-0 z-10 text-sm font-semibold text-gray-800 px-4 py-3 border-b border-gray-200 first:rounded-tl-xl last:rounded-tr-xl">{h}</th>
                      ))}
                    </tr>
                  </thead>
                  <tbody>
                    {rows.slice(1).map((r, ri) => (
                      <tr key={ri} className="even:bg-gray-50">
                        {r.map((c, ci) => (
                          <td key={ci} className="align-top text-gray-700 px-4 py-3 border-b border-gray-100">{c}</td>
                        ))}
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <Prose text={description} />
            )}
          </div>
        </div>
      </div>
    </section>
  );
}


