import React from 'react';

export default function RatingOverview({ title, description, stats, features }) {
    if (!stats) return null;

    const renderStars = (rating) => {
        return [...Array(5)].map((_, index) => (
            <svg
                key={index}
                className={`w-5 h-5 ${
                    index < rating ? "text-yellow-400" : "text-gray-300"
                }`}
                fill="currentColor"
                viewBox="0 0 20 20"
            >
                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
            </svg>
        ));
    };

    const ratingBreakdown = [
        { stars: 5, count: stats.fiveStars },
        { stars: 4, count: stats.fourStars },
        { stars: 3, count: stats.threeStars },
        { stars: 2, count: stats.twoStars },
        { stars: 1, count: stats.oneStars },
    ];

    return (
        <section className="py-16 bg-white">
            <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                    {/* Left: Rating Details */}
                    <div>
                        {title && (
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
                                {title}
                            </h2>
                        )}
                        {description && (
                            <div 
                                className="text-xl text-gray-600 mb-8 content-area"
                                dangerouslySetInnerHTML={{ __html: description }}
                            />
                        )}

                        <div className="flex items-center mb-6">
                            <div className="flex mr-4">
                                {renderStars(5)}
                            </div>
                            <span className="text-2xl font-bold text-gray-900 mr-2">
                                {stats.averageRating}
                            </span>
                            <span className="text-gray-600">
                                out of 5 stars
                            </span>
                        </div>

                        {/* Rating Bars */}
                        <div className="space-y-3">
                            {ratingBreakdown.map(({ stars, count }) => {
                                const percentage = (count / stats.totalReviews) * 100;
                                return (
                                    <div key={stars} className="flex items-center">
                                        <div className="flex items-center w-16">
                                            <span className="text-sm font-medium text-gray-900 mr-2">
                                                {stars}
                                            </span>
                                            <svg
                                                className="w-4 h-4 text-yellow-400"
                                                fill="currentColor"
                                                viewBox="0 0 20 20"
                                            >
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                            </svg>
                                        </div>
                                        <div className="flex-1 mx-4">
                                            <div className="bg-gray-200 rounded-full h-2">
                                                <div
                                                    className="bg-yellow-400 rounded-full h-2"
                                                    style={{ width: `${percentage}%` }}
                                                />
                                            </div>
                                        </div>
                                        <div className="w-16 text-right">
                                            <span className="text-sm text-gray-600">
                                                {count.toLocaleString()}
                                            </span>
                                        </div>
                                    </div>
                                );
                            })}
                        </div>
                    </div>

                    {/* Right: Key Features */}
                    {features && features.length > 0 && (
                        <div className="bg-blue-50 rounded-lg p-8">
                            <h3 className="text-2xl font-bold text-gray-900 mb-6">
                                Why Students Love Our Service
                            </h3>
                            <div className="space-y-4">
                                {features.map((feature, index) => (
                                    <div key={index} className="flex items-start">
                                        <svg
                                            className="w-6 h-6 text-green-500 mr-3 mt-1"
                                            fill="currentColor"
                                            viewBox="0 0 20 20"
                                        >
                                            <path
                                                fillRule="evenodd"
                                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                                clipRule="evenodd"
                                            />
                                        </svg>
                                        <div>
                                            <div className="font-semibold text-gray-900">
                                                {feature.title}
                                            </div>
                                            <div 
                                                className="text-gray-600 content-area"
                                                dangerouslySetInnerHTML={{ __html: feature.description }}
                                            />
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </section>
    );
}




