import React, { useState } from 'react';
import { useForm } from '@inertiajs/react';
import { 
    Clock, 
    AlertTriangle, 
    CheckCircle, 
    XCircle, 
    Play, 
    Upload,
    Eye,
    Edit,
    User,
    Calendar,
    FileText,
    AlertCircle
} from 'lucide-react';

const RevisionManagement = ({ 
    revisions = [], 
    order,
    onRevisionUpdate,
    showActions = true 
}) => {
    const [expandedRevision, setExpandedRevision] = useState(null);
    const [editingRevision, setEditingRevision] = useState(null);

    const { patch, processing } = useForm();

    const getStatusIcon = (status) => {
        switch (status) {
            case 'requested':
                return <AlertCircle className="w-4 h-4 text-orange-500" />;
            case 'acknowledged':
                return <Clock className="w-4 h-4 text-blue-500" />;
            case 'in_progress':
                return <Play className="w-4 h-4 text-yellow-500" />;
            case 'submitted':
                return <Upload className="w-4 h-4 text-indigo-500" />;
            case 'reviewed':
                return <CheckCircle className="w-4 h-4 text-green-500" />;
            default:
                return <AlertCircle className="w-4 h-4 text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'requested':
                return 'bg-orange-100 text-orange-800 dark:bg-orange-900/20 dark:text-orange-200';
            case 'acknowledged':
                return 'bg-blue-100 text-blue-800 dark:bg-blue-900/20 dark:text-blue-200';
            case 'in_progress':
                return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-200';
            case 'submitted':
                return 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900/20 dark:text-indigo-200';
            case 'reviewed':
                return 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-200';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-200';
        }
    };

    const getPriorityColor = (priority) => {
        switch (priority) {
            case 'high':
                return 'bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-200';
            case 'medium':
                return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-200';
            case 'low':
                return 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-200';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-200';
        }
    };

    const getTypeIcon = (type) => {
        return <FileText className="w-4 h-4" />;
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'Not set';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const isOverdue = (revision) => {
        if (!revision.revision_deadline || revision.revision_status === 'reviewed') {
            return false;
        }
        return new Date() > new Date(revision.revision_deadline);
    };

    const getOverdueHours = (revision) => {
        if (!isOverdue(revision)) return 0;
        const deadline = new Date(revision.revision_deadline);
        const now = new Date();
        return Math.floor((now - deadline) / (1000 * 60 * 60));
    };

    const handleStatusUpdate = (revisionId, newStatus) => {
        patch(`/admin/revisions/${revisionId}/status`, {
            data: { status: newStatus },
            onSuccess: () => {
                if (onRevisionUpdate) onRevisionUpdate();
            }
        });
    };

    const toggleExpanded = (revisionId) => {
        setExpandedRevision(expandedRevision === revisionId ? null : revisionId);
    };

    const startEditing = (revision) => {
        setEditingRevision(revision);
    };

    const cancelEditing = () => {
        setEditingRevision(null);
    };

    if (revisions.length === 0) {
        return (
            <div className="text-center py-8">
                <FileText className="mx-auto h-12 w-12 text-gray-400" />
                <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">No revisions</h3>
                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                    This order has no revision requests yet.
                </p>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            <div className="flex items-center justify-between">
                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                    Revision Requests ({revisions.length})
                </h3>
                {order && (
                    <div className="text-sm text-gray-500 dark:text-gray-400">
                        Order #{order.order_number}
                    </div>
                )}
            </div>

            {revisions.map((revision) => (
                <div
                    key={revision.id}
                    className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm"
                >
                    {/* Revision Header */}
                    <div className="p-4">
                        <div className="flex items-start justify-between">
                            <div className="flex items-start space-x-3">
                                <div className="flex-shrink-0">
                                    {getStatusIcon(revision.revision_status)}
                                </div>
                                <div className="flex-1 min-w-0">
                                    <div className="flex items-center space-x-2 mb-2">
                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(revision.revision_status)}`}>
                                            {revision.revision_status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                        </span>
                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getPriorityColor(revision.revision_priority)}`}>
                                            {revision.revision_priority.charAt(0).toUpperCase() + revision.revision_priority.slice(1)} Priority
                                        </span>
                                        {isOverdue(revision) && (
                                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-200">
                                                <AlertTriangle className="w-3 h-3 mr-1" />
                                                Overdue ({getOverdueHours(revision)}h)
                                            </span>
                                        )}
                                    </div>
                                    
                                    <div className="flex items-center space-x-4 text-sm text-gray-500 dark:text-gray-400 mb-2">
                                        <div className="flex items-center">
                                            {getTypeIcon(revision.revision_type)}
                                            <span className="ml-1">{revision.revision_type.charAt(0).toUpperCase() + revision.revision_type.slice(1)} Revision</span>
                                        </div>
                                        {revision.revision_section && (
                                            <div className="flex items-center">
                                                <span className="ml-1">Section: {revision.revision_section}</span>
                                            </div>
                                        )}
                                        <div className="flex items-center">
                                            <Calendar className="w-3 h-3 mr-1" />
                                            <span className="ml-1">#{revision.revision_number}</span>
                                        </div>
                                    </div>

                                    <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-1">
                                        {revision.revision_reason}
                                    </h4>

                                    {revision.specific_instructions && (
                                        <p className="text-sm text-gray-600 dark:text-gray-300">
                                            {revision.specific_instructions}
                                        </p>
                                    )}
                                </div>
                            </div>

                            <div className="flex items-center space-x-2">
                                <button
                                    onClick={() => toggleExpanded(revision.id)}
                                    className="p-1 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                >
                                    <Eye className="w-4 h-4" />
                                </button>
                                {showActions && (
                                    <button
                                        onClick={() => startEditing(revision)}
                                        className="p-1 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                    >
                                        <Edit className="w-4 h-4" />
                                    </button>
                                )}
                            </div>
                        </div>

                        {/* Assignment and Deadline Info */}
                        <div className="mt-3 flex items-center justify-between text-sm">
                            <div className="flex items-center space-x-4">
                                {revision.assigned_to && (
                                    <div className="flex items-center text-gray-500 dark:text-gray-400">
                                        <User className="w-3 h-3 mr-1" />
                                        <span>Assigned to: {revision.assigned_to_user?.name || 'Unknown'}</span>
                                    </div>
                                )}
                                <div className="flex items-center text-gray-500 dark:text-gray-400">
                                    <Clock className="w-3 h-3 mr-1" />
                                    <span>Deadline: {formatDate(revision.revision_deadline)}</span>
                                </div>
                            </div>
                            
                            <div className="text-xs text-gray-400 dark:text-gray-500">
                                Requested {formatDate(revision.revision_requested_at)}
                            </div>
                        </div>
                    </div>

                    {/* Expanded Details */}
                    {expandedRevision === revision.id && (
                        <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-700">
                            <div className="pt-4 space-y-3">
                                {/* Client Feedback */}
                                {revision.revision_client_feedback && (
                                    <div>
                                        <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Client Feedback
                                        </h5>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 bg-gray-50 dark:bg-gray-700 p-3 rounded">
                                            {revision.revision_client_feedback}
                                        </p>
                                    </div>
                                )}

                                {/* Admin Notes */}
                                {revision.revision_admin_notes && (
                                    <div>
                                        <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Admin Notes
                                        </h5>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 bg-blue-50 dark:bg-blue-900/20 p-3 rounded">
                                            {revision.revision_admin_notes}
                                        </p>
                                    </div>
                                )}

                                {/* Status Actions */}
                                {showActions && revision.revision_status !== 'reviewed' && (
                                    <div>
                                        <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Update Status
                                        </h5>
                                        <div className="flex flex-wrap gap-2">
                                            {revision.revision_status === 'requested' && (
                                                <>
                                                    <button
                                                        onClick={() => handleStatusUpdate(revision.id, 'acknowledged')}
                                                        disabled={processing}
                                                        className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
                                                    >
                                                        <Clock className="w-3 h-3 mr-1" />
                                                        Acknowledge
                                                    </button>
                                                    <button
                                                        onClick={() => handleStatusUpdate(revision.id, 'in_progress')}
                                                        disabled={processing}
                                                        className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-white bg-yellow-600 hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 disabled:opacity-50"
                                                    >
                                                        <Play className="w-3 h-3 mr-1" />
                                                        Start Work
                                                    </button>
                                                </>
                                            )}
                                            
                                            {revision.revision_status === 'acknowledged' && (
                                                <button
                                                    onClick={() => handleStatusUpdate(revision.id, 'in_progress')}
                                                    disabled={processing}
                                                    className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-white bg-yellow-600 hover:bg-yellow-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 disabled:opacity-50"
                                                >
                                                    <Play className="w-3 h-3 mr-1" />
                                                    Start Work
                                                </button>
                                            )}
                                            
                                            {revision.revision_status === 'in_progress' && (
                                                <button
                                                    onClick={() => handleStatusUpdate(revision.id, 'submitted')}
                                                    disabled={processing}
                                                    className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50"
                                                >
                                                    <Upload className="w-3 h-3 mr-1" />
                                                    Mark Submitted
                                                </button>
                                            )}
                                            
                                            {revision.revision_status === 'submitted' && (
                                                <button
                                                    onClick={() => handleStatusUpdate(revision.id, 'reviewed')}
                                                    disabled={processing}
                                                    className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50"
                                                >
                                                    <CheckCircle className="w-3 h-3 mr-1" />
                                                    Mark Reviewed
                                                </button>
                                            )}
                                        </div>
                                    </div>
                                )}

                                {/* Metadata */}
                                {revision.revision_metadata && Object.keys(revision.revision_metadata).length > 0 && (
                                    <div>
                                        <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Additional Information
                                        </h5>
                                        <div className="bg-gray-50 dark:bg-gray-700 p-3 rounded text-sm">
                                            <pre className="text-gray-600 dark:text-gray-400 whitespace-pre-wrap">
                                                {JSON.stringify(revision.revision_metadata, null, 2)}
                                            </pre>
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>
                    )}
                </div>
            ))}
        </div>
    );
};

export default RevisionManagement; 