import React from "react";
import { motion } from "framer-motion";
import { FileCheck, Shield, Search, Lock, CheckCircle, AlertCircle } from "lucide-react";
import PropTypes from "prop-types";

/**
 * Originality Section Component
 * Displays originality guarantees and plagiarism prevention measures
 */
export default function OriginalitySection({
  title = "100% Original Content Guarantee",
  description = "",
  guarantees = [],
  process = [],
}) {
  const iconMap = {
    FileCheck,
    Shield,
    Search,
    Lock,
    CheckCircle,
    AlertCircle,
  };

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
  };

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.15,
      },
    },
  };

  return (
    <section className="py-20 bg-gray-50 dark:bg-gray-800 w-full">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={fadeIn}
          className="text-center mb-16"
        >
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
            {title}
          </h2>
          <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
          {description && (
            <div 
              className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
              dangerouslySetInnerHTML={{ __html: description }}
            />
          )}
        </motion.div>

        {/* Guarantees Grid */}
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={staggerContainer}
          className="grid md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12"
        >
          {guarantees.map((guarantee, index) => {
            const IconComponent = iconMap[guarantee.icon] || FileCheck;
            return (
              <motion.div
                key={index}
                variants={fadeIn}
                className="bg-white dark:bg-gray-900 rounded-lg p-6 text-center shadow-md border border-gray-200 dark:border-gray-700
                           transition-all duration-300 hover:shadow-xl hover:-translate-y-1"
              >
                <div className="flex items-center justify-center h-14 w-14 rounded-full bg-green-100 dark:bg-green-900 text-green-600 dark:text-green-400 mx-auto mb-4">
                  <IconComponent className="h-7 w-7" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  {guarantee.title}
                </h3>
                <p className="text-gray-600 dark:text-gray-300 text-sm">
                  {guarantee.description}
                </p>
              </motion.div>
            );
          })}
        </motion.div>

        {/* Originality Process */}
        {process.length > 0 && (
          <motion.div
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true }}
            variants={staggerContainer}
            className="max-w-4xl mx-auto"
          >
            <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-8 text-center">
              Our Originality Check Process
            </h3>
            <div className="space-y-6">
              {process.map((step, index) => {
                const IconComponent = iconMap[step.icon] || CheckCircle;
                return (
                  <motion.div
                    key={index}
                    variants={fadeIn}
                    className="flex bg-white dark:bg-gray-900 rounded-lg p-6 shadow-md border-l-4 border-blue-600"
                  >
                    <div className="flex-shrink-0">
                      <div className="flex items-center justify-center h-10 w-10 rounded-full bg-blue-100 dark:bg-blue-900 text-blue-600 dark:text-blue-400">
                        <IconComponent className="h-5 w-5" />
                      </div>
                    </div>
                    <div className="ml-4">
                      <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                        Step {index + 1}: {step.title}
                      </h4>
                      <div 
                        className="text-gray-600 dark:text-gray-300 content-area"
                        dangerouslySetInnerHTML={{ __html: step.description }}
                      />
                    </div>
                  </motion.div>
                );
              })}
            </div>
          </motion.div>
        )}
      </div>
    </section>
  );
}

OriginalitySection.propTypes = {
  title: PropTypes.string,
  description: PropTypes.string,
  guarantees: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
      icon: PropTypes.string,
    })
  ),
  process: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
      icon: PropTypes.string,
    })
  ),
};









