import React from "react";
import { motion } from "framer-motion";
import { CheckCircle, Star, Award, TrendingUp, Shield, Clock } from "lucide-react";
import PropTypes from "prop-types";

/**
 * Reasons Section Component
 * Displays reasons to choose the service in a card grid
 */
export default function ReasonsSection({
  title = "Why Choose Us",
  description = "",
  reasons = [],
  highlights = [],
}) {
  const iconMap = {
    CheckCircle,
    Star,
    Award,
    TrendingUp,
    Shield,
    Clock,
  };

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
  };

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.15,
      },
    },
  };

  return (
    <section className="py-20 bg-white dark:bg-gray-900 w-full">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={fadeIn}
          className="text-center mb-16"
        >
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
            {title}
          </h2>
          <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
          {description && (
            <div 
              className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
              dangerouslySetInnerHTML={{ __html: description }}
            />
          )}
        </motion.div>

        {/* Highlights Banner */}
        {highlights.length > 0 && (
          <motion.div
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true }}
            variants={fadeIn}
            className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-lg p-8 mb-12 text-white"
          >
            <div className="grid md:grid-cols-3 gap-6">
              {highlights.map((highlight, index) => (
                <div key={index} className="text-center">
                  <div className="text-3xl font-bold mb-2">{highlight.value}</div>
                  <div className="text-blue-100">{highlight.label}</div>
                </div>
              ))}
            </div>
          </motion.div>
        )}

        {/* Reasons Grid */}
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={staggerContainer}
          className="grid md:grid-cols-2 gap-8"
        >
          {reasons.map((reason, index) => {
            const IconComponent = iconMap[reason.icon] || CheckCircle;
            return (
              <motion.div
                key={index}
                variants={fadeIn}
                className="flex bg-gradient-to-br from-gray-50 to-blue-50 dark:bg-gray-800 rounded-lg p-6 border border-gray-200 dark:border-gray-700
                           transition-all duration-300 hover:shadow-lg hover:-translate-y-1"
              >
                <div className="flex-shrink-0">
                  <div className="flex items-center justify-center h-12 w-12 rounded-md bg-blue-600 text-white">
                    <IconComponent className="h-6 w-6" />
                  </div>
                </div>
                <div className="ml-4">
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                    {reason.title}
                  </h3>
                  <div 
                    className="text-gray-600 dark:text-gray-300 leading-relaxed content-area"
                    dangerouslySetInnerHTML={{ __html: reason.description }}
                  />
                </div>
              </motion.div>
            );
          })}
        </motion.div>
      </div>
    </section>
  );
}

ReasonsSection.propTypes = {
  title: PropTypes.string,
  description: PropTypes.string,
  reasons: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
      icon: PropTypes.string,
    })
  ),
  highlights: PropTypes.arrayOf(
    PropTypes.shape({
      value: PropTypes.string.isRequired,
      label: PropTypes.string.isRequired,
    })
  ),
};









