import { motion } from "framer-motion";
import {
    CheckCircle,
    Award,
    Clock,
    Shield,
    Zap,
    BookOpen,
    Users,
    DollarSign,
    Headset,
    MessageCircle,
    Phone,
    Percent,
    Star,
    ShieldCheck,
    Mail,
    Target,
    Heart,
    FileText,
    Search,
} from "lucide-react";
import PropTypes from "prop-types";
import ServiceImage from "../ServiceImage";

const defaultBenefits = [
    {
        title: "Expert Writers",
        description:
            "PhD-level academics with expertise in your specific field",
        icon: "Award",
    },
    {
        title: "Plagiarism-Free Guarantee",
        description:
            "Original content written from scratch with thorough plagiarism checks",
        icon: "Shield",
    },
    {
        title: "On-Time Delivery",
        description:
            "Meet your deadlines with our punctual delivery system",
        icon: "Clock",
    },
    {
        title: "Unlimited Revisions",
        description:
            "Free revisions until you're completely satisfied with your paper",
        icon: "CheckCircle",
    },
    {
        title: "24/7 Support",
        description: "Our support team is always available to assist you",
        icon: "Zap",
    },
];

const iconMap = {
    CheckCircle,
    Award,
    Clock,
    Shield,
    Zap,
    BookOpen,
    Users,
    DollarSign,
    Headset,
    MessageCircle,
    Phone,
    Percent,
    Star,
    ShieldCheck,
    Mail,
    Target,
    Heart,
    FileText,
    Search,
};

const channelIconMap = {
    chat: MessageCircle,
    phone: Phone,
    email: Mail,
    support: Headset,
};

const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
};

const renderHtmlContent = (value, className = "") => {
    if (!value) return null;
    return (
        <div
            className={`prose prose-sm mx-auto text-gray-600 dark:text-slate-300 ${className}`.trim()}
            dangerouslySetInnerHTML={{ __html: value }}
        />
    );
};

export default function ServiceBenefits({
    eyebrow = null,
    title = "Benefits of Our Service",
    description = "Discover why our academic writing service stands out from the rest",
    benefits = [],
    image = null,
    imageAlt = "Service benefits",
    ctaText = "Get Started",
    ctaLink = "/contact",
    serviceName = null,
    quickStart = null,
    offer = null,
    support = null,
    trustSignals = [],
    guarantees = null,
}) {
    const displayBenefits = benefits.length > 0 ? benefits : defaultBenefits;
    const activeBenefits = displayBenefits.filter(
        (benefit) => benefit.is_active !== false
    );

    const guaranteeItems = Array.isArray(guarantees?.items)
        ? guarantees.items.filter((item) => item?.is_active !== false)
        : Array.isArray(guarantees)
        ? guarantees.filter((item) => item?.is_active !== false)
        : [];

    const quickStartSteps = Array.isArray(quickStart?.steps)
        ? quickStart.steps.filter((step) => step?.is_active !== false)
        : [];

    const supportChannels = Array.isArray(support?.channels)
        ? support.channels.filter((channel) => channel?.is_active !== false)
        : [];

    const trustSignalItems = Array.isArray(trustSignals)
        ? trustSignals.filter((signal) => signal?.is_active !== false)
        : [];

    const hasGuarantees =
        guarantees &&
        (guarantees?.title ||
            guarantees?.description ||
            guaranteeItems.length > 0);

    const hasQuickStart =
        quickStart &&
        (quickStart.title ||
            quickStart.description ||
            quickStart.description_html ||
            quickStartSteps.length > 0);

    const hasOffer =
        offer &&
        (offer.title ||
            offer.description ||
            offer.description_html ||
            offer.badge ||
            offer.code ||
            (offer.ctaText && offer.ctaLink));

    const hasSupport =
        support &&
        (support.title ||
            support.description ||
            support.description_html ||
            supportChannels.length > 0);

    const hasTrustSignals = trustSignalItems.length > 0;

    return (
        <section
            id="service-benefits"
            className="py-20 bg-gray-50 dark:bg-gray-900 w-full"
        >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    {eyebrow && (
                        <span className="inline-flex items-center rounded-full bg-blue-100 px-3 py-1 text-xs font-semibold uppercase tracking-wide text-blue-700 dark:bg-blue-900/40 dark:text-blue-200">
                            {eyebrow}
                        </span>
                    )}
                    <h2 className="mt-4 text-3xl font-bold text-gray-900 dark:text-white">
                        {title}
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
                    {renderHtmlContent(description, "mt-6 max-w-3xl text-lg content-area")}
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className="grid gap-6 lg:grid-cols-3"
                >
                    <div className="lg:col-span-2">
                        <div className="grid sm:grid-cols-2 gap-6">
                            {activeBenefits.map((benefit, index) => {
                                const IconComponent = iconMap[benefit.icon] || CheckCircle;
                                const descriptionHtml =
                                    benefit.description_html || benefit.description;

                                return (
                                    <div
                                        key={`benefit-${index}`}
                                        className="bg-white dark:bg-slate-900/60 border border-slate-100 dark:border-slate-700 rounded-xl shadow-sm hover:shadow-lg transition-shadow duration-300 p-6 flex flex-col h-full"
                                    >
                                        <div className="flex items-center gap-3 mb-4">
                                            <span className="inline-flex h-11 w-11 items-center justify-center rounded-full bg-blue-600 text-white">
                                                <IconComponent className="h-6 w-6" />
                                            </span>
                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                                {benefit.title}
                                            </h3>
                                        </div>
                                        {renderHtmlContent(
                                            descriptionHtml,
                                            "text-sm leading-relaxed text-left flex-1 content-area"
                                        )}
                                    </div>
                                );
                            })}
                        </div>
                    </div>

                    <motion.div
                        initial={{ opacity: 0, scale: 0.95 }}
                        whileInView={{ opacity: 1, scale: 1 }}
                        transition={{ duration: 0.5 }}
                        viewport={{ once: true }}
                        className="bg-gradient-to-br from-blue-600 to-blue-700 dark:from-blue-700 dark:to-blue-800 text-white rounded-2xl shadow-xl overflow-hidden flex flex-col"
                    >
                        {(image || serviceName) && (
                            <div className="h-64 overflow-hidden">
                                {serviceName ? (
                                    <ServiceImage
                                        service={serviceName}
                                        alt={imageAlt}
                                        className="w-full h-full object-cover"
                                        sizes="(max-width: 1024px) 100vw, 400px"
                                    />
                                ) : (
                                    <img
                                        src={image}
                                        alt={imageAlt}
                                        className="w-full h-full object-cover"
                                        loading="lazy"
                                    />
                                )}
                            </div>
                        )}
                        <div className="p-6 flex-1 flex flex-col space-y-4">
                            <h3 className="text-xl font-semibold">
                                {offer?.sidebarTitle || "Ready to experience the difference?"}
                            </h3>
                            <p className="text-sm text-blue-100 flex-1">
                                {offer?.sidebarDescription ||
                                    "Join thousands of students who rely on Academic Scribe for original, professionally crafted essays delivered on time."}
                            </p>
                            <a
                                href={offer?.sidebarCtaLink || ctaLink}
                                className="inline-flex items-center justify-center w-full bg-white text-blue-700 font-semibold py-3 rounded-lg hover:bg-blue-50 transition-colors duration-200"
                            >
                                {offer?.sidebarCtaText || ctaText}
                            </a>
                            <p className="text-xs text-blue-100 text-center">
                                {offer?.sidebarFootnote ||
                                    "Confidential • Money-back guarantee • Free revisions"}
                            </p>
                        </div>
                    </motion.div>
                </motion.div>

                {(hasGuarantees || hasQuickStart) && (
                    <div className="mt-12 grid gap-6 lg:grid-cols-2">
                        {hasGuarantees && (
                            <motion.div
                                initial={{ opacity: 0, y: 20 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.5, delay: 0.1 }}
                                className="bg-white dark:bg-slate-900/60 border border-slate-100 dark:border-slate-700 rounded-2xl shadow-lg p-8"
                            >
                                <div className="flex items-center gap-3 mb-6">
                                    <ShieldCheck className="h-8 w-8 text-blue-600 dark:text-blue-400" />
                                    <div>
                                        <h3 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            {guarantees?.title || "Our Guarantees"}
                                        </h3>
                                        {guarantees?.description && (
                                            <p className="text-sm text-gray-500 dark:text-slate-300 mt-1">
                                                {guarantees.description}
                                            </p>
                                        )}
                                    </div>
                                </div>
                                <div className="space-y-4">
                                    {guaranteeItems.map((item, index) => (
                                        <div key={`guarantee-${index}`} className="flex gap-4">
                                            <span className="mt-1 inline-flex h-6 w-6 shrink-0 items-center justify-center rounded-full bg-blue-100 text-blue-600 dark:bg-blue-900/60 dark:text-blue-300">
                                                <CheckCircle className="h-4 w-4" />
                                            </span>
                                            <div>
                                                <h4 className="text-base font-semibold text-gray-900 dark:text-white">
                                                    {item.title}
                                                </h4>
                                                {renderHtmlContent(
                                                    item.description_html || item.description,
                                                    "text-sm mt-1 text-left"
                                                )}
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </motion.div>
                        )}

                        {hasQuickStart && (
                            <motion.div
                                initial={{ opacity: 0, y: 20 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.5, delay: 0.2 }}
                                className="bg-gradient-to-br from-gray-900 via-gray-900 to-blue-900 text-white rounded-2xl shadow-lg p-8"
                            >
                                <div className="flex items-center gap-3 mb-6">
                                    <Zap className="h-8 w-8 text-blue-300" />
                                    <div>
                                        <h3 className="text-xl font-semibold">
                                            {quickStart?.title || "Get Started Now"}
                                        </h3>
                                        {renderHtmlContent(
                                            quickStart?.description_html || quickStart?.description,
                                            "text-sm text-blue-100 mt-2 text-left"
                                        )}
                                    </div>
                                </div>
                                <div className="space-y-5">
                                    {quickStartSteps.map((step, index) => (
                                        <div
                                            key={`quickstep-${index}`}
                                            className="flex gap-4 rounded-xl bg-white/5 p-4 ring-1 ring-white/10"
                                        >
                                            <span className="flex h-10 w-10 items-center justify-center rounded-full bg-blue-500 text-white font-semibold">
                                                {index + 1}
                                            </span>
                                            <div>
                                                <h4 className="text-lg font-semibold">
                                                    {step.title || `Step ${index + 1}`}
                                                </h4>
                                                {renderHtmlContent(
                                                    step.description_html || step.description,
                                                    "text-sm mt-1 text-blue-100 text-left"
                                                )}
                                            </div>
                                        </div>
                                    ))}
                                </div>
                                {quickStart?.footnote && (
                                    <p className="mt-6 text-xs uppercase tracking-wide text-blue-200">
                                        {quickStart.footnote}
                                    </p>
                                )}
                            </motion.div>
                        )}
                    </div>
                )}

                {(hasOffer || hasSupport) && (
                    <div className="mt-12 grid gap-6 lg:grid-cols-3">
                        {hasOffer && (
                            <motion.div
                                initial={{ opacity: 0, y: 20 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.5, delay: 0.1 }}
                                className="lg:col-span-2 bg-white dark:bg-slate-900/60 border border-slate-100 dark:border-slate-700 rounded-2xl shadow-lg p-8"
                            >
                                <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-6">
                                    <div className="flex-1">
                                        <div className="flex flex-wrap items-center gap-2">
                                            {offer?.badge && (
                                                <span className="inline-flex items-center rounded-full bg-blue-100 px-3 py-1 text-sm font-semibold text-blue-700 dark:bg-blue-900/40 dark:text-blue-200">
                                                    {offer.badge}
                                                </span>
                                            )}
                                            {offer?.code && (
                                                <span className="inline-flex items-center rounded-full bg-blue-50 px-3 py-1 text-sm font-semibold text-blue-600 dark:bg-blue-900/60 dark:text-blue-200">
                                                    Code: {offer.code}
                                                </span>
                                            )}
                                        </div>
                                        <h3 className="mt-4 text-2xl font-bold text-gray-900 dark:text-white">
                                            {offer?.title || "Save 15% on your first order"}
                                        </h3>
                                        {renderHtmlContent(
                                            offer?.description_html || offer?.description,
                                            "mt-4 text-left text-base"
                                        )}
                                        {offer?.note && (
                                            <p className="mt-4 text-sm text-gray-500 dark:text-slate-300">
                                                {offer.note}
                                            </p>
                                        )}
                                    </div>
                                    {offer?.ctaText && offer?.ctaLink && (
                                        <a
                                            href={offer.ctaLink}
                                            className="inline-flex items-center justify-center rounded-xl bg-blue-600 px-6 py-3 text-base font-semibold text-white shadow hover:bg-blue-700 transition-colors"
                                        >
                                            {offer.ctaText}
                                        </a>
                                    )}
                                </div>
                            </motion.div>
                        )}

                        {hasSupport && (
                            <motion.div
                                initial={{ opacity: 0, y: 20 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.5, delay: 0.2 }}
                                className="bg-gray-900 text-white rounded-2xl shadow-lg p-8 space-y-4"
                            >
                                <h3 className="text-xl font-semibold">
                                    {support?.title || "Need a hand?"}
                                </h3>
                                {renderHtmlContent(
                                    support?.description_html || support?.description,
                                    "text-sm text-gray-200 text-left"
                                )}
                                <div className="space-y-3">
                                    {supportChannels.map((channel, index) => {
                                        const IconComponent =
                                            channelIconMap[channel.type] || Headset;
                                        return (
                                            <a
                                                key={`support-${index}`}
                                                href={channel.href || "#"}
                                                className="flex items-center gap-3 rounded-xl bg-white/10 px-4 py-3 backdrop-blur transition hover:bg-white/20"
                                            >
                                                <IconComponent className="h-5 w-5 text-blue-200" />
                                                <div>
                                                    <p className="text-sm font-semibold text-white">
                                                        {channel.label}
                                                    </p>
                                                    <p className="text-xs text-gray-300">
                                                        {channel.value}
                                                    </p>
                                                </div>
                                            </a>
                                        );
                                    })}
                                </div>
                            </motion.div>
                        )}
                    </div>
                )}

                {hasTrustSignals && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                        transition={{ duration: 0.5, delay: 0.15 }}
                        className="mt-12 rounded-3xl bg-blue-600/10 border border-blue-600/20 dark:bg-blue-900/20 dark:border-blue-800/40 p-10"
                    >
                        <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                            {trustSignalItems.map((signal, index) => (
                                <div
                                    key={`trust-${index}`}
                                    className="rounded-2xl bg-white/70 dark:bg-slate-900/70 px-6 py-5 shadow-inner"
                                >
                                    <p className="text-3xl font-bold text-blue-700 dark:text-blue-300">
                                        {signal.value}
                                    </p>
                                    <p className="mt-1 text-sm font-semibold text-gray-900 dark:text-white">
                                        {signal.label}
                                    </p>
                                    {signal.description && (
                                        <p className="mt-2 text-xs text-gray-600 dark:text-slate-300">
                                            {signal.description}
                                        </p>
                                    )}
                                </div>
                            ))}
                        </div>
                    </motion.div>
                )}
            </div>
        </section>
    );
}

ServiceBenefits.propTypes = {
    eyebrow: PropTypes.string,
    title: PropTypes.string,
    description: PropTypes.string,
    benefits: PropTypes.arrayOf(
        PropTypes.shape({
            title: PropTypes.string.isRequired,
            description: PropTypes.string,
            description_html: PropTypes.string,
            icon: PropTypes.string,
            is_active: PropTypes.bool,
        })
    ),
    image: PropTypes.string,
    imageAlt: PropTypes.string,
    ctaText: PropTypes.string,
    ctaLink: PropTypes.string,
    serviceName: PropTypes.string,
    quickStart: PropTypes.shape({
        title: PropTypes.string,
        description: PropTypes.string,
        description_html: PropTypes.string,
        footnote: PropTypes.string,
        steps: PropTypes.arrayOf(
            PropTypes.shape({
                title: PropTypes.string,
                description: PropTypes.string,
                description_html: PropTypes.string,
                is_active: PropTypes.bool,
            })
        ),
    }),
    offer: PropTypes.shape({
        badge: PropTypes.string,
        title: PropTypes.string,
        description: PropTypes.string,
        description_html: PropTypes.string,
        code: PropTypes.string,
        note: PropTypes.string,
        ctaText: PropTypes.string,
        ctaLink: PropTypes.string,
        sidebarTitle: PropTypes.string,
        sidebarDescription: PropTypes.string,
        sidebarFootnote: PropTypes.string,
        sidebarCtaText: PropTypes.string,
        sidebarCtaLink: PropTypes.string,
    }),
    support: PropTypes.shape({
        title: PropTypes.string,
        description: PropTypes.string,
        description_html: PropTypes.string,
        channels: PropTypes.arrayOf(
            PropTypes.shape({
                label: PropTypes.string,
                value: PropTypes.string,
                href: PropTypes.string,
                type: PropTypes.string,
                is_active: PropTypes.bool,
            })
        ),
    }),
    trustSignals: PropTypes.arrayOf(
        PropTypes.shape({
            value: PropTypes.string,
            label: PropTypes.string,
            description: PropTypes.string,
            is_active: PropTypes.bool,
        })
    ),
    guarantees: PropTypes.oneOfType([
        PropTypes.shape({
            title: PropTypes.string,
            description: PropTypes.string,
            items: PropTypes.arrayOf(
                PropTypes.shape({
                    title: PropTypes.string,
                    description: PropTypes.string,
                    description_html: PropTypes.string,
                    is_active: PropTypes.bool,
                })
            ),
        }),
        PropTypes.arrayOf(
            PropTypes.shape({
                title: PropTypes.string,
                description: PropTypes.string,
                description_html: PropTypes.string,
                is_active: PropTypes.bool,
            })
        ),
        PropTypes.oneOf([null, undefined]),
    ]),
};
