import { motion } from "framer-motion";
import { ArrowRight, Clock, Award, CheckCircle } from "lucide-react";
import PropTypes from "prop-types";

export default function ServiceCTA({
    title = "Ready to Get Started?",
    description = "Our team of expert writers is ready to help you achieve academic excellence",
    benefits = [],
    primaryButtonText = "Place an Order",
    primaryButtonLink = "/place-order",
    secondaryButtonText = "Contact Us",
    secondaryButtonLink = "#contact",
    bgColor = "blue",
}) {
    // Default benefits if none provided
    const defaultBenefits = [
        {
            text: "Professional writers with advanced degrees",
            icon: "Award",
        },
        {
            text: "On-time delivery guaranteed",
            icon: "Clock",
        },
        {
            text: "100% original, plagiarism-free papers",
            icon: "CheckCircle",
        },
    ];

    // Use provided benefits or default ones
    const displayBenefits = benefits.length > 0 ? benefits : defaultBenefits;

    // Map icon names to actual components
    const iconMap = {
        Award: Award,
        Clock: Clock,
        CheckCircle: CheckCircle,
    };

    // Background color classes based on prop
    const bgClasses = {
        blue: "bg-gradient-to-r from-blue-600 to-indigo-700 dark:from-blue-800 dark:to-indigo-900",
        gray: "bg-gradient-to-r from-gray-100 to-gray-200 dark:from-gray-800 dark:to-gray-900",
        green: "bg-gradient-to-r from-green-500 to-teal-600 dark:from-green-700 dark:to-teal-800",
    };

    // Text color classes based on bg color
    const textClasses = {
        blue: "text-white",
        gray: "text-gray-900 dark:text-white",
        green: "text-white",
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    return (
        <section
            id="service-cta"
            className={`py-20 w-full ${bgClasses[bgColor] || bgClasses.blue}`}
        >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeIn}
                    className={`text-center mb-12 ${
                        textClasses[bgColor] || textClasses.blue
                    }`}
                >
                    <h2 className="text-3xl md:text-4xl font-bold mb-6">
                        {title}
                    </h2>
                    <div 
                        className="text-xl max-w-3xl mx-auto opacity-90 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={staggerContainer}
                    className="flex flex-col md:flex-row justify-center items-center md:items-start space-y-8 md:space-y-0 md:space-x-12 mb-12"
                >
                    {displayBenefits.map((benefit, index) => {
                        // Get the icon component or default to CheckCircle
                        const IconComponent =
                            iconMap[benefit.icon] || CheckCircle;
                        const benefitText =
                            typeof benefit === "string"
                                ? benefit
                                : benefit?.text || "";
                        const isHtml =
                            typeof benefitText === "string" &&
                            /<([a-z][^>]*)>/i.test(benefitText || "");

                        return (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className={`flex flex-col items-center max-w-xs ${
                                    textClasses[bgColor] || textClasses.blue
                                }`}
                            >
                                <div className="bg-white dark:bg-gray-800 p-4 rounded-full mb-4 shadow-lg">
                                    <IconComponent
                                        className={`h-8 w-8 ${
                                            bgColor === "gray"
                                                ? "text-blue-600 dark:text-blue-400"
                                                : "text-blue-600"
                                        }`}
                                    />
                                </div>
                                {isHtml ? (
                                    <div
                                        className="text-center text-lg font-medium content-area"
                                        dangerouslySetInnerHTML={{
                                            __html: benefitText,
                                        }}
                                    />
                                ) : (
                                    <p className="text-center text-lg font-medium">
                                        {benefitText}
                                    </p>
                                )}
                            </motion.div>
                        );
                    })}
                </motion.div>

                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.3, duration: 0.6 }}
                    viewport={{ once: true }}
                    className="flex flex-col sm:flex-row justify-center items-center space-y-4 sm:space-y-0 sm:space-x-6"
                >
                    <a
                        href={primaryButtonLink}
                        className={`px-8 py-4 rounded-md font-bold text-lg shadow-xl transition-transform duration-200 transform hover:scale-105 flex items-center justify-center ${
                            bgColor === "gray"
                                ? "bg-blue-600 hover:bg-blue-700 text-white"
                                : "bg-white hover:bg-gray-100 text-blue-600"
                        }`}
                    >
                        {primaryButtonText}{" "}
                        <ArrowRight className="ml-2 h-5 w-5" />
                    </a>
                    <a
                        href={secondaryButtonLink}
                        className={`px-8 py-4 rounded-md font-bold text-lg transition-colors duration-200 ${
                            bgColor === "gray"
                                ? "border-2 border-blue-600 text-blue-600 dark:text-blue-400 hover:bg-blue-600 hover:text-white"
                                : "border-2 border-white text-white hover:bg-white hover:text-blue-600"
                        }`}
                    >
                        {secondaryButtonText}
                    </a>
                </motion.div>
            </div>
        </section>
    );
}

ServiceCTA.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    benefits: PropTypes.arrayOf(
        PropTypes.oneOfType([
            PropTypes.string,
            PropTypes.shape({
                text: PropTypes.string,
                icon: PropTypes.string,
            }),
        ])
    ),
    primaryButtonText: PropTypes.string,
    primaryButtonLink: PropTypes.string,
    secondaryButtonText: PropTypes.string,
    secondaryButtonLink: PropTypes.string,
    bgColor: PropTypes.oneOf(["blue", "gray", "green"]),
};
