import { motion } from "framer-motion";
import { FileText, Users, Edit, CheckCircle, Send, Clock } from "lucide-react";
import PropTypes from "prop-types";

export default function ServiceProcess({
    title = "Our Process",
    description = "We follow a structured approach to deliver high-quality academic papers",
    steps = [],
}) {
    // Default steps if none provided
    const defaultSteps = [
        {
            number: 1,
            title: "Submit Requirements",
            description:
                "Provide your paper details, including topic, academic level, deadline, and specific instructions.",
            icon: "FileText",
        },
        {
            number: 2,
            title: "Writer Assignment",
            description:
                "We match your project with a qualified writer who has expertise in your subject area.",
            icon: "Users",
        },
        {
            number: 3,
            title: "Research & Writing",
            description:
                "The writer conducts thorough research and crafts your paper according to your requirements.",
            icon: "Edit",
        },
        {
            number: 4,
            title: "Quality Check",
            description:
                "Our editors review the paper for quality, originality, and adherence to your instructions.",
            icon: "CheckCircle",
        },
        {
            number: 5,
            title: "Delivery",
            description:
                "You receive the completed paper before your deadline with time for review and revision requests.",
            icon: "Send",
        },
    ];

    // Use provided steps or default ones
    const displaySteps = steps.length > 0 ? steps : defaultSteps;

    // Map icon names to actual components
    const iconMap = {
        FileText: FileText,
        Users: Users,
        Edit: Edit,
        CheckCircle: CheckCircle,
        Send: Send,
        Clock: Clock,
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.2,
            },
        },
    };

    return (
        <section
            id="service-process"
            className="py-20 bg-gray-100 dark:bg-gray-800 w-full"
        >
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, margin: "-100px" }}
                    variants={fadeIn}
                    className="text-center mb-16"
                >
                    <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
                        {title}
                    </h2>
                    <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
                    <div 
                        className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                </motion.div>

                <motion.div
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, margin: "-50px" }}
                    variants={staggerContainer}
                    className="relative"
                >
                    {/* Timeline connector line */}
                    <div className="hidden md:block absolute left-1/2 transform -translate-x-1/2 h-full w-1 bg-blue-200 dark:bg-gray-700"></div>

                    {displaySteps.map((step, index) => {
                        // Get the icon component or default to Clock
                        const IconComponent = iconMap[step.icon] || Clock;
                        const isEven = index % 2 === 0;

                        return (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className={`md:flex items-center mb-12 ${
                                    isEven
                                        ? "md:flex-row"
                                        : "md:flex-row-reverse"
                                }`}
                            >
                                {/* Step content */}
                                <div
                                    className={`md:w-5/12 ${
                                        isEven
                                            ? "md:text-right md:pr-8"
                                            : "md:text-left md:pl-8"
                                    }`}
                                >
                                    <div className="mb-2">
                                        <span className="inline-block bg-blue-600 text-white text-sm py-1 px-3 rounded-full font-semibold">
                                            Step {step.number}
                                        </span>
                                    </div>
                                    <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-2">
                                        {step.title}
                                    </h3>
                                    <div 
                                        className="text-gray-600 dark:text-gray-300 content-area"
                                        dangerouslySetInnerHTML={{ __html: step.description }}
                                    />
                                </div>

                                {/* Center point with icon */}
                                <div className="hidden md:flex md:w-2/12 justify-center items-center">
                                    <div className="flex items-center justify-center h-16 w-16 rounded-full bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-blue-600 dark:text-blue-400 p-3 shadow-lg">
                                        <IconComponent className="h-8 w-8" />
                                    </div>
                                </div>

                                {/* Empty space for alternating layout */}
                                <div className="hidden md:block md:w-5/12"></div>

                                {/* Mobile version - always visible on small screens */}
                                <div className="flex md:hidden items-center mb-4">
                                    <div className="flex items-center justify-center h-12 w-12 rounded-full bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-gray-700 text-blue-600 dark:text-blue-400 p-2 mr-4 shadow-md">
                                        <IconComponent className="h-6 w-6" />
                                    </div>
                                    <div>
                                        <span className="inline-block bg-blue-600 text-white text-sm py-1 px-2 rounded-full font-semibold">
                                            Step {step.number}
                                        </span>
                                    </div>
                                </div>
                            </motion.div>
                        );
                    })}
                </motion.div>
            </div>
        </section>
    );
}

ServiceProcess.propTypes = {
    title: PropTypes.string,
    description: PropTypes.string,
    steps: PropTypes.arrayOf(
        PropTypes.shape({
            number: PropTypes.number.isRequired,
            title: PropTypes.string.isRequired,
            description: PropTypes.string.isRequired,
            icon: PropTypes.string,
        })
    ),
};
