import React from "react";
import { motion } from "framer-motion";
import { CheckCircle, Shield, Award, Users, FileText, Clock } from "lucide-react";

/**
 * Styled content section component matching existing service pages
 * Displays markdown content with professional styling, cards, and grids
 */
export default function StyledContentSection({
  title,
  description,
  bgColor = "white", // "white" | "gray" | "blue"
  eyebrow,
  className = "",
}) {
  // Convert markdown to HTML and extract structured content
  const processMarkdown = (markdown) => {
    if (!markdown || typeof markdown !== 'string') return { html: '', keyPoints: [], features: [] };
    
    const lines = markdown.split('\n');
    const keyPoints = [];
    const features = [];
    let html = '';
    let currentList = [];
    
    lines.forEach((line, index) => {
      const trimmed = line.trim();
      
      // Skip first header
      if (index === 0 && trimmed.match(/^#{1,6}\s+/)) {
        return;
      }
      
      // Extract ## headers as feature titles
      if (trimmed.match(/^##\s+(.+)$/)) {
        const featureTitle = trimmed.replace(/^##\s+/, '');
        features.push({ title: featureTitle, description: '' });
        return;
      }
      
      // Extract ### headers as subsection titles
      if (trimmed.match(/^###\s+(.+)$/)) {
        html += `<h3 class="text-xl font-semibold text-gray-900 mt-8 mb-4">${trimmed.replace(/^###\s+/, '')}</h3>`;
        return;
      }
      
      // Extract list items (- or * or numbered)
      if (trimmed.match(/^[-*]\s+(.+)$/)) {
        const item = trimmed.replace(/^[-*]\s+/, '');
        // Extract bold text as title, rest as description
        const boldMatch = item.match(/\*\*(.+?)\*\*(.*)/);
        if (boldMatch && boldMatch[1].trim().length < 50) {
          features.push({
            title: boldMatch[1].trim(),
            description: boldMatch[2].trim() || item.replace(/\*\*/g, '')
          });
        } else {
          keyPoints.push(item);
          currentList.push(item);
        }
        return;
      }
      
      // Process numbered lists
      if (trimmed.match(/^\d+[\.)]\s+(.+)$/)) {
        const item = trimmed.replace(/^\d+[\.)]\s+/, '');
        keyPoints.push(item);
        currentList.push(item);
        return;
      }
      
      // Regular paragraphs
      if (trimmed) {
        if (currentList.length > 0) {
          html += `<ul class="list-disc pl-6 space-y-2 mb-6">${currentList.map(item => `<li class="text-gray-700">${item}</li>`).join('')}</ul>`;
          currentList = [];
        }
        
        let processed = trimmed
          .replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>')
          .replace(/__([^_]+)__/g, '<strong>$1</strong>');
        
        html += `<p class="mb-4 text-gray-700 leading-relaxed">${processed}</p>`;
      } else {
        if (currentList.length > 0) {
          html += `<ul class="list-disc pl-6 space-y-2 mb-6">${currentList.map(item => `<li class="text-gray-700">${item}</li>`).join('')}</ul>`;
          currentList = [];
        }
      }
    });
    
    if (currentList.length > 0) {
      html += `<ul class="list-disc pl-6 space-y-2 mb-6">${currentList.map(item => `<li class="text-gray-700">${item}</li>`).join('')}</ul>`;
    }
    
    return { html: html.trim(), keyPoints, features };
  };

  const bgClass = bgColor === "gray" 
    ? "bg-gray-50 dark:bg-gray-900" 
    : bgColor === "blue" 
    ? "bg-blue-50 dark:bg-blue-900/20" 
    : "bg-white dark:bg-gray-800";

  const { html, keyPoints, features } = description ? processMarkdown(description) : { html: '', keyPoints: [], features: [] };
  
  // Icon mapping for features
  const iconMap = {
    CheckCircle,
    Shield,
    Award,
    Users,
    FileText,
    Clock,
  };

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
  };

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.15,
      },
    },
  };

  return (
    <section className={`py-16 ${bgClass} ${className}`}>
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          className="text-center mb-12"
        >
          {eyebrow && (
            <p className="text-sm font-semibold text-blue-600 dark:text-blue-400 uppercase tracking-wide mb-2">
              {eyebrow}
            </p>
          )}
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
            {title || "Section Title"}
          </h2>
          <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
        </motion.div>
        
        {/* Features Grid - If we have structured features, display in cards */}
        {features.length > 0 && (
          <motion.div
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true }}
            variants={staggerContainer}
            className="grid md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12"
          >
            {features.map((feature, index) => {
              const IconComponent = CheckCircle;
              return (
                <motion.div
                  key={index}
                  variants={fadeIn}
                  className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 hover:shadow-lg transition-all duration-300 hover:-translate-y-1 border border-gray-200 dark:border-gray-700"
                >
                  <div className="flex items-center justify-center h-12 w-12 rounded-md bg-blue-600 text-white mb-4">
                    <IconComponent className="h-6 w-6" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                    {feature.title}
                  </h3>
                  {feature.description && (
                    <p className="text-gray-600 dark:text-gray-300 text-sm leading-relaxed">
                      {feature.description}
                    </p>
                  )}
                </motion.div>
              );
            })}
          </motion.div>
        )}
        
        {/* Key Points - If we have key points, display in styled list */}
        {keyPoints.length > 0 && features.length === 0 && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            className="max-w-4xl mx-auto mb-8"
          >
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-8">
              <ul className="space-y-4">
                {keyPoints.map((point, index) => (
                  <li key={index} className="flex items-start">
                    <CheckCircle className="h-6 w-6 text-blue-600 dark:text-blue-400 mr-3 flex-shrink-0 mt-0.5" />
                    <span className="text-gray-700 dark:text-gray-300 leading-relaxed">{point}</span>
                  </li>
                ))}
              </ul>
            </div>
          </motion.div>
        )}
        
        {/* Main Content - Display prose content */}
        {html && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ delay: features.length > 0 || keyPoints.length > 0 ? 0.3 : 0 }}
            className="max-w-4xl mx-auto prose prose-lg dark:prose-invert prose-headings:text-gray-900 dark:prose-headings:text-white prose-p:text-gray-700 dark:prose-p:text-gray-300 prose-strong:text-gray-900 dark:prose-strong:text-white content-area"
            dangerouslySetInnerHTML={{ __html: html }}
          />
        )}
      </div>
    </section>
  );
}

