import React from "react";
import { motion } from "framer-motion";
import { Shield, Award, Users, CheckCircle, Star, TrendingUp } from "lucide-react";
import PropTypes from "prop-types";

/**
 * Trust Section Component - E-E-A-T Content
 * Displays trust factors in card-based grid layout
 */
export default function TrustSection({
  title = "Why Students Trust Us",
  description = "",
  trustFactors = [],
  metrics = [],
}) {
  const iconMap = {
    Shield,
    Award,
    Users,
    CheckCircle,
    Star,
    TrendingUp,
  };

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
  };

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.15,
      },
    },
  };

  return (
    <section className="py-20 bg-white dark:bg-gray-900 w-full">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={fadeIn}
          className="text-center mb-16"
        >
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white">
            {title}
          </h2>
          <div className="mt-4 h-1 w-24 bg-blue-600 mx-auto rounded-full"></div>
          {description && (
            <div 
              className="mt-6 max-w-2xl mx-auto text-xl text-gray-600 dark:text-gray-300 content-area"
              dangerouslySetInnerHTML={{ __html: description }}
            />
          )}
        </motion.div>

        {/* Trust Metrics */}
        {metrics.length > 0 && (
          <motion.div
            initial="hidden"
            whileInView="visible"
            viewport={{ once: true }}
            variants={staggerContainer}
            className="grid grid-cols-2 md:grid-cols-4 gap-6 mb-16"
          >
            {metrics.map((metric, index) => (
              <motion.div
                key={index}
                variants={fadeIn}
                className="text-center bg-gradient-to-br from-blue-50 to-indigo-50 dark:bg-gray-800 rounded-lg p-6 border border-blue-100 dark:border-gray-700"
              >
                <div className="text-4xl font-bold text-blue-600 dark:text-blue-400 mb-2">
                  {metric.value}
                </div>
                <div className="text-sm font-medium text-gray-700 dark:text-gray-300">
                  {metric.label}
                </div>
              </motion.div>
            ))}
          </motion.div>
        )}

        {/* Trust Factors Grid */}
        <motion.div
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          variants={staggerContainer}
          className="grid md:grid-cols-2 lg:grid-cols-3 gap-8"
        >
          {trustFactors.map((factor, index) => {
            const IconComponent = iconMap[factor.icon] || Shield;
            return (
              <motion.div
                key={index}
                variants={fadeIn}
                className="bg-gradient-to-r from-blue-50 to-cyan-50 dark:bg-gray-800 rounded-lg p-6 border border-gray-200 dark:border-gray-700
                           transition-all duration-300 hover:shadow-xl hover:-translate-y-1"
              >
                <div className="flex items-center justify-center h-14 w-14 rounded-full bg-blue-600 text-white mb-4">
                  <IconComponent className="h-7 w-7" />
                </div>
                <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-3">
                  {factor.title}
                </h3>
                <div 
                  className="text-gray-600 dark:text-gray-300 leading-relaxed content-area"
                  dangerouslySetInnerHTML={{ __html: factor.description }}
                />
              </motion.div>
            );
          })}
        </motion.div>
      </div>
    </section>
  );
}

TrustSection.propTypes = {
  title: PropTypes.string,
  description: PropTypes.string,
  trustFactors: PropTypes.arrayOf(
    PropTypes.shape({
      title: PropTypes.string.isRequired,
      description: PropTypes.string.isRequired,
      icon: PropTypes.string,
    })
  ),
  metrics: PropTypes.arrayOf(
    PropTypes.shape({
      value: PropTypes.string.isRequired,
      label: PropTypes.string.isRequired,
    })
  ),
};









