import React, { useState } from 'react';
import { 
    AlertCircle, 
    CheckCircle, 
    Clock, 
    Users, 
    Play, 
    FileText, 
    Eye, 
    RotateCcw,
    XCircle,
    Edit3,
    Send
} from 'lucide-react';
import { router } from '@inertiajs/react';

const StatusManagement = ({ order, nextPossibleStatuses = [], className = '' }) => {
    const [selectedStatus, setSelectedStatus] = useState('');
    const [reason, setReason] = useState('');
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [error, setError] = useState('');
    const [isCompletingReview, setIsCompletingReview] = useState(false);
    const [reviewNotes, setReviewNotes] = useState('');

    const statusConfig = {
        'pending': {
            name: 'Pending',
            icon: Clock,
            color: 'text-gray-500',
            bgColor: 'bg-gray-100',
            borderColor: 'border-gray-300'
        },
        'bidding': {
            name: 'Bidding',
            icon: Users,
            color: 'text-blue-500',
            bgColor: 'bg-blue-100',
            borderColor: 'border-blue-300'
        },
        'assigned': {
            name: 'Assigned',
            icon: Users,
            color: 'text-purple-500',
            bgColor: 'bg-purple-100',
            borderColor: 'border-purple-300'
        },
        'in_progress': {
            name: 'In Progress',
            icon: Play,
            color: 'text-orange-500',
            bgColor: 'bg-orange-100',
            borderColor: 'border-orange-300'
        },
        'submitted': {
            name: 'Submitted',
            icon: FileText,
            color: 'text-indigo-500',
            bgColor: 'bg-indigo-100',
            borderColor: 'border-indigo-300'
        },
        'under_review': {
            name: 'Under Review',
            icon: Eye,
            color: 'text-yellow-500',
            bgColor: 'bg-yellow-100',
            borderColor: 'border-yellow-300'
        },
        'approved': {
            name: 'Approved',
            icon: CheckCircle,
            color: 'text-green-500',
            bgColor: 'bg-green-100',
            borderColor: 'border-green-300'
        },
        'revision_requested': {
            name: 'Revision Requested',
            icon: RotateCcw,
            color: 'text-orange-500',
            bgColor: 'bg-orange-100',
            borderColor: 'border-orange-300'
        },
        'cancelled': {
            name: 'Cancelled',
            icon: XCircle,
            color: 'text-red-500',
            bgColor: 'bg-red-100',
            borderColor: 'border-red-300'
        }
    };

    const getCurrentStatusConfig = () => {
        return statusConfig[order.order_status] || statusConfig['pending'];
    };

    const handleStatusChange = async (e) => {
        e.preventDefault();
        
        if (!selectedStatus) {
            setError('Please select a status');
            return;
        }

        setIsSubmitting(true);
        setError('');

        try {
            await router.put(`/admin/orders/${order.id}/status`, {
                new_status: selectedStatus,
                reason: reason || 'Status changed by admin'
            }, {
                onSuccess: () => {
                    setSelectedStatus('');
                    setReason('');
                },
                onError: (errors) => {
                    setError(errors.status || errors.reason || 'Failed to update status');
                }
            });
        } catch (err) {
            setError('An error occurred while updating the status');
        } finally {
            setIsSubmitting(false);
        }
    };

    const handleCompleteReview = async (e) => {
        e.preventDefault();
        
        if (!reviewNotes.trim()) {
            setError('Please provide review notes');
            return;
        }

        setIsCompletingReview(true);
        setError('');

        try {
            await router.post(`/admin/orders/${order.id}/complete-review`, {
                review_notes: reviewNotes
            }, {
                onSuccess: () => {
                    setReviewNotes('');
                    // Refresh the page to show updated status
                    window.location.reload();
                },
                onError: (errors) => {
                    setError(errors.review_notes || 'Failed to complete review');
                }
            });
        } catch (err) {
            setError('An error occurred while completing the review');
        } finally {
            setIsCompletingReview(false);
        }
    };

    const formatDate = (dateString) => {
        if (!dateString) return null;
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getStatusHistory = () => {
        if (!order.status_history || !Array.isArray(order.status_history)) {
            return [];
        }
        return order.status_history.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
    };

    return (
        <div className={`bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6 ${className}`}>
            <div className="flex items-center mb-6">
                <Edit3 className="w-5 h-5 text-blue-600 dark:text-blue-400 mr-2" />
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                    Status Management
                </h3>
            </div>

            {/* Current Status */}
            <div className="mb-6">
                <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                    Current Status
                </h4>
                <div className="flex items-center">
                    <div className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${
                        getCurrentStatusConfig().bgColor
                    } ${getCurrentStatusConfig().color} ${getCurrentStatusConfig().borderColor} border`}>
                        {(() => {
                            const Icon = getCurrentStatusConfig().icon;
                            return <Icon className="w-4 h-4 mr-2" />;
                        })()}
                        {getCurrentStatusConfig().name}
                    </div>
                    {order.status_changed_at && (
                        <span className="ml-3 text-sm text-gray-500 dark:text-gray-400">
                            Since {formatDate(order.status_changed_at)}
                        </span>
                    )}
                </div>
                {order.status_change_reason && (
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-2">
                        Reason: {order.status_change_reason}
                    </p>
                )}
            </div>

            {/* Status Change Form */}
            {nextPossibleStatuses.length > 0 && (
                <div className="mb-6">
                    <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                        Change Status
                    </h4>
                    <form onSubmit={handleStatusChange} className="space-y-4">
                        <div>
                            <label htmlFor="status" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                New Status
                            </label>
                            <select
                                id="status"
                                value={selectedStatus}
                                onChange={(e) => setSelectedStatus(e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                required
                            >
                                <option value="">Select a status</option>
                                {nextPossibleStatuses.map((status) => {
                                    const config = statusConfig[status];
                                    return (
                                        <option key={status} value={status}>
                                            {config?.name || status}
                                        </option>
                                    );
                                })}
                            </select>
                        </div>

                        <div>
                            <label htmlFor="reason" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Reason for Change
                            </label>
                            <textarea
                                id="reason"
                                value={reason}
                                onChange={(e) => setReason(e.target.value)}
                                rows={3}
                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                placeholder="Explain why you're changing the status..."
                            />
                        </div>

                        {error && (
                            <div className="flex items-center p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-700 rounded-md">
                                <AlertCircle className="w-4 h-4 text-red-600 dark:text-red-400 mr-2" />
                                <span className="text-sm text-red-700 dark:text-red-300">{error}</span>
                            </div>
                        )}

                        <button
                            type="submit"
                            disabled={isSubmitting || !selectedStatus}
                            className="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-400 text-white font-medium rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors duration-200"
                        >
                            {isSubmitting ? 'Updating...' : 'Update Status'}
                        </button>
                    </form>
                </div>
            )}

            {/* Complete Review Form */}
            {order.order_status === 'under_review' && (
                <div className="mb-6">
                    <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                        Complete Review & Send to Client
                    </h4>
                    <form onSubmit={handleCompleteReview} className="space-y-4">
                        <div>
                            <label htmlFor="reviewNotes" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Review Notes
                            </label>
                            <textarea
                                id="reviewNotes"
                                value={reviewNotes}
                                onChange={(e) => setReviewNotes(e.target.value)}
                                rows={3}
                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                placeholder="Provide your review notes and any feedback for the client..."
                                required
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={isCompletingReview || !reviewNotes.trim()}
                            className="w-full px-4 py-2 bg-green-600 hover:bg-green-700 disabled:bg-gray-400 text-white font-medium rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition-colors duration-200 flex items-center justify-center"
                        >
                            {isCompletingReview ? (
                                <>
                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                    Completing Review...
                                </>
                            ) : (
                                <>
                                    <Send className="w-4 h-4 mr-2" />
                                    Complete Review & Send to Client
                                </>
                            )}
                        </button>
                    </form>
                </div>
            )}

            {/* Recent Status Changes */}
            {getStatusHistory().length > 0 && (
                <div>
                    <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                        Recent Status Changes
                    </h4>
                    <div className="space-y-3">
                        {getStatusHistory().slice(0, 5).map((history, index) => {
                            const config = statusConfig[history.to_status];
                            return (
                                <div key={index} className="flex items-center justify-between p-3 bg-white dark:bg-gray-800 rounded-md border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center">
                                        <div className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                            config?.bgColor || 'bg-gray-100'
                                        } ${config?.color || 'text-gray-800'}`}>
                                            {config?.name || history.to_status}
                                        </div>
                                        {history.reason && (
                                            <span className="ml-2 text-sm text-gray-600 dark:text-gray-400">
                                                - {history.reason}
                                            </span>
                                        )}
                                    </div>
                                    <span className="text-xs text-gray-500 dark:text-gray-400">
                                        {formatDate(history.created_at)}
                                    </span>
                                </div>
                            );
                        })}
                    </div>
                </div>
            )}

            {/* Key Timeline Dates */}
            <div className="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                    Key Timeline Dates
                </h4>
                <div className="space-y-2 text-sm">
                    {order.dateposted && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Order Created:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.dateposted)}</span>
                        </div>
                    )}
                    {order.deadline && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Client Deadline:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.deadline)}</span>
                        </div>
                    )}
                    {order.writer_deadline && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Writer Deadline:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.writer_deadline)}</span>
                        </div>
                    )}
                    {order.writer_assigned_at && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Writer Assigned:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.writer_assigned_at)}</span>
                        </div>
                    )}
                    {order.work_started_at && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Work Started:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.work_started_at)}</span>
                        </div>
                    )}
                    {order.work_submitted_at && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Work Submitted:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.work_submitted_at)}</span>
                        </div>
                    )}
                    {order.admin_reviewed_at && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Admin Reviewed:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.admin_reviewed_at)}</span>
                        </div>
                    )}
                    {order.client_approved_at && (
                        <div className="flex justify-between">
                            <span className="text-gray-600 dark:text-gray-400">Client Approved:</span>
                            <span className="text-gray-900 dark:text-white">{formatDate(order.client_approved_at)}</span>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

export default StatusManagement; 