import React, { useState, useEffect } from 'react';
import { Card } from './ui/Card';
import { Badge } from './ui/Badge';
import { Button } from './ui/Button';
import { Input } from './ui/Input';
import { Clock, AlertTriangle, AlertCircle, AlertOctagon } from 'lucide-react';

const UrgentOrdersWidget = ({ urgentOrders, urgentOrdersCount, userType = 'admin' }) => {
    // Wrap the entire component in a try-catch to prevent crashes
    try {
        // Ensure userType has a valid value
        const normalizedUserType = userType && ['admin', 'writer'].includes(userType) ? userType : 'admin';
        
        // Ensure urgentOrders has the expected structure
        const normalizedUrgentOrders = urgentOrders && typeof urgentOrders === 'object' ? urgentOrders : {
            overdue: [],
            critical_12h: [],
            urgent_24h: [],
            warning_48h: []
        };
        
        // Ensure urgentOrdersCount has default values
        const normalizedUrgentOrdersCount = urgentOrdersCount && typeof urgentOrdersCount === 'object' ? urgentOrdersCount : {
            overdue: 0,
            critical_12h: 0,
            urgent_24h: 0,
            warning_48h: 0,
            total: 0
        };

        const [filteredOrders, setFilteredOrders] = useState(normalizedUrgentOrders);
        const [filters, setFilters] = useState({
            urgency_level: '',
            order_status: '',
            search: ''
        });
        const [filterOptions, setFilterOptions] = useState({
            urgency_levels: {},
            order_statuses: []
        });
        const [isLoading, setIsLoading] = useState(false);
        const [error, setError] = useState(null);

        // Reset filters to default values
        const resetFilters = () => {
            try {
                setFilters({
                    urgency_level: '',
                    order_status: '',
                    search: ''
                });
                
                if (normalizedUrgentOrders && typeof normalizedUrgentOrders === 'object') {
                    setFilteredOrders(normalizedUrgentOrders);
                } else {
                    // Set default empty structure if normalizedUrgentOrders is invalid
                    setFilteredOrders({
                        overdue: [],
                        critical_12h: [],
                        urgent_24h: [],
                        warning_48h: []
                    });
                }
            } catch (error) {
                console.error('Error resetting filters:', error);
                // Set default values if there's an error
                setFilters({
                    urgency_level: '',
                    order_status: '',
                    search: ''
                });
                setFilteredOrders({
                    overdue: [],
                    critical_12h: [],
                    urgent_24h: [],
                    warning_48h: []
                });
            }
        };

        useEffect(() => {
            fetchFilterOptions();
            
            // Cleanup function to prevent memory leaks
            return () => {
                // Cancel any pending requests or cleanup state
                setIsLoading(false);
                setError(null);
            };
        }, []);

        useEffect(() => {
            try {
                if (normalizedUrgentOrders && typeof normalizedUrgentOrders === 'object') {
                    setFilteredOrders(normalizedUrgentOrders);
                    // Debug logging
                    console.log('UrgentOrdersWidget received data:', {
                        urgentOrders,
                        normalizedUrgentOrders,
                        hasUrgentOrders: Object.values(normalizedUrgentOrders).some(category => Array.isArray(category) && category.length > 0)
                    });
                }
            } catch (error) {
                console.error('Error updating filtered orders:', error);
                // Set default empty structure if there's an error
                setFilteredOrders({
                    overdue: [],
                    critical_12h: [],
                    urgent_24h: [],
                    warning_48h: []
                });
            }
        }, [urgentOrders]);

        const fetchFilterOptions = async () => {
            try {
                setIsLoading(true);
                setError(null);
                
                const response = await fetch('/urgent-orders/filter-options', {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    credentials: 'same-origin'
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const data = await response.json();
                if (data.success && data.data) {
                    // Ensure the filter options have the expected structure
                    const normalizedOptions = {
                        urgency_levels: data.data.urgency_levels && typeof data.data.urgency_levels === 'object' ? data.data.urgency_levels : {},
                        order_statuses: Array.isArray(data.data.order_statuses) ? data.data.order_statuses : []
                    };
                    setFilterOptions(normalizedOptions);
                } else {
                    setError('Failed to fetch filter options');
                }
            } catch (error) {
                console.error('Failed to fetch filter options:', error);
                setError('Failed to fetch filter options: ' + error.message);
                // Set default filter options to prevent crashes
                setFilterOptions({
                    urgency_levels: {
                        overdue: 'Overdue',
                        critical_12h: 'Critical (12h)',
                        urgent_24h: 'Urgent (24h)',
                        warning_48h: 'Warning (48h)'
                    },
                    order_statuses: []
                });
            } finally {
                setIsLoading(false);
            }
        };

        const applyFilters = async () => {
            try {
                setIsLoading(true);
                setError(null);
                
                // Validate filters before sending request
                if (!filters || typeof filters !== 'object') {
                    setError('Invalid filter configuration');
                    return;
                }
                
                // Get CSRF token from meta tag or cookie
                const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || 
                                 document.cookie.split('; ').find(row => row.startsWith('XSRF-TOKEN='))?.split('=')[1];
                
                const queryParams = new URLSearchParams();
                Object.entries(filters).forEach(([key, value]) => {
                    if (value && value.toString().trim() !== '') {
                        queryParams.append(key, value.toString().trim());
                    }
                });

                const response = await fetch(`/urgent-orders?${queryParams}`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json'
                    },
                    credentials: 'same-origin'
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const data = await response.json();
                if (data.success) {
                    // Ensure the filtered data has the expected structure
                    const normalizedData = data.data && typeof data.data === 'object' ? data.data : {
                        overdue: [],
                        critical_12h: [],
                        urgent_24h: [],
                        warning_48h: []
                    };
                    console.log('Filtered orders data:', normalizedData);
                    setFilteredOrders(normalizedData);
                } else {
                    setError('Failed to apply filters');
                }
            } catch (error) {
                console.error('Failed to apply filters:', error);
                setError('Failed to apply filters: ' + error.message);
            } finally {
                setIsLoading(false);
            }
        };

        const getUrgencyIcon = (urgencyLevel) => {
            switch (urgencyLevel) {
                case 'overdue':
                    return <AlertOctagon className="w-4 h-4 text-red-600" />;
                case 'critical':
                    return <AlertOctagon className="w-4 h-4 text-red-600" />;
                case 'urgent':
                    return <AlertTriangle className="w-4 h-4 text-orange-600" />;
                case 'warning':
                    return <AlertCircle className="w-4 h-4 text-yellow-600" />;
                case 'default':
                    return <Clock className="w-4 h-4 text-gray-600" />;
                default:
                    return <Clock className="w-4 h-4 text-gray-600" />;
            }
        };

        const getUrgencyColor = (urgencyLevel) => {
            switch (urgencyLevel) {
                case 'overdue':
                    return 'bg-red-100 text-red-800 border-red-200';
                case 'critical':
                    return 'bg-red-100 text-red-800 border-red-200';
                case 'urgent':
                    return 'bg-orange-100 text-orange-800 border-orange-200';
                case 'warning':
                    return 'bg-yellow-100 text-yellow-800 border-yellow-200';
                case 'default':
                    return 'bg-gray-100 text-gray-800 border-gray-200';
                default:
                    return 'bg-gray-100 text-gray-800 border-gray-200';
            }
        };

        const getStatusColor = (orderStatus) => {
            switch (orderStatus) {
                case 'assigned':
                    return 'bg-blue-100 text-blue-800 border-blue-200';
                case 'in_progress':
                    return 'bg-yellow-100 text-yellow-800 border-yellow-200';
                case 'submitted':
                    return 'bg-purple-100 text-purple-800 border-purple-200';
                case 'under_review':
                    return 'bg-orange-100 text-orange-800 border-orange-200';
                case 'approved':
                    return 'bg-green-100 text-green-800 border-green-200';
                case 'revision_requested':
                    return 'bg-red-100 text-red-800 border-red-200';
                case 'bidding':
                    return 'bg-indigo-100 text-indigo-800 border-indigo-200';
                case 'pending':
                    return 'bg-gray-100 text-gray-800 border-gray-200';
                case 'unknown':
                    return 'bg-gray-100 text-gray-800 border-gray-200';
                default:
                    return 'bg-gray-100 text-gray-800 border-gray-200';
            }
        };

        const formatTimeRemaining = (hours) => {
            // Handle undefined, null, or non-numeric values
            if (hours === undefined || hours === null || isNaN(hours)) {
                return 'Unknown';
            }
            
            if (hours < 0) {
                const overdueHours = Math.abs(hours);
                if (overdueHours < 24) {
                    return `${Math.round(overdueHours)}h overdue`;
                }
                const overdueDays = Math.floor(overdueHours / 24);
                const remainingHours = overdueHours % 24;
                return `${overdueDays}d ${Math.round(remainingHours)}h overdue`;
            }
            if (hours < 1) {
                const minutes = Math.round(hours * 60);
                return `${minutes} minute${minutes !== 1 ? 's' : ''}`;
            }
            if (hours < 24) {
                return `${Math.round(hours)} hour${Math.round(hours) !== 1 ? 's' : ''}`;
            }
            const days = Math.floor(hours / 24);
            const remainingHours = hours % 24;
            return `${days} day${days !== 1 ? 's' : ''}${remainingHours > 0 ? ` ${Math.round(remainingHours)}h` : ''}`;
        };

        if (isLoading && !urgentOrders) {
            return (
                <Card className="p-6">
                    <div className="text-center text-gray-500">
                        <Clock className="w-12 h-12 mx-auto mb-2 text-gray-300 animate-spin" />
                        <p>Loading urgent orders...</p>
                    </div>
                </Card>
            );
        }

        // Check if there are any urgent orders to display
        const hasUrgentOrders = (() => {
            try {
                if (!normalizedUrgentOrders || typeof normalizedUrgentOrders !== 'object') {
                    return false;
                }
                
                const values = Object.values(normalizedUrgentOrders);
                if (!Array.isArray(values)) {
                    return false;
                }
                
                return values.some(category => Array.isArray(category) && category.length > 0);
            } catch (error) {
                console.error('Error checking for urgent orders:', error);
                return false;
            }
        })();

        if (!hasUrgentOrders) {
            return (
                <Card className="p-6">
                    <div className="text-center text-gray-500">
                        <Clock className="w-12 h-12 mx-auto mb-2 text-gray-300" />
                        <p>No urgent orders at the moment</p>
                    </div>
                </Card>
            );
        }

        return (
            <Card className="p-6">
                <div className="flex items-center justify-between mb-6">
                    <div>
                        <h3 className="text-lg font-semibold text-gray-900">Urgent Orders</h3>
                        <p className="text-sm text-gray-600">Orders requiring immediate attention</p>
                    </div>
                    <div className="flex gap-2">
                        <Badge variant="destructive" className="px-3 py-1">
                            {normalizedUrgentOrdersCount?.overdue || 0} Overdue
                        </Badge>
                        <Badge variant="destructive" className="px-3 py-1">
                            {normalizedUrgentOrdersCount?.critical_12h || 0} Critical
                        </Badge>
                        <Badge variant="warning" className="px-3 py-1">
                            {normalizedUrgentOrdersCount?.urgent_24h || 0} Urgent
                        </Badge>
                        <Badge variant="info" className="px-3 py-1">
                            {normalizedUrgentOrdersCount?.warning_48h || 0} Warning
                        </Badge>
                    </div>
                </div>

                {/* Error Display */}
                {error && (
                    <div className="mb-4 p-3 bg-red-100 border border-red-400 text-red-700 rounded">
                        {error}
                    </div>
                )}

                {/* Filters */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                    <select
                        value={filters?.urgency_level || ''}
                        onChange={(e) => {
                            try {
                                const value = e.target.value || '';
                                setFilters(prev => ({ ...prev, urgency_level: value }));
                            } catch (error) {
                                console.error('Error updating urgency level filter:', error);
                            }
                        }}
                        className="flex h-10 w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:border-gray-700 dark:text-white"
                    >
                        <option value="">All Urgency Levels</option>
                        {filterOptions?.urgency_levels && typeof filterOptions.urgency_levels === 'object' && 
                         Object.entries(filterOptions.urgency_levels).map(([key, label]) => (
                            <option key={key} value={key}>{label}</option>
                        ))}
                    </select>

                    <select
                        value={filters?.order_status || ''}
                        onChange={(e) => {
                            try {
                                const value = e.target.value || '';
                                setFilters(prev => ({ ...prev, order_status: value }));
                            } catch (error) {
                                console.error('Error updating order status filter:', error);
                            }
                        }}
                        className="flex h-10 w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:border-gray-700 dark:text-white"
                    >
                        <option value="">All Statuses</option>
                        {Array.isArray(filterOptions?.order_statuses) && filterOptions.order_statuses.map(status => (
                            <option key={status} value={status}>
                                {status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                            </option>
                        ))}
                    </select>

                    <Input
                        placeholder="Search orders..."
                        value={filters?.search || ''}
                        onChange={(e) => {
                            try {
                                const value = e.target.value || '';
                                setFilters(prev => ({ ...prev, search: value }));
                            } catch (error) {
                                console.error('Error updating search filter:', error);
                            }
                        }}
                    />

                    <Button onClick={() => {
                        try {
                            applyFilters();
                        } catch (error) {
                            console.error('Error applying filters:', error);
                            setError('Failed to apply filters: ' + error.message);
                        }
                    }} disabled={isLoading} className="w-full md:w-auto">
                        {isLoading ? 'Loading...' : 'Apply Filters'}
                    </Button>
                    
                    <Button 
                        onClick={() => {
                            try {
                                resetFilters();
                            } catch (error) {
                                console.error('Error resetting filters:', error);
                                setError('Failed to reset filters: ' + error.message);
                            }
                        }} 
                        variant="outline" 
                        disabled={isLoading} 
                        className="w-full md:w-auto"
                    >
                        Reset Filters
                    </Button>
                </div>

                {/* Urgent Orders List */}
                <div className="space-y-4">
                    {(() => {
                        try {
                            if (!filteredOrders || typeof filteredOrders !== 'object') {
                                return (
                                    <div className="text-center py-8 text-gray-500">
                                        <Clock className="w-12 h-12 mx-auto mb-2 text-gray-300" />
                                        <p>No urgent orders data available</p>
                                    </div>
                                );
                            }

                            // Ensure filteredOrders is not null and has entries
                            const entries = Object.entries(filteredOrders);
                            if (!Array.isArray(entries) || entries.length === 0) {
                                return (
                                    <div className="text-center py-8 text-gray-500">
                                        <Clock className="w-12 h-12 mx-auto mb-2 text-gray-300" />
                                        <p>No urgent orders categories found</p>
                                    </div>
                                );
                            }

                            return entries.map(([urgencyLevel, orders]) => {
                                // Ensure orders is always an array and has items
                                if (!Array.isArray(orders) || orders.length === 0) return null;
                                
                                const urgencyLabel = urgencyLevel === 'overdue' ? 'Overdue' : urgencyLevel.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
                                
                                return (
                                    <div key={urgencyLevel} className="border rounded-lg overflow-hidden">
                                        <div className={`px-4 py-2 ${getUrgencyColor(urgencyLevel === 'overdue' ? 'overdue' : (urgencyLevel?.split('_')[0] || 'default'))}`}>
                                            <div className="flex items-center gap-2">
                                                {getUrgencyIcon(urgencyLevel === 'overdue' ? 'overdue' : (urgencyLevel?.split('_')[0] || 'default'))}
                                                <span className="font-medium">{urgencyLabel}</span>
                                                <Badge variant="outline">{orders.length} order{orders.length !== 1 ? 's' : ''}</Badge>
                                            </div>
                                        </div>
                                        
                                        <div className="divide-y">
                                            {Array.isArray(orders) && orders.map((order) => (
                                                <div key={order?.id || Math.random()} className="px-4 py-3 hover:bg-gray-50">
                                                    <div className="flex items-center justify-between">
                                                        <div className="flex-1">
                                                            <div className="flex items-center gap-2 mb-1">
                                                                <span className="font-medium text-gray-900">
                                                                    #{order?.order_number || 'N/A'}
                                                                </span>
                                                                <Badge 
                                                                    className={`text-xs font-medium ${getStatusColor(order?.order_status || 'unknown')}`}
                                                                >
                                                                    {(order?.order_status || 'unknown').replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                                                </Badge>
                                                            </div>
                                                            <p className="text-sm text-gray-600 mb-1">{order?.title || 'No title'}</p>
                                                            <div className="flex items-center gap-4 text-xs text-gray-500">
                                                                <span>Client: {order?.user?.name || 'Unknown'}</span>
                                                                {order?.writer && <span>Writer: {order.writer.name}</span>}
                                                                <span>Pages: {order?.pages || 'N/A'}</span>
                                                            </div>
                                                        </div>
                                                        <div className="text-right">
                                                            <div className="text-sm font-medium text-gray-900">
                                                                {formatTimeRemaining(order?.hours_remaining || 0)}
                                                            </div>
                                                            <div className="text-xs text-gray-500">
                                                                Due: {order?.deadline ? new Date(order.deadline).toLocaleDateString() : 'N/A'}
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                );
                            });
                        } catch (error) {
                            console.error('Error rendering urgent orders:', error);
                            return (
                                <div className="text-center py-8 text-red-500">
                                    <AlertTriangle className="w-12 h-12 mx-auto mb-2 text-red-300" />
                                    <p>Error loading urgent orders</p>
                                    <p className="text-sm text-red-400 mt-1">{error.message}</p>
                                </div>
                            );
                        }
                    })()}
                    
                    {/* No Results Message */}
                    {(() => {
                        try {
                            if (!filteredOrders || typeof filteredOrders !== 'object') {
                                return null;
                            }
                            
                            const values = Object.values(filteredOrders);
                            if (!Array.isArray(values)) {
                                return null;
                            }
                            
                            return values.every(category => !Array.isArray(category) || category.length === 0) ? (
                                <div className="text-center py-8 text-gray-500">
                                    <Clock className="w-12 h-12 mx-auto mb-2 text-gray-300" />
                                    <p>No orders match the selected filters</p>
                                </div>
                            ) : null;
                        } catch (error) {
                            console.error('Error checking for no results:', error);
                            return null;
                        }
                    })()}
                </div>
            </Card>
        );
    } catch (error) {
        console.error('Error rendering UrgentOrdersWidget:', error);
        return (
            <Card className="p-6">
                <div className="text-center text-gray-500">
                    <AlertTriangle className="w-12 h-12 mx-auto mb-2 text-red-300" />
                    <p>Error loading urgent orders widget.</p>
                    <p className="text-sm text-red-400 mt-1">{error.message}</p>
                </div>
            </Card>
        );
    }
};

export default UrgentOrdersWidget; 