import React, { useState } from 'react';
import { router } from '@inertiajs/react';
import { X, DollarSign, CreditCard, AlertCircle, CheckCircle } from 'lucide-react';

export default function WithdrawalRequestForm({ isOpen, onClose, eligibility, availableAmount }) {
    const [formData, setFormData] = useState({
        amount: '',
        payment_method: 'paypal',
        payment_details: {
            paypal_email: '',
            bank_account: '',
            bank_routing: '',
            account_holder: ''
        }
    });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [errors, setErrors] = useState({});

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const validateForm = () => {
        const newErrors = {};

        // Validate amount
        if (!formData.amount || parseFloat(formData.amount) <= 0) {
            newErrors.amount = 'Please enter a valid amount';
        } else if (parseFloat(formData.amount) > availableAmount) {
            newErrors.amount = 'Amount cannot exceed available balance';
        } else if (parseFloat(formData.amount) < eligibility.minimum_withdrawal) {
            newErrors.amount = `Minimum withdrawal amount is ${formatCurrency(eligibility.minimum_withdrawal)}`;
        }

        // Validate payment method specific details
        if (formData.payment_method === 'paypal') {
            if (!formData.payment_details.paypal_email) {
                newErrors.paypal_email = 'PayPal email is required';
            } else if (!/\S+@\S+\.\S+/.test(formData.payment_details.paypal_email)) {
                newErrors.paypal_email = 'Please enter a valid email address';
            }
        } else if (formData.payment_method === 'bank_transfer') {
            if (!formData.payment_details.bank_account) {
                newErrors.bank_account = 'Bank account number is required';
            }
            if (!formData.payment_details.bank_routing) {
                newErrors.bank_routing = 'Bank routing number is required';
            }
            if (!formData.payment_details.account_holder) {
                newErrors.account_holder = 'Account holder name is required';
            }
        }

        setErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        if (!validateForm()) {
            return;
        }

        setIsSubmitting(true);
        
        try {
            // Use fetch to submit the form and handle the response
            const response = await fetch('/writer/payment/withdrawal-request', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({
                    amount: parseFloat(formData.amount),
                    payment_method: formData.payment_method,
                    payment_details: formData.payment_details
                })
            });

            const result = await response.json();

            if (response.ok && result.success) {
                // Success - close form and show success message
                onClose();
                // You could add a success toast here or redirect
                window.location.reload(); // Refresh to show success message
            } else {
                // Error from server
                setErrors({ 
                    general: result.message || 'Failed to submit withdrawal request. Please try again.' 
                });
            }

        } catch (error) {
            console.error('Failed to submit withdrawal request:', error);
            setErrors({ 
                general: 'Network error. Please check your connection and try again.' 
            });
        } finally {
            setIsSubmitting(false);
        }
    };

    const handleInputChange = (field, value) => {
        setFormData(prev => ({
            ...prev,
            [field]: value
        }));
        
        // Clear field-specific errors when user starts typing
        if (errors[field]) {
            setErrors(prev => ({ ...prev, [field]: '' }));
        }
    };

    const handlePaymentDetailChange = (field, value) => {
        setFormData(prev => ({
            ...prev,
            payment_details: {
                ...prev.payment_details,
                [field]: value
            }
        }));
        
        // Clear field-specific errors when user starts typing
        if (errors[field]) {
            setErrors(prev => ({ ...prev, [field]: '' }));
        }
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-20 mx-auto p-5 border w-full max-w-md shadow-lg rounded-md bg-white">
                <div className="flex items-center justify-between mb-4">
                    <h3 className="text-lg font-medium text-gray-900">Request Withdrawal</h3>
                    <button
                        onClick={onClose}
                        className="text-gray-400 hover:text-gray-600"
                    >
                        <X size={20} />
                    </button>
                </div>

                {/* Eligibility Status */}
                {!eligibility.can_withdraw && (
                    <div className="mb-4 p-3 bg-yellow-100 border border-yellow-400 text-yellow-800 rounded-md">
                        <div className="flex items-start">
                            <AlertCircle size={20} className="mr-2 mt-0.5 flex-shrink-0" />
                            <div className="text-sm">
                                <p className="font-medium mb-1">⚠️ Withdrawal Currently Unavailable</p>
                                <ul className="space-y-1">
                                    {eligibility.reasons && eligibility.reasons.map((reason, index) => (
                                        <li key={index} className="flex items-center">
                                            <span className="w-1.5 h-1.5 bg-yellow-600 rounded-full mr-2"></span>
                                            {reason}
                                        </li>
                                    ))}
                                </ul>
                                {eligibility.next_withdrawal_date && (
                                    <p className="mt-2 text-xs">
                                        <strong>Next withdrawal date:</strong> {new Date(eligibility.next_withdrawal_date).toLocaleDateString('en-US', {
                                            year: 'numeric',
                                            month: 'long',
                                            day: 'numeric'
                                        })}
                                    </p>
                                )}
                            </div>
                        </div>
                    </div>
                )}

                {errors.general && (
                    <div className="mb-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded-md">
                        <div className="flex items-start">
                            <AlertCircle size={20} className="mr-3 mt-0.5 flex-shrink-0" />
                            <div>
                                <p className="font-medium mb-1">Withdrawal Request Failed</p>
                                <p className="text-sm">{errors.general}</p>
                                {errors.general.includes('15th and 30th') && (
                                    <div className="mt-2 p-2 bg-red-50 border border-red-200 rounded text-xs">
                                        <p className="font-medium mb-1">💡 Tip:</p>
                                        <ul className="space-y-1">
                                            <li>• Withdrawals are only processed on the 15th and 30th of each month</li>
                                            <li>• This helps us batch process payments efficiently</li>
                                            <li>• You can still submit your request now - it will be processed on the next valid date</li>
                                        </ul>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                )}

                <form onSubmit={handleSubmit} className="space-y-4">
                    {/* Amount */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                            Withdrawal Amount *
                        </label>
                        <div className="relative">
                            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <DollarSign size={16} className="text-gray-400" />
                            </div>
                            <input
                                type="number"
                                step="0.01"
                                min={eligibility.minimum_withdrawal}
                                max={availableAmount}
                                value={formData.amount}
                                onChange={(e) => handleInputChange('amount', e.target.value)}
                                className={`w-full pl-10 pr-3 py-2 border rounded-md ${
                                    errors.amount ? 'border-red-300' : 'border-gray-300'
                                } focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                                placeholder={`${formatCurrency(eligibility.minimum_withdrawal)} - ${formatCurrency(availableAmount)}`}
                            />
                        </div>
                        {errors.amount && (
                            <p className="mt-1 text-sm text-red-600">{errors.amount}</p>
                        )}
                        <p className="mt-1 text-xs text-gray-500">
                            Available: {formatCurrency(availableAmount)} | Min: {formatCurrency(eligibility.minimum_withdrawal)}
                        </p>
                    </div>

                    {/* Payment Method */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                            Payment Method *
                        </label>
                        <select
                            value={formData.payment_method}
                            onChange={(e) => handleInputChange('payment_method', e.target.value)}
                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        >
                            <option value="paypal">PayPal</option>
                            <option value="bank_transfer">Bank Transfer</option>
                        </select>
                    </div>

                    {/* PayPal Details */}
                    {formData.payment_method === 'paypal' && (
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                PayPal Email *
                            </label>
                            <input
                                type="email"
                                value={formData.payment_details.paypal_email}
                                onChange={(e) => handlePaymentDetailChange('paypal_email', e.target.value)}
                                className={`w-full border rounded-md px-3 py-2 ${
                                    errors.paypal_email ? 'border-red-300' : 'border-gray-300'
                                } focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                                placeholder="your-email@example.com"
                            />
                            {errors.paypal_email && (
                                <p className="mt-1 text-sm text-red-600">{errors.paypal_email}</p>
                            )}
                        </div>
                    )}

                    {/* Bank Transfer Details */}
                    {formData.payment_method === 'bank_transfer' && (
                        <div className="space-y-3">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Account Holder Name *
                                </label>
                                <input
                                    type="text"
                                    value={formData.payment_details.account_holder}
                                    onChange={(e) => handlePaymentDetailChange('account_holder', e.target.value)}
                                    className={`w-full border rounded-md px-3 py-2 ${
                                        errors.account_holder ? 'border-red-300' : 'border-gray-300'
                                    } focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                                    placeholder="John Doe"
                                />
                                {errors.account_holder && (
                                    <p className="mt-1 text-sm text-red-600">{errors.account_holder}</p>
                                )}
                            </div>
                            
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Bank Account Number *
                                </label>
                                <input
                                    type="text"
                                    value={formData.payment_details.bank_account}
                                    onChange={(e) => handlePaymentDetailChange('bank_account', e.target.value)}
                                    className={`w-full border rounded-md px-3 py-2 ${
                                        errors.bank_account ? 'border-red-300' : 'border-gray-300'
                                    } focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                                    placeholder="1234567890"
                                />
                                {errors.bank_account && (
                                    <p className="mt-1 text-sm text-red-600">{errors.bank_account}</p>
                                )}
                            </div>
                            
                            <div>
                                <label className="block text-sm font-medium text-gray-700 mb-1">
                                    Bank Routing Number *
                                </label>
                                <input
                                    type="text"
                                    value={formData.payment_details.bank_routing}
                                    onChange={(e) => handlePaymentDetailChange('bank_routing', e.target.value)}
                                    className={`w-full border rounded-md px-3 py-2 ${
                                        errors.bank_routing ? 'border-red-300' : 'border-gray-300'
                                    } focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                                    placeholder="123456789"
                                />
                                {errors.bank_routing && (
                                    <p className="mt-1 text-sm text-red-600">{errors.bank_routing}</p>
                                )}
                            </div>
                        </div>
                    )}

                    {/* Important Notes */}
                    <div className="bg-blue-50 border border-blue-200 rounded-md p-3">
                        <div className="flex items-start">
                            <AlertCircle size={16} className="text-blue-600 mr-2 mt-0.5 flex-shrink-0" />
                            <div className="text-sm text-blue-800">
                                <p className="font-medium mb-1">Important Information:</p>
                                <ul className="space-y-1 text-xs">
                                    <li>• Withdrawals are processed on the 15th and 30th of each month</li>
                                    <li>• Processing time: 3-5 business days after approval</li>
                                    <li>• Minimum withdrawal: {formatCurrency(eligibility.minimum_withdrawal)}</li>
                                    <li>• You can only have one pending withdrawal at a time</li>
                                </ul>
                            </div>
                        </div>
                    </div>

                    {/* Submit Button */}
                    <div className="flex justify-end space-x-3 pt-4">
                        <button
                            type="button"
                            onClick={onClose}
                            className="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50"
                        >
                            Cancel
                        </button>
                        <button
                            type="submit"
                            disabled={isSubmitting || !eligibility.can_withdraw}
                            className={`px-6 py-2 rounded-md flex items-center ${
                                eligibility.can_withdraw 
                                    ? 'bg-green-600 text-white hover:bg-green-700 disabled:opacity-50' 
                                    : 'bg-gray-400 text-gray-600 cursor-not-allowed'
                            }`}
                        >
                            {isSubmitting ? (
                                <>
                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                    Submitting...
                                </>
                            ) : !eligibility.can_withdraw ? (
                                <>
                                    <AlertCircle size={16} className="mr-2" />
                                    Cannot Withdraw
                                </>
                            ) : (
                                <>
                                    <CheckCircle size={16} className="mr-2" />
                                    Submit Request
                                </>
                            )}
                        </button>
                    </div>
                    
                    {!eligibility.can_withdraw && (
                        <div className="mt-3 text-center">
                            <p className="text-xs text-gray-500">
                                Please resolve the issues above before submitting a withdrawal request.
                            </p>
                        </div>
                    )}
                </form>
            </div>
        </div>
    );
} 