import React, { useState, useEffect, useRef } from "react";
import { Calendar as CalendarIcon } from "lucide-react";
import { cn, formatDate } from "@/lib/utils";
import { Button } from "@/Components/ui/Button";
import { Calendar } from "@/Components/ui/Calendar";
import { Popover } from "@headlessui/react";

const DatePicker = ({
    date,
    onSelect,
    className,
    placeholder = "Select date...",
    disabled = false,
    ...props
}) => {
    const [selectedDate, setSelectedDate] = useState(date);
    const [isOpen, setIsOpen] = useState(false);
    const popoverRef = useRef(null);

    // Update internal state when the date prop changes
    useEffect(() => {
        setSelectedDate(date);
    }, [date]);

    // Handle clicks outside to close the popover
    useEffect(() => {
        const handleClickOutside = (event) => {
            if (
                popoverRef.current &&
                !popoverRef.current.contains(event.target)
            ) {
                setIsOpen(false);
            }
        };

        document.addEventListener("mousedown", handleClickOutside);
        return () => {
            document.removeEventListener("mousedown", handleClickOutside);
        };
    }, []);

    const handleSelect = (newDate) => {
        setSelectedDate(newDate);
        onSelect?.(newDate);
        setIsOpen(false);
    };

    const handleClear = (e) => {
        e.stopPropagation();
        setSelectedDate(null);
        onSelect?.(null);
    };

    return (
        <div className={cn("relative", className)} {...props}>
            <Popover>
                <Popover.Button
                    as="div"
                    className="w-full"
                    onClick={() => !disabled && setIsOpen(!isOpen)}
                >
                    <Button
                        variant="outline"
                        role="combobox"
                        aria-expanded={isOpen}
                        disabled={disabled}
                        className={cn(
                            "w-full justify-start text-left font-normal",
                            !selectedDate && "text-gray-500 dark:text-gray-400"
                        )}
                    >
                        <CalendarIcon className="mr-2 h-4 w-4" />
                        {selectedDate ? (
                            formatDate(selectedDate, { dateStyle: "medium" })
                        ) : (
                            <span>{placeholder}</span>
                        )}
                    </Button>
                </Popover.Button>
                {isOpen && (
                    <Popover.Panel
                        static
                        ref={popoverRef}
                        className="absolute z-50 mt-1 bg-white dark:bg-gray-800 rounded-md shadow-lg p-2 border border-gray-200 dark:border-gray-700"
                    >
                        <Calendar
                            mode="single"
                            selected={
                                selectedDate
                                    ? new Date(selectedDate)
                                    : undefined
                            }
                            onSelect={handleSelect}
                            initialFocus
                        />
                        {selectedDate && (
                            <div className="flex justify-end mt-2">
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="sm"
                                    onClick={handleClear}
                                >
                                    Clear
                                </Button>
                            </div>
                        )}
                    </Popover.Panel>
                )}
            </Popover>
        </div>
    );
};

export { DatePicker };
