import { Label } from "@/Components/ui/Label";
import { useState, useEffect } from "react";

export const DeadlineSelect = ({
    id,
    label,
    value,
    setData,
    name,
    options,
    error,
    placeholder = "Select Deadline",
    required,
}) => {
    const [clientDeadline, setClientDeadline] = useState("");
    const [timezone, setTimezone] = useState("");

    // Get the current timezone when the component mounts
    useEffect(() => {
        // Get timezone name
        const timezoneName = Intl.DateTimeFormat().resolvedOptions().timeZone;
        setTimezone(timezoneName);
    }, []);

    useEffect(() => {
        if (!value) {
            setClientDeadline("");
            return;
        }

        const deadlineInfo = value.split("#");
        if (deadlineInfo.length < 3) return;

        // Parse the urgency as the actual deadline value
        const urgencyValue = parseFloat(deadlineInfo[1]);
        const duration = deadlineInfo[2];

        // Calculate deadline date
        const now = new Date();
        const deadlineDate = new Date(now.getTime()); // Create copy of current time

        // Add the hours/days using the local timezone
        if (duration.toLowerCase() === "hours") {
            // Use UTC methods to ensure consistent timezone handling
            deadlineDate.setUTCHours(deadlineDate.getUTCHours() + urgencyValue);
        } else if (duration.toLowerCase() === "days") {
            deadlineDate.setUTCDate(deadlineDate.getUTCDate() + urgencyValue);
        }

        // Format using Intl.DateTimeFormat for consistent timezone handling
        // Format display date using the user's timezone
        const displayFormatter = new Intl.DateTimeFormat("en-US", {
            weekday: "short",
            year: "numeric",
            month: "short",
            day: "numeric",
            hour: "2-digit",
            minute: "2-digit",
            timeZoneName: "short",
            timeZone: timezone,
        });

        setClientDeadline(displayFormatter.format(deadlineDate));

        // Convert to ISO string in the user's timezone
        const userTimezoneOffset = deadlineDate.getTimezoneOffset() * 60000;
        const localISOTime = new Date(
            deadlineDate - userTimezoneOffset
        ).toISOString();

        // Update the form data with both deadline and timezone
        setData("clientDeadline", localISOTime);
        setData("clientTimezone", timezone);
    }, [value, timezone, setData]);

    return (
        <div>
            <Label htmlFor={id}>{label}</Label>
            <select
                id={id}
                value={value}
                onChange={(e) => setData(name, e.target.value)}
                className="mt-1 block w-full rounded-md border border-gray-700 bg-gray-800 px-3 py-2 text-gray-100 focus:border-blue-500 focus:ring-blue-500 sm:text-sm"
                required={required}
            >
                <option value="" disabled>
                    {placeholder}
                </option>
                {options.map((deadline) => (
                    <option
                        key={deadline.id}
                        value={`${deadline.value}#${deadline.urgency}#${deadline.duration}`}
                    >
                        {deadline.urgency} {deadline.duration}
                    </option>
                ))}
            </select>
            <div className="flex flex-col md:flex-row md:justify-between mt-1">
                {clientDeadline && (
                    <span className="text-green-500 text-sm mt-2 block">
                        Deadline: {clientDeadline}
                    </span>
                )}
                {clientDeadline && (
                    <span className="text-green-500 text-sm mt-2 block">
                        Timezone: {timezone}
                    </span>
                )}
            </div>

            {error && <p className="text-sm text-red-500">{error}</p>}
        </div>
    );
};
