import React from "react";
import { cn } from "@/lib/utils";

const Progress = React.forwardRef(
    (
        {
            className,
            value,
            max = 100,
            color = "blue",
            size = "md",
            indeterminate = false,
            label,
            ...props
        },
        ref
    ) => {
        const percentage =
            value != null ? Math.min(Math.max(0, (value / max) * 100), 100) : 0;

        // Color variants
        const colorVariants = {
            blue: "bg-blue-600 dark:bg-blue-500",
            green: "bg-green-600 dark:bg-green-500",
            red: "bg-red-600 dark:bg-red-500",
            yellow: "bg-yellow-600 dark:bg-yellow-500",
            purple: "bg-purple-600 dark:bg-purple-500",
            gray: "bg-gray-600 dark:bg-gray-500",
        };

        // Size variants
        const sizeVariants = {
            xs: "h-1",
            sm: "h-1.5",
            md: "h-2",
            lg: "h-3",
            xl: "h-4",
        };

        // Animation for indeterminate state
        const indeterminateStyles = indeterminate
            ? "relative overflow-hidden before:absolute before:inset-0 before:bg-current before:animate-indeterminate-progress"
            : "";

        return (
            <div
                className={cn(
                    "w-full overflow-hidden rounded-full bg-gray-200 dark:bg-gray-700",
                    className
                )}
                role="progressbar"
                aria-valuenow={indeterminate ? undefined : value}
                aria-valuemin={0}
                aria-valuemax={max}
                aria-valuetext={`${percentage.toFixed(0)}%`}
                aria-label={label}
                ref={ref}
                style={{
                    // Custom size if sizeVariants doesn't include the provided size
                    height: !sizeVariants[size] ? size : undefined,
                }}
                {...props}
            >
                <div
                    className={cn(
                        "transition-all ease-in-out duration-300 h-full w-full",
                        sizeVariants[size] || "",
                        colorVariants[color] || "bg-blue-600 dark:bg-blue-500",
                        indeterminateStyles
                    )}
                    style={{
                        width: indeterminate ? "100%" : `${percentage}%`,
                    }}
                />
            </div>
        );
    }
);

Progress.displayName = "Progress";

export { Progress };

// Add CSS for the indeterminate animation to your global CSS or add this style tag to your HTML
// This adds the custom keyframes for the indeterminate animation
document.head.insertAdjacentHTML(
    "beforeend",
    `<style>
    @keyframes indeterminate-progress {
      0% {
        transform: translateX(-100%);
      }
      50% {
        transform: translateX(0%);
      }
      100% {
        transform: translateX(100%);
      }
    }
    .animate-indeterminate-progress {
      animation: indeterminate-progress 1.5s infinite ease-in-out;
    }
  </style>`
);
