import * as React from "react";
import { Check, ChevronDown } from "lucide-react";
import { cn } from "@/lib/utils";

const Select = React.forwardRef(
    (
        {
            children,
            value,
            onChange,
            onValueChange,
            defaultValue,
            disabled,
            ...props
        },
        ref
    ) => {
        const [open, setOpen] = React.useState(false);
        const [selectedValue, setSelectedValue] = React.useState(
            value || defaultValue || ""
        );

        React.useEffect(() => {
            if (value !== undefined) {
                setSelectedValue(value);
            }
        }, [value]);

        const handleSelect = (newValue) => {
            setSelectedValue(newValue);
            setOpen(false);
            if (onChange) {
                // Create a synthetic event object
                const event = {
                    target: {
                        name: props.name,
                        value: newValue,
                    },
                };
                onChange(event);
            }

            // Also support onValueChange for compatibility
            if (onValueChange) {
                onValueChange(newValue);
            }
        };

        return (
            <SelectContext.Provider
                value={{
                    open,
                    setOpen,
                    value: selectedValue,
                    onSelect: handleSelect,
                    disabled,
                }}
            >
                <div ref={ref} className="relative" {...props}>
                    {children}
                </div>
            </SelectContext.Provider>
        );
    }
);

Select.displayName = "Select";

const SelectContext = React.createContext({});

const useSelectContext = () => {
    const context = React.useContext(SelectContext);
    if (!context) {
        throw new Error("Select components must be used within a Select");
    }
    return context;
};

const SelectTrigger = React.forwardRef(
    ({ className, children, ...props }, ref) => {
        const { open, setOpen, value, disabled } = useSelectContext();

        return (
            <button
                type="button"
                ref={ref}
                onClick={() => !disabled && setOpen(!open)}
                className={cn(
                    "flex items-center justify-between w-full rounded-md border border-gray-300 bg-white p-2 text-sm shadow-sm transition-colors",
                    "focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500",
                    "dark:bg-gray-800 dark:border-gray-700 dark:text-white",
                    disabled && "opacity-50 cursor-not-allowed",
                    className
                )}
                aria-expanded={open}
                aria-disabled={disabled}
                {...props}
            >
                {children}
                <ChevronDown className="h-4 w-4 opacity-50" />
            </button>
        );
    }
);

SelectTrigger.displayName = "SelectTrigger";

const SelectValue = React.forwardRef(
    ({ className, placeholder = "Select an option", ...props }, ref) => {
        const { value } = useSelectContext();

        return (
            <span
                ref={ref}
                className={cn("flex-1 text-left truncate", className)}
                {...props}
            >
                {value || placeholder}
            </span>
        );
    }
);

SelectValue.displayName = "SelectValue";

const SelectContent = React.forwardRef(
    ({ className, align = "center", children, ...props }, ref) => {
        const { open, disabled } = useSelectContext();

        if (!open || disabled) return null;

        return (
            <div
                ref={ref}
                className={cn(
                    "absolute z-50 min-w-[8rem] mt-1 overflow-hidden rounded-md border border-gray-300 bg-white shadow-md",
                    "dark:bg-gray-800 dark:border-gray-700 dark:text-white",
                    align === "center" && "left-1/2 -translate-x-1/2",
                    align === "left" && "left-0",
                    align === "right" && "right-0",
                    className
                )}
                {...props}
            >
                <div className="p-1">{children}</div>
            </div>
        );
    }
);

SelectContent.displayName = "SelectContent";

const SelectItem = React.forwardRef(
    ({ className, children, value, ...props }, ref) => {
        const { value: selectedValue, onSelect } = useSelectContext();
        const isSelected = selectedValue === value;

        return (
            <div
                ref={ref}
                className={cn(
                    "relative flex w-full cursor-pointer select-none items-center rounded-sm py-1.5 px-2 text-sm outline-none",
                    "hover:bg-gray-100 hover:text-gray-900 focus:bg-gray-100 focus:text-gray-900",
                    "dark:hover:bg-gray-700 dark:hover:text-white dark:focus:bg-gray-700 dark:focus:text-white",
                    isSelected &&
                        "bg-gray-100 text-gray-900 dark:bg-gray-700 dark:text-white",
                    className
                )}
                onClick={() => onSelect(value)}
                aria-selected={isSelected}
                role="option"
                {...props}
            >
                <span className="flex-1">{children}</span>
                {isSelected && (
                    <Check className="h-4 w-4 ml-2 text-blue-600 dark:text-blue-400" />
                )}
            </div>
        );
    }
);

SelectItem.displayName = "SelectItem";

export { Select, SelectTrigger, SelectValue, SelectContent, SelectItem };
