import React, { useState, useEffect, useRef } from "react";
import { cn } from "@/lib/utils";

// Create context for tooltip state
const TooltipContext = React.createContext({
    open: false,
    setOpen: () => {},
    position: "top",
    arrow: true,
    maxWidth: "15rem",
});

const TooltipProvider = ({ children }) => {
    return <>{children}</>;
};

const Tooltip = ({
    children,
    content,
    position = "top",
    className = "",
    delay = 300,
    arrow = true,
    maxWidth = "15rem",
    asChild = false,
    ...props
}) => {
    // For backward compatibility - simple usage
    if (content) {
        return (
            <TooltipRoot
                content={content}
                position={position}
                className={className}
                delay={delay}
                arrow={arrow}
                maxWidth={maxWidth}
                {...props}
            >
                {children}
            </TooltipRoot>
        );
    }

    // For compound component pattern
    return (
        <TooltipPrimitive
            position={position}
            arrow={arrow}
            maxWidth={maxWidth}
            delay={delay}
            className={className}
            {...props}
        >
            {children}
        </TooltipPrimitive>
    );
};

// Root component used for backward compatibility
const TooltipRoot = ({
    children,
    content,
    position = "top",
    className = "",
    delay = 300,
    arrow = true,
    maxWidth = "15rem",
    ...props
}) => {
    const [open, setOpen] = useState(false);
    const timeoutRef = useRef(null);

    // Cleanup on unmount
    useEffect(() => {
        return () => {
            if (timeoutRef.current) clearTimeout(timeoutRef.current);
        };
    }, []);

    const handleMouseEnter = () => {
        if (timeoutRef.current) clearTimeout(timeoutRef.current);
        timeoutRef.current = setTimeout(() => {
            setOpen(true);
        }, delay);
    };

    const handleMouseLeave = () => {
        if (timeoutRef.current) clearTimeout(timeoutRef.current);
        timeoutRef.current = setTimeout(() => {
            setOpen(false);
        }, 100);
    };

    return (
        <TooltipContext.Provider
            value={{
                open,
                setOpen,
                position,
                arrow,
                maxWidth,
            }}
        >
            <div
                className="relative inline-flex"
                onMouseEnter={handleMouseEnter}
                onMouseLeave={handleMouseLeave}
                onFocus={handleMouseEnter}
                onBlur={handleMouseLeave}
                {...props}
            >
                {children}

                {open && (
                    <TooltipContentImpl className={className}>
                        {content}
                    </TooltipContentImpl>
                )}
            </div>
        </TooltipContext.Provider>
    );
};

// For compound component pattern
const TooltipPrimitive = ({
    children,
    position = "top",
    delay = 300,
    arrow = true,
    maxWidth = "15rem",
    className = "",
    ...props
}) => {
    const [open, setOpen] = useState(false);
    const timeoutRef = useRef(null);

    // Cleanup on unmount
    useEffect(() => {
        return () => {
            if (timeoutRef.current) clearTimeout(timeoutRef.current);
        };
    }, []);

    const handleMouseEnter = () => {
        if (timeoutRef.current) clearTimeout(timeoutRef.current);
        timeoutRef.current = setTimeout(() => {
            setOpen(true);
        }, delay);
    };

    const handleMouseLeave = () => {
        if (timeoutRef.current) clearTimeout(timeoutRef.current);
        timeoutRef.current = setTimeout(() => {
            setOpen(false);
        }, 100);
    };

    // Context value
    const value = {
        open,
        setOpen,
        position,
        arrow,
        maxWidth,
        handleMouseEnter,
        handleMouseLeave,
    };

    return (
        <TooltipContext.Provider value={value}>
            <div className="relative inline-flex" {...props}>
                {children}
            </div>
        </TooltipContext.Provider>
    );
};

const TooltipTrigger = ({ children, asChild = false, ...props }) => {
    const { handleMouseEnter, handleMouseLeave } =
        React.useContext(TooltipContext);

    // If asChild is true, clone the child and add event handlers
    if (asChild && React.isValidElement(children)) {
        return React.cloneElement(children, {
            ...props,
            onMouseEnter: (e) => {
                handleMouseEnter(e);
                children.props.onMouseEnter?.(e);
            },
            onMouseLeave: (e) => {
                handleMouseLeave(e);
                children.props.onMouseLeave?.(e);
            },
            onFocus: (e) => {
                handleMouseEnter(e);
                children.props.onFocus?.(e);
            },
            onBlur: (e) => {
                handleMouseLeave(e);
                children.props.onBlur?.(e);
            },
        });
    }

    // Otherwise, wrap in a span
    return (
        <span
            onMouseEnter={handleMouseEnter}
            onMouseLeave={handleMouseLeave}
            onFocus={handleMouseEnter}
            onBlur={handleMouseLeave}
            {...props}
        >
            {children}
        </span>
    );
};

const TooltipContent = ({ children, className, ...props }) => {
    const { open } = React.useContext(TooltipContext);

    if (!open) return null;

    return (
        <TooltipContentImpl className={className} {...props}>
            {children}
        </TooltipContentImpl>
    );
};

// Internal implementation of tooltip content
const TooltipContentImpl = ({ children, className, ...props }) => {
    const { position, arrow, maxWidth } = React.useContext(TooltipContext);

    // Calculate positioning classes
    const getPositionClasses = () => {
        switch (position) {
            case "top":
                return "bottom-full left-1/2 transform -translate-x-1/2 -translate-y-2 mb-2";
            case "bottom":
                return "top-full left-1/2 transform -translate-x-1/2 translate-y-2 mt-2";
            case "left":
                return "right-full top-1/2 transform -translate-y-1/2 -translate-x-2 mr-2";
            case "right":
                return "left-full top-1/2 transform -translate-y-1/2 translate-x-2 ml-2";
            default:
                return "bottom-full left-1/2 transform -translate-x-1/2 -translate-y-2 mb-2";
        }
    };

    // Calculate arrow classes
    const getArrowClasses = () => {
        if (!arrow) return "";

        switch (position) {
            case "top":
                return "after:content-[''] after:absolute after:left-1/2 after:top-full after:-translate-x-1/2 after:border-8 after:border-transparent after:border-t-slate-700 dark:after:border-t-slate-800";
            case "bottom":
                return "after:content-[''] after:absolute after:left-1/2 after:bottom-full after:-translate-x-1/2 after:border-8 after:border-transparent after:border-b-slate-700 dark:after:border-b-slate-800";
            case "left":
                return "after:content-[''] after:absolute after:top-1/2 after:left-full after:-translate-y-1/2 after:border-8 after:border-transparent after:border-l-slate-700 dark:after:border-l-slate-800";
            case "right":
                return "after:content-[''] after:absolute after:top-1/2 after:right-full after:-translate-y-1/2 after:border-8 after:border-transparent after:border-r-slate-700 dark:after:border-r-slate-800";
            default:
                return "";
        }
    };

    return (
        <div
            role="tooltip"
            className={cn(
                "absolute z-50 px-3 py-2 text-sm font-medium text-white bg-slate-700 dark:bg-slate-800 rounded shadow-md",
                getPositionClasses(),
                getArrowClasses(),
                className
            )}
            style={{ maxWidth }}
            {...props}
        >
            {children}
        </div>
    );
};

export { Tooltip, TooltipProvider, TooltipTrigger, TooltipContent };
export default Tooltip;
