import { useState, useEffect } from "react";
import { Link, usePage } from "@inertiajs/react";
import {
    Moon,
    Sun,
    Menu,
    X,
    Users,
    User,
    LogOut,
    LayoutDashboard,
    Settings,
    ChevronDown,
    Settings2,
    ListOrdered,
    CreditCard,
    BookOpen,
    FileText,
    GraduationCap,
    BarChart,
    Tag,
    Award,
    CalendarRange,
} from "lucide-react";

import { ThemeProvider, useTheme } from "@/Contexts/ThemeContext";

export default function AdminLayout({ children }) {
    return (
        <ThemeProvider>
            <AdminLayoutContent>{children}</AdminLayoutContent>
        </ThemeProvider>
    );
}

function AdminLayoutContent({ children }) {
    const { darkMode, toggleDarkMode } = useTheme();
    const [sidebarOpen, setSidebarOpen] = useState(true);
    const [profileOpen, setProfileOpen] = useState(false);
    const [settingsOpen, setSettingsOpen] = useState(false);
    const [testsOpen, setTestsOpen] = useState(false);
    const [marketingOpen, setMarketingOpen] = useState(false); // New state for marketing dropdown
    const { auth } = usePage().props;

    const isMobile = () => window.innerWidth < 768;

    useEffect(() => {
        const handleResize = () => {
            setSidebarOpen(!isMobile());
        };

        setSidebarOpen(!isMobile());

        window.addEventListener("resize", handleResize);

        return () => window.removeEventListener("resize", handleResize);
    }, []);

    const handleParentMenuClick = (hasSubmenus) => {
        if (isMobile() && hasSubmenus) {
            setSidebarOpen(true);
        }
    };

    return (
        <div
            className={`flex flex-col min-h-screen ${
                darkMode
                    ? "bg-gray-900 text-white"
                    : "bg-gray-100 text-gray-900"
            }`}
        >
            <header className="w-full flex justify-between items-center p-4 bg-gradient-to-r from-cyan-600 to-indigo-800 dark:bg-none  dark:bg-gray-800 shadow-md border-b border-gray-200 dark:border-gray-700">
                <button
                    onClick={() => setSidebarOpen(!sidebarOpen)}
                    className="p-1 hover:bg-gray-200 dark:hover:bg-gray-700 rounded-lg"
                >
                    {sidebarOpen ? <X size={24} /> : <Menu size={24} />}
                </button>
                <div className="text-lg text-gray-50 mx-3 font-bold  md:block">
                    AcademicScribe
                </div>
                <div className="flex items-center gap-4 ml-auto">
                    <button
                        onClick={toggleDarkMode}
                        className="p-1 rounded-full bg-gray-200 dark:bg-gray-700"
                    >
                        {darkMode ? <Sun size={20} /> : <Moon size={20} />}
                    </button>
                    <div className="relative">
                        <button
                            onClick={() => setProfileOpen(!profileOpen)}
                            className="flex items-center text-white gap-2 rounded-full"
                        >
                            {auth.user.user_type === "admin" && (
                                <span className="bg-red-500 text-white text-xs font-bold px-1.5 py-0.5 rounded-full">
                                    AD
                                </span>
                            )}
                            {auth.user?.name?.split(" ")[0] || "User"}

                            <ChevronDown className="h-4 w-4" />
                        </button>
                        {profileOpen && (
                            <div className="absolute right-0 mt-4 p-2 w-48 bg-gradient-to-r from-cyan-600 to-indigo-800 dark:bg-none dark:bg-gray-800 shadow-md rounded z-50">
                                <Link
                                    href="/profile"
                                    className="flex items-center gap-2 p-2 text-white hover:bg-gray-400 dark:hover:bg-gray-700 rounded-lg"
                                >
                                    <User size={20} /> Profile
                                </Link>
                                <Link
                                    href={route("logout")}
                                    method="post"
                                    as="button"
                                    className="flex items-center gap-2 p-2 text-white hover:bg-gray-400 dark:hover:bg-gray-700 w-full text-left rounded-lg"
                                >
                                    <LogOut size={20} /> Logout
                                </Link>
                            </div>
                        )}
                    </div>
                </div>
            </header>

            <div className="flex flex-1">
                <aside
                    className={`fixed md:relative transition-all duration-300 h-[calc(100vh-4rem)] ${
                        sidebarOpen
                            ? "translate-x-0 w-64"
                            : "-translate-x-full md:translate-x-0 md:w-16"
                    } bg-gradient-to-r from-indigo-800 to-cyan-600 dark:bg-none text-gray-50 rounded-lg dark:bg-gray-800 p-4 dark:text-white z-20`}
                >
                    <nav className="flex flex-col gap-4">
                        <Link
                            href="/admin/dashboard"
                            className="flex items-center gap-4 p-2  hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <LayoutDashboard size={24} />
                            {sidebarOpen && <span>Dashboard</span>}
                        </Link>

                        <Link
                            href="/admin/orders"
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <ListOrdered size={24} />
                            {sidebarOpen && <span>Orders</span>}
                        </Link>
                        <Link
                            href="/admin/users"
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <Users size={24} />
                            {sidebarOpen && <span>Users</span>}
                        </Link>

                        {/* New Marketing & Loyalty Dropdown */}
                        <div>
                            <button
                                onClick={() => {
                                    setMarketingOpen(!marketingOpen);
                                    handleParentMenuClick(true);
                                }}
                                className="w-full flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            >
                                <Tag size={24} />
                                {sidebarOpen && (
                                    <div className="flex items-center justify-between flex-1 text-md font-bold">
                                        <span>Marketing</span>
                                        <ChevronDown
                                            size={16}
                                            className={`transition-transform ${
                                                marketingOpen
                                                    ? "rotate-180"
                                                    : ""
                                            }`}
                                        />
                                    </div>
                                )}
                            </button>

                            {marketingOpen && sidebarOpen && (
                                <div className="ml-8 mt-2 flex flex-col gap-2 text-sm">
                                    <Link
                                        href={route("admin.coupons.index")}
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <Tag size={18} />
                                        <span>Coupons</span>
                                    </Link>
                                    <Link
                                        href={route(
                                            "admin.loyalty-tiers.index"
                                        )}
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <Award size={18} />
                                        <span>Loyalty Tiers</span>
                                    </Link>
                                    <Link
                                        href="/admin/seasonal-types"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <CalendarRange size={18} />
                                        <span>Seasonal Types</span>
                                    </Link>
                                    <Link
                                        href={route("admin.coupons.statistics")}
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <BarChart size={18} />
                                        <span>Coupon Statistics</span>
                                    </Link>
                                </div>
                            )}
                        </div>

                        {/* Payment Methods Link */}
                        <Link
                            href="/admin/payment-methods"
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <CreditCard size={24} />
                            {sidebarOpen && <span>Payment Methods</span>}
                        </Link>

                        <Link
                            href={route("admin.payment.analytics")}
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <BarChart size={24} />
                            {sidebarOpen && <span>Payment Analytics</span>}
                        </Link>

                        <Link
                            href={route("admin.payments.history")}
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <CreditCard size={24} />
                            {sidebarOpen && <span>Payment History</span>}
                        </Link>

                        {/* Tests and Essays Dropdown */}
                        <div>
                            <button
                                onClick={() => {
                                    setTestsOpen(!testsOpen);
                                    handleParentMenuClick(true);
                                }}
                                className="w-full flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            >
                                <GraduationCap size={24} />
                                {sidebarOpen && (
                                    <div className="flex items-center justify-between flex-1 text-md font-bold">
                                        <span>Tests</span>
                                        <ChevronDown
                                            size={16}
                                            className={`transition-transform ${
                                                testsOpen ? "rotate-180" : ""
                                            }`}
                                        />
                                    </div>
                                )}
                            </button>

                            {testsOpen && sidebarOpen && (
                                <div className="ml-8 mt-2 flex flex-col gap-2 text-sm">
                                    <Link
                                        href="/admin/test-questions"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <FileText size={18} />
                                        <span>Test Questions</span>
                                    </Link>
                                    <Link
                                        href="/admin/essay-topics"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <BookOpen size={18} />
                                        <span>Essay Topics</span>
                                    </Link>
                                    <Link
                                        href="/admin/essays"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <FileText size={18} />
                                        <span>Essays</span>
                                    </Link>
                                    <Link
                                        href="/admin/writers/tests"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <Users size={18} />
                                        <span>Writer Tests</span>
                                    </Link>
                                </div>
                            )}
                        </div>

                        <div>
                            <button
                                onClick={() => {
                                    setSettingsOpen(!settingsOpen);
                                    handleParentMenuClick(true);
                                }}
                                className="w-full flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            >
                                <Settings2 size={24} />
                                {sidebarOpen && (
                                    <div className="flex items-center justify-between flex-1 text-md font-bold">
                                        <span>Preferences</span>
                                        <ChevronDown
                                            size={16}
                                            className={`transition-transform ${
                                                settingsOpen ? "rotate-180" : ""
                                            }`}
                                        />
                                    </div>
                                )}
                            </button>

                            {settingsOpen && sidebarOpen && (
                                <div className="ml-8 mt-2 flex flex-col gap-2 text-sm">
                                    <Link
                                        href="/academic-levels"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <span>Academic Levels</span>
                                    </Link>
                                    <Link
                                        href="/subjects"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <span>Subjects</span>
                                    </Link>
                                    <Link
                                        href="/paper-types"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <span>Paper Types</span>
                                    </Link>
                                    <Link
                                        href="/pricings"
                                        className="flex items-center gap-2 p-2 hover:bg-gray-700 rounded"
                                    >
                                        <span>Urgencies</span>
                                    </Link>
                                </div>
                            )}
                        </div>

                        <Link
                            href="/admin/settings/1"
                            className="flex items-center gap-4 p-2 hover:bg-gray-700 rounded"
                            onClick={() => handleParentMenuClick(false)}
                        >
                            <Settings size={24} />
                            {sidebarOpen && <span>Settings</span>}
                        </Link>
                    </nav>
                </aside>

                <main className="flex-1 p-6 bg-gradient-to-b from-blue-100 to-white dark:bg-none">
                    {children}
                </main>
            </div>
        </div>
    );
}
