import { useState, useEffect, useRef } from "react";
import { Link, usePage, router } from "@inertiajs/react";
import {
    LayoutDashboard,
    ListOrdered,
    FileText,
    User,
    LogOut,
    Menu,
    X,
    Search,
    ChevronDown,
    ChevronRight,
    Settings,
    Bell,
    Sun,
    Moon,
    Clock,
    CreditCard,
    HelpCircle,
    MessageSquare,
    Gavel,
} from "lucide-react";
import { ThemeProvider, useTheme } from "@/Contexts/ThemeContext";
import Tooltip from "@/Components/ui/Tooltip";
import NotificationBell from "@/Components/Notifications/NotificationBell";
import MessagingWidget from "@/Components/Messaging/MessagingWidget";

export default function ClientLayout({ children }) {
    return (
        <ThemeProvider>
            <ClientLayoutContent>{children}</ClientLayoutContent>
        </ThemeProvider>
    );
}

function ClientLayoutContent({ children }) {
    const { darkMode, toggleDarkMode } = useTheme();
    const user = usePage().props.auth.user;
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [profileOpen, setProfileOpen] = useState(false);
    const [projectsOpen, setProjectsOpen] = useState(false);
    const [supportOpen, setSupportOpen] = useState(false);
    const [ordersOpen, setOrdersOpen] = useState(false);
    const [searchQuery, setSearchQuery] = useState("");
    const sidebarRef = useRef(null);

    // Logout handler - Inertia handles CSRF automatically via cookie
    const handleLogout = () => {
        router.post(route('logout'));
    };

    const isMobile = () => window.innerWidth < 768;

    useEffect(() => {
        const handleResize = () => {
            setSidebarOpen(!isMobile());
        };

        setSidebarOpen(!isMobile());

        window.addEventListener("resize", handleResize);

        return () => window.removeEventListener("resize", handleResize);
    }, []);

    const handleParentMenuClick = (hasSubmenus, e) => {
        if (!sidebarOpen && hasSubmenus) {
            setSidebarOpen(true);
            e.preventDefault();
        }
    };

    // Filter menu items based on search query
    const filterMenuItems = (items) => {
        if (!searchQuery) return true;
        return items.toLowerCase().includes(searchQuery.toLowerCase());
    };

    return (
        <div
            className={`flex flex-col min-h-screen ${
                darkMode
                    ? "bg-gray-900 text-white"
                    : "bg-gray-100 text-gray-900"
            }`}
        >
            <header className="w-full flex justify-between items-center p-4 bg-gradient-to-r from-cyan-600 to-indigo-800 dark:bg-none dark:bg-gray-800 shadow-md border-b border-gray-200 dark:border-gray-700 sticky top-0 z-30">
                {/* Header content remains the same */}
                <button
                    onClick={() => setSidebarOpen(!sidebarOpen)}
                    className="p-1 hover:bg-gray-200/20 dark:hover:bg-gray-700 rounded-lg text-white transition-colors duration-200"
                    aria-label={sidebarOpen ? "Close sidebar" : "Open sidebar"}
                >
                    {sidebarOpen ? <X size={24} /> : <Menu size={24} />}
                </button>
                <div className="text-lg text-gray-50 mx-3 font-bold md:block">
                    <Link href="/">AcademicScribe</Link>
                </div>
                <div className="flex items-center gap-4 ml-auto">
                    <button
                        onClick={toggleDarkMode}
                        className="p-2 rounded-full bg-gray-200/20 dark:bg-gray-700 hover:bg-gray-200/30 dark:hover:bg-gray-600 transition-colors duration-200 text-white"
                        aria-label={
                            darkMode
                                ? "Switch to light mode"
                                : "Switch to dark mode"
                        }
                    >
                        {darkMode ? <Sun size={20} /> : <Moon size={20} />}
                    </button>

                    <NotificationBell />

                    <div className="relative">
                        <button
                            onClick={() => setProfileOpen(!profileOpen)}
                            className="flex items-center text-white gap-2 rounded-full hover:bg-gray-200/20 dark:hover:bg-gray-700 px-3 py-1.5 transition-colors duration-200"
                        >
                            {user.user_type === "client" && (
                                <span className="bg-blue-500 text-white text-xs font-bold px-1.5 py-0.5 rounded-full">
                                    CL
                                </span>
                            )}
                            <span className="hidden sm:block">
                                {user?.name?.split(" ")[0] || "User"}
                            </span>

                            <ChevronDown className="h-4 w-4" />
                        </button>
                        {profileOpen && (
                            <div className="absolute right-0 mt-2 p-2 w-48 bg-white dark:bg-gray-800 shadow-lg rounded-lg z-50 border border-gray-200 dark:border-gray-700 animate-in fade-in duration-200">
                                {/* Profile dropdown content remains the same */}
                                <div className="px-3 py-2 border-b border-gray-200 dark:border-gray-700 mb-1">
                                    <p className="text-sm font-medium text-gray-900 dark:text-white">
                                        {user?.name}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400 truncate">
                                        {user?.email}
                                    </p>
                                </div>
                                <Link
                                    href="/profile"
                                    className="flex items-center gap-2 p-2 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors duration-200"
                                    onClick={() => setProfileOpen(false)}
                                >
                                    <User size={18} /> Profile
                                </Link>
                                <button
                                    onClick={handleLogout}
                                    className="flex items-center gap-2 p-2 text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 w-full text-left rounded-md transition-colors duration-200"
                                >
                                    <LogOut size={18} /> Logout
                                </button>
                            </div>
                        )}
                    </div>
                </div>
            </header>

            <div className="flex flex-1">
                <aside
                    ref={sidebarRef}
                    className={`fixed md:relative transition-all duration-300 h-[calc(100vh-4rem)] ${
                        sidebarOpen
                            ? "translate-x-0 w-64"
                            : "-translate-x-full md:translate-x-0 md:w-16"
                    } bg-gradient-to-r from-indigo-800 to-cyan-600 dark:bg-none text-gray-50 rounded-lg dark:bg-gray-800 shadow-xl border-r border-gray-200 dark:border-gray-700 z-20 overflow-hidden`}
                >
                    <div className="flex flex-col h-full">
                        {/* Search box - visible only when sidebar is open */}
                        {sidebarOpen && (
                            <div className="px-4 py-3 border-b border-indigo-700/50 dark:border-gray-700">
                                <div className="relative">
                                    <Search className="absolute left-2 top-2.5 h-4 w-4 text-gray-400" />
                                    <input
                                        type="text"
                                        placeholder="Search menu..."
                                        className="w-full pl-8 pr-3 py-2 bg-indigo-900/30 dark:bg-gray-700 border border-indigo-700/50 dark:border-gray-600 rounded-md text-sm text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-cyan-500 focus:border-transparent"
                                        value={searchQuery}
                                        onChange={(e) =>
                                            setSearchQuery(e.target.value)
                                        }
                                    />
                                </div>
                            </div>
                        )}

                        {/* Scrollable menu container */}
                        <nav className="flex-1 overflow-y-auto overflow-x-hidden py-4 px-3 flex flex-col space-y-1.5 scrollbar-thin scrollbar-thumb-indigo-700 dark:scrollbar-thumb-gray-700">
                            {/* Dashboard Link */}
                            {filterMenuItems("Dashboard") && (
                                <Tooltip
                                    content="Dashboard"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <Link
                                        href="/dashboard"
                                        className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("dashboard")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <LayoutDashboard size={20} />
                                        {sidebarOpen && <span>Dashboard</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Orders Dropdown - Modified to include submenu */}
                            {filterMenuItems("Orders") && (
                                <Tooltip
                                    content="My Orders"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setOrdersOpen(!ordersOpen);
                                                handleParentMenuClick(
                                                    true,
                                                    event
                                                );
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                route().current("orders.*") ||
                                                route().current("client.bids.*")
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <ListOrdered size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>My Orders</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            ordersOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                                        {ordersOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                <Link
                                                    href="/orders"
                                                    className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                        route().current(
                                                            "orders.index"
                                                        )
                                                            ? "bg-white/10 dark:bg-gray-600"
                                                            : ""
                                                    }`}
                                                >
                                                    <ListOrdered size={16} />
                                                    <span>Orders List</span>
                                                </Link>
                                                <Link
                                                    href="/client/bids"
                                                    className={`flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                        route().current(
                                                            "client.bids.index"
                                                        )
                                                            ? "bg-white/10 dark:bg-gray-600"
                                                            : ""
                                                    }`}
                                                >
                                                    <Gavel size={16} />
                                                    <span>My Bids</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Projects Dropdown */}
                            {filterMenuItems("Projects") && (
                                <Tooltip
                                    content="Projects"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setProjectsOpen(!projectsOpen);
                                                handleParentMenuClick(
                                                    true,
                                                    event
                                                );
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                                projectsOpen
                                                    ? "bg-white/20 dark:bg-gray-700"
                                                    : ""
                                            }`}
                                        >
                                            <FileText size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Projects</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            projectsOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                                        {projectsOpen && sidebarOpen && (
                                            <div className="ml-6 mt-1 pl-2 border-l border-indigo-500/50 dark:border-gray-600 space-y-1 animate-in slide-in-from-left duration-200">
                                                <Link
                                                    href={route('projects.active')}
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <Clock size={16} />
                                                    <span>Active Projects</span>
                                                </Link>
                                                <Link
                                                    href={route('projects.completed')}
                                                    className="flex items-center gap-2 p-2 text-sm hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200"
                                                >
                                                    <FileText size={16} />
                                                    <span>Completed Work</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Settings Link */}
                            {filterMenuItems("Settings") && (
                                <Tooltip
                                    content="Settings"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href={route("settings.index")}
                                        className={`flex items-center gap-4 p-2.5 hover:bg-white/10 dark:hover:bg-gray-700 rounded-md text-white transition-colors duration-200 ${
                                            route().current("settings.*")
                                                ? "bg-white/20 dark:bg-gray-700"
                                                : ""
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <Settings size={20} />
                                        {sidebarOpen && <span>Settings</span>}
                                    </Link>
                                </Tooltip>
                            )}
                        </nav>

                        {/* Sidebar footer */}
                        {sidebarOpen && (
                            <div className="p-3 mt-auto border-t border-indigo-700/50 dark:border-gray-700">
                                <div className="flex items-center justify-between text-xs text-indigo-200 dark:text-gray-400">
                                    <span>v1.0.0</span>
                                    <button
                                        type="button"
                                        className="hover:text-white transition-colors duration-200 text-xs"
                                        onClick={() => setSidebarOpen(false)}
                                    >
                                        Collapse
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>
                </aside>

                <main className="flex-1 p-6 bg-gradient-to-b from-blue-50 to-white dark:from-gray-900 dark:to-gray-800 dark:bg-none overflow-auto min-h-[calc(100vh-4rem)]">
                    <div className="max-w-7xl mx-auto">{children}</div>
                </main>
            </div>
            <MessagingWidget />
        </div>
    );
}
