import { useState, useEffect, useRef } from "react";
import { Link, usePage, router } from "@inertiajs/react";
import {
    LayoutDashboard,
    FileText,
    User,
    LogOut,
    Menu,
    X,
    Search,
    ChevronDown,
    ChevronRight,
    Settings,
    ListOrdered,
    Wallet,
    BookOpen,
    PenTool,
    Users,
    Bell,
    Sun,
    Moon,
    Clock,
    Award,
    MessageSquare,
    HelpCircle,
} from "lucide-react";
import { ThemeProvider, useTheme } from "@/Contexts/ThemeContext";
import Tooltip from "@/Components/ui/Tooltip";
import NotificationBell from "@/Components/Notifications/NotificationBell";
import MessagingWidget from "@/Components/Messaging/MessagingWidget";
import WriterPaymentStatus from "@/Components/Writer/WriterPaymentStatus";

export default function WriterLayout({ children }) {
    return (
        <ThemeProvider>
            <WriterLayoutContent>{children}</WriterLayoutContent>
        </ThemeProvider>
    );
}

function WriterLayoutContent({ children }) {
    const { darkMode, toggleDarkMode } = useTheme();
    const user = usePage().props.auth.user;
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [profileOpen, setProfileOpen] = useState(false);
    const [ordersOpen, setOrdersOpen] = useState(false);
    const [earningsOpen, setEarningsOpen] = useState(false);
    const [resourcesOpen, setResourcesOpen] = useState(false);
    const [searchQuery, setSearchQuery] = useState("");
    const sidebarRef = useRef(null);

    // Logout handler - Inertia handles CSRF automatically via cookie
    const handleLogout = () => {
        router.post(route('logout'));
    };

    const isMobile = () => window.innerWidth < 768;

    useEffect(() => {
        const handleResize = () => {
            setSidebarOpen(!isMobile());
        };

        setSidebarOpen(!isMobile());

        window.addEventListener("resize", handleResize);

        return () => window.removeEventListener("resize", handleResize);
    }, []);

    const handleParentMenuClick = (hasSubmenus, e) => {
        if (!sidebarOpen && hasSubmenus) {
            setSidebarOpen(true);
            e.preventDefault();
        }
    };

    // Filter menu items based on search query
    const filterMenuItems = (items) => {
        if (!searchQuery) return true;
        return items.toLowerCase().includes(searchQuery.toLowerCase());
    };

    return (
        <div
            className={`flex flex-col min-h-screen ${
                darkMode
                    ? "bg-gray-900 text-white"
                    : "bg-gray-100 text-gray-900"
            }`}
        >
            {/* Header - Mobile-responsive with stacked layout */}
            <header
                className={`w-full ${
                    darkMode
                        ? "bg-gray-800 border-gray-700"
                        : "bg-gradient-to-tr from-green-400 via-green-600 to-green-700 border-gray-200"
                } shadow-md border-b sticky top-0 z-30`}
            >
                {/* Main Header Row */}
                <div className="flex justify-between items-center p-3 sm:p-4">
                    <div className="flex items-center gap-2 sm:gap-3">
                        <button
                            onClick={() => setSidebarOpen(!sidebarOpen)}
                            className={`p-1 rounded-lg transition-colors duration-200 ${
                                darkMode
                                    ? "text-white hover:bg-gray-700"
                                    : "text-white hover:bg-green-600"
                            }`}
                            aria-label={sidebarOpen ? "Close sidebar" : "Open sidebar"}
                        >
                            {sidebarOpen ? <X size={20} className="sm:hidden" /> : <Menu size={20} className="sm:hidden" />}
                            {sidebarOpen ? <X size={24} className="hidden sm:block" /> : <Menu size={24} className="hidden sm:block" />}
                        </button>
                        <div className="text-base sm:text-lg text-white font-bold">
                            <Link href="/writer/dashboard">AcademicScribe</Link>
                        </div>
                    </div>

                    <div className="flex items-center gap-1 sm:gap-2 md:gap-3">
                        <button
                            onClick={toggleDarkMode}
                            className={`p-1.5 sm:p-2 rounded-full transition-colors duration-200 ${
                                darkMode
                                    ? "bg-gray-700 text-white hover:bg-gray-600"
                                    : "bg-green-600/40 text-white hover:bg-green-600/60"
                            }`}
                            aria-label={
                                darkMode
                                    ? "Switch to light mode"
                                    : "Switch to dark mode"
                            }
                        >
                            {darkMode ? <Sun size={16} className="sm:hidden" /> : <Moon size={16} className="sm:hidden" />}
                            {darkMode ? <Sun size={20} className="hidden sm:block" /> : <Moon size={20} className="hidden sm:block" />}
                        </button>

                        <NotificationBell />
                        
                        {/* Payment Status - Hidden on mobile, shown in separate row */}
                        <div className="hidden sm:block">
                            <WriterPaymentStatus />
                        </div>

                            <div className="relative">
                                <button
                                    onClick={() => setProfileOpen(!profileOpen)}
                                    className={`flex items-center gap-1 sm:gap-2 rounded-full px-2 sm:px-3 py-1 sm:py-1.5 transition-colors duration-200 ${
                                        darkMode
                                            ? "text-white hover:bg-gray-700"
                                            : "text-white hover:bg-green-600/60"
                                    }`}
                                >
                                    {user.user_type === "writer" && (
                                        <span className="bg-red-500 text-white text-xs font-bold px-1 sm:px-1.5 py-0.5 rounded-full">
                                            WR
                                        </span>
                                    )}
                                    <span className="hidden sm:block">
                                        {user?.name?.split(" ")[0] || "User"}
                                    </span>

                                    <ChevronDown className="h-3 w-3 sm:h-4 sm:w-4" />
                                </button>
                        {profileOpen && (
                            <div
                                className={`absolute right-0 mt-2 p-2 w-48 shadow-lg rounded-lg z-50 border animate-in fade-in duration-200 ${
                                    darkMode
                                        ? "bg-gray-800 border-gray-700 text-gray-200"
                                        : "bg-white border-gray-200 text-gray-700"
                                }`}
                            >
                                <div
                                    className={`px-3 py-2 border-b mb-1 ${
                                        darkMode
                                            ? "border-gray-700"
                                            : "border-gray-200"
                                    }`}
                                >
                                    <p
                                        className={`text-sm font-medium ${
                                            darkMode
                                                ? "text-white"
                                                : "text-gray-900"
                                        }`}
                                    >
                                        {user?.name}
                                    </p>
                                    <p
                                        className={`text-xs truncate ${
                                            darkMode
                                                ? "text-gray-400"
                                                : "text-gray-500"
                                        }`}
                                    >
                                        {user?.email}
                                    </p>
                                </div>
                                <Link
                                    href="/writer/profile"
                                    className={`flex items-center gap-2 p-2 rounded-md transition-colors duration-200 ${
                                        darkMode
                                            ? "hover:bg-gray-700"
                                            : "hover:bg-gray-100"
                                    }`}
                                    onClick={() => setProfileOpen(false)}
                                >
                                    <User size={18} /> Profile
                                </Link>
                                <Link
                                    href={route("logout")}
                                    method="post"
                                    as="button"
                                    className={`flex items-center gap-2 p-2 w-full text-left rounded-md transition-colors duration-200 ${
                                        darkMode
                                            ? "hover:bg-gray-700"
                                            : "hover:bg-gray-100"
                                    }`}
                                    onClick={handleLogout}
                                >
                                    <LogOut size={18} /> Logout
                                </Link>
                            </div>
                        )}
                            </div>
                        </div>
                    </div>

                {/* Mobile Payment Status Row - Only visible on mobile */}
                <div className={`sm:hidden border-t px-3 py-2 ${
                    darkMode 
                        ? "border-gray-600/50" 
                        : "border-green-500/20"
                }`}>
                    <div className="flex justify-center">
                        <WriterPaymentStatus />
                    </div>
                </div>
            </header>

            <div className="flex flex-1">
                {/* Sidebar - Properly handles light/dark mode */}
                <aside
                    ref={sidebarRef}
                    className={`fixed md:relative transition-all duration-300 h-[calc(100vh-4rem)] ${
                        sidebarOpen
                            ? "translate-x-0 w-64"
                            : "-translate-x-full md:translate-x-0 md:w-16"
                    } ${
                        darkMode
                            ? "bg-gray-800 border-gray-700 text-gray-50"
                            : "bg-gradient-to-r from-green-700 to-green-600 border-gray-200 text-white"
                    } rounded-lg shadow-xl border-r z-20 overflow-hidden`}
                >
                    <div className="flex flex-col h-full">
                        {/* Search box - visible only when sidebar is open */}
                        {sidebarOpen && (
                            <div
                                className={`px-4 py-3 border-b ${
                                    darkMode
                                        ? "border-gray-700"
                                        : "border-green-600/50"
                                }`}
                            >
                                <div className="relative">
                                    <Search className="absolute left-2 top-2.5 h-4 w-4 text-gray-400" />
                                    <input
                                        type="text"
                                        placeholder="Search menu..."
                                        className={`w-full pl-8 pr-3 py-2 rounded-md text-sm text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:border-transparent ${
                                            darkMode
                                                ? "bg-gray-700 border-gray-600 focus:ring-blue-500"
                                                : "bg-green-800/40 border-green-600/50 focus:ring-green-500"
                                        } border`}
                                        value={searchQuery}
                                        onChange={(e) =>
                                            setSearchQuery(e.target.value)
                                        }
                                    />
                                </div>
                            </div>
                        )}

                        {/* Scrollable menu container */}
                        <nav
                            className={`flex-1 overflow-y-auto overflow-x-hidden py-4 px-3 flex flex-col space-y-1.5 ${
                                darkMode
                                    ? "scrollbar-thin scrollbar-thumb-gray-700"
                                    : "scrollbar-thin scrollbar-thumb-green-600/50"
                            }`}
                        >
                            {/* Dashboard Link */}
                            {filterMenuItems("Dashboard") && (
                                <Tooltip
                                    content="Dashboard"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <Link
                                        href="/writer/dashboard"
                                        className={`flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                            route().current("writer.dashboard")
                                                ? darkMode
                                                    ? "bg-gray-700"
                                                    : "bg-green-600/50"
                                                : ""
                                        } ${
                                            darkMode
                                                ? "hover:bg-gray-700"
                                                : "hover:bg-green-600/30"
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <LayoutDashboard size={20} />
                                        {sidebarOpen && <span>Dashboard</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Orders Dropdown */}
                            {filterMenuItems("Orders") && (
                                <Tooltip
                                    content="Orders"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setOrdersOpen(!ordersOpen);
                                                handleParentMenuClick(
                                                    true,
                                                    event
                                                );
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                                ordersOpen
                                                    ? darkMode
                                                        ? "bg-gray-700"
                                                        : "bg-green-600/50"
                                                    : ""
                                            } ${
                                                darkMode
                                                    ? "hover:bg-gray-700"
                                                    : "hover:bg-green-600/30"
                                            }`}
                                        >
                                            <ListOrdered size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Orders</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            ordersOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                                        {ordersOpen && sidebarOpen && (
                                            <div
                                                className={`ml-6 mt-1 pl-2 space-y-1 animate-in slide-in-from-left duration-200 border-l ${
                                                    darkMode
                                                        ? "border-gray-600"
                                                        : "border-green-500/50"
                                                }`}
                                            >
                                                <Link
                                                    href="/writer/orders"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <ListOrdered size={16} />
                                                    <span>
                                                        Available Orders
                                                    </span>
                                                </Link>
                                                <Link
                                                    href="/writer/bids"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <FileText size={16} />
                                                    <span>My Bids</span>
                                                </Link>
                                                <Link
                                                    href="/writer/assignments"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <PenTool size={16} />
                                                    <span>My Assignments</span>
                                                </Link>
                                                <Link
                                                    href="/writer/order-history"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <FileText size={16} />
                                                    <span>Order History</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Earnings & Payments */}
                            {filterMenuItems("Earnings") && (
                                <Tooltip
                                    content="Earnings"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setEarningsOpen(!earningsOpen);
                                                handleParentMenuClick(
                                                    true,
                                                    event
                                                );
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                                earningsOpen
                                                    ? darkMode
                                                        ? "bg-gray-700"
                                                        : "bg-green-600/50"
                                                    : ""
                                            } ${
                                                darkMode
                                                    ? "hover:bg-gray-700"
                                                    : "hover:bg-green-600/30"
                                            }`}
                                        >
                                            <Wallet size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Earnings</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            earningsOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                                        {earningsOpen && sidebarOpen && (
                                            <div
                                                className={`ml-6 mt-1 pl-2 space-y-1 animate-in slide-in-from-left duration-200 border-l ${
                                                    darkMode
                                                        ? "border-gray-600"
                                                        : "border-green-500/50"
                                                }`}
                                            >
                                                <Link
                                                    href="/writer/earnings"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <Wallet size={16} />
                                                    <span>
                                                        Earnings Overview
                                                    </span>
                                                </Link>
                                                <Link
                                                    href="/writer/payments"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <Clock size={16} />
                                                    <span>Payment History</span>
                                                </Link>
                                                <Link
                                                    href="/writer/payment-methods"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <Settings size={16} />
                                                    <span>Payment Methods</span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Resources & Development */}
                            {filterMenuItems("Resources") && (
                                <Tooltip
                                    content="Resources"
                                    position={sidebarOpen ? "right" : "right"}
                                    className="whitespace-nowrap"
                                >
                                    <div>
                                        <button
                                            onClick={() => {
                                                setResourcesOpen(!resourcesOpen);
                                                handleParentMenuClick(
                                                    true,
                                                    event
                                                );
                                            }}
                                            className={`w-full flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                                resourcesOpen
                                                    ? darkMode
                                                        ? "bg-gray-700"
                                                        : "bg-green-600/50"
                                                    : ""
                                            } ${
                                                darkMode
                                                    ? "hover:bg-gray-700"
                                                    : "hover:bg-green-600/30"
                                            }`}
                                        >
                                            <BookOpen size={20} />
                                            {sidebarOpen && (
                                                <div className="flex items-center justify-between flex-1 text-md font-medium">
                                                    <span>Resources</span>
                                                    <ChevronDown
                                                        size={16}
                                                        className={`transition-transform duration-200 ${
                                                            resourcesOpen
                                                                ? "rotate-180"
                                                                : ""
                                                        }`}
                                                    />
                                                </div>
                                            )}
                                        </button>

                                        {resourcesOpen && sidebarOpen && (
                                            <div
                                                className={`ml-6 mt-1 pl-2 space-y-1 animate-in slide-in-from-left duration-200 border-l ${
                                                    darkMode
                                                        ? "border-gray-600"
                                                        : "border-green-500/50"
                                                }`}
                                            >
                                                <Link
                                                    href="/writer/resources/guides"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <BookOpen size={16} />
                                                    <span>Writer Guides</span>
                                                </Link>
                                                <Link
                                                    href="/writer/resources/training"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <Award size={16} />
                                                    <span>
                                                        Training Materials
                                                    </span>
                                                </Link>
                                                <Link
                                                    href="/writer/community"
                                                    className={`flex items-center gap-2 p-2 text-sm rounded-md text-white transition-colors duration-200 ${
                                                        darkMode
                                                            ? "hover:bg-gray-700"
                                                            : "hover:bg-green-600/30"
                                                    }`}
                                                >
                                                    <Users size={16} />
                                                    <span>
                                                        Writer Community
                                                    </span>
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </Tooltip>
                            )}

                            {/* Support Link */}
                            {filterMenuItems("Support") && (
                                <Tooltip
                                    content="Support"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/writer/support"
                                        className={`flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                            route().current("writer.support")
                                                ? darkMode
                                                    ? "bg-gray-700"
                                                    : "bg-green-600/50"
                                                : ""
                                        } ${
                                            darkMode
                                                ? "hover:bg-gray-700"
                                                : "hover:bg-green-600/30"
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <MessageSquare size={20} />
                                        {sidebarOpen && <span>Support</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Profile Link */}
                            {filterMenuItems("Profile") && (
                                <Tooltip
                                    content="Profile"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/writer/profile"
                                        className={`flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                            route().current("writer.settings")
                                                ? darkMode
                                                    ? "bg-gray-700"
                                                    : "bg-green-600/50"
                                                : ""
                                        } ${
                                            darkMode
                                                ? "hover:bg-gray-700"
                                                : "hover:bg-green-600/30"
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <Settings size={20} />
                                        {sidebarOpen && <span>Profile</span>}
                                    </Link>
                                </Tooltip>
                            )}

                            {/* Settings Link */}
                            {filterMenuItems("Settings") && (
                                <Tooltip
                                    content="Settings"
                                    position={sidebarOpen ? "right" : "right"}
                                >
                                    <Link
                                        href="/writer/settings"
                                        className={`flex items-center gap-4 p-2.5 rounded-md text-white transition-colors duration-200 ${
                                            route().current("writer.settings")
                                                ? darkMode
                                                    ? "bg-gray-700"
                                                    : "bg-green-600/50"
                                                : ""
                                        } ${
                                            darkMode
                                                ? "hover:bg-gray-700"
                                                : "hover:bg-green-600/30"
                                        }`}
                                        onClick={(e) =>
                                            handleParentMenuClick(false, e)
                                        }
                                    >
                                        <User size={18} />
                                        {sidebarOpen && <span>Settings</span>}
                                    </Link>
                                </Tooltip>
                            )}
                        </nav>

                        {/* Sidebar footer */}
                        {sidebarOpen && (
                            <div
                                className={`p-3 mt-auto border-t ${
                                    darkMode
                                        ? "border-gray-700 text-gray-400"
                                        : "border-green-600/50 text-green-200"
                                }`}
                            >
                                <div className="flex items-center justify-between text-xs">
                                    <span>v1.0.0</span>
                                    <button
                                        type="button"
                                        className={`text-xs ${
                                            darkMode
                                                ? "hover:text-white"
                                                : "hover:text-white"
                                        } transition-colors duration-200`}
                                        onClick={() => setSidebarOpen(false)}
                                    >
                                        Collapse
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>
                </aside>

                <main
                    className={`flex-1 p-6 overflow-auto min-h-[calc(100vh-4rem)] ${
                        darkMode
                            ? "bg-gradient-to-b from-gray-900 to-gray-800"
                            : "bg-gradient-to-b from-blue-50 to-white"
                    }`}
                >
                    <div className="max-w-7xl mx-auto">{children}</div>
                </main>
            </div>
            <MessagingWidget />
        </div>
    );
}
