import React, { useState, useCallback, useMemo } from "react";
import { Head, Link, router, useForm } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Plus,
    Edit,
    Trash2,
    Search,
    Filter,
    MoreHorizontal,
    X,
    Check,
    AlertCircle,
    User,
    Users,
    Mail,
    Globe,
    Github,
    Twitter,
    Linkedin,
    Facebook,
    Instagram,
    Youtube,
    Link as LinkIcon,
    ArrowUpDown,
    ArrowUp,
    ArrowDown,
    CheckCircle,
    XCircle,
    Loader2,
    FileText,
    Eye,
    Calendar,
    UserPlus,
    Upload,
    Camera,
    Image,
    Save,
    RefreshCw,
    TrendingUp,
    BarChart3,
    Award,
    MessageCircle,
    Heart,
    Share2,
    ExternalLink,
    Clock,
    Activity,
    Zap,
    Target,
    Star,
    BookOpen,
    PenTool,
    Settings,
    Shield,
    Crown,
    Badge,
    Verified,
    UserCheck,
    UserX,
    Copy,
    Download,
    Filter as FilterIcon,
    SortAsc,
    SortDesc,
    Grid,
    List,
    ChevronDown,
    ChevronUp,
    Maximize2,
    Minimize2,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";

// Social platform configurations
const SOCIAL_PLATFORMS = {
    website: {
        name: "Website",
        icon: Globe,
        color: "#6B7280",
        placeholder: "https://example.com",
    },
    twitter: {
        name: "Twitter",
        icon: Twitter,
        color: "#1DA1F2",
        placeholder: "https://twitter.com/username",
    },
    linkedin: {
        name: "LinkedIn",
        icon: Linkedin,
        color: "#0077B5",
        placeholder: "https://linkedin.com/in/username",
    },
    github: {
        name: "GitHub",
        icon: Github,
        color: "#333333",
        placeholder: "https://github.com/username",
    },
    facebook: {
        name: "Facebook",
        icon: Facebook,
        color: "#1877F2",
        placeholder: "https://facebook.com/username",
    },
    instagram: {
        name: "Instagram",
        icon: Instagram,
        color: "#E4405F",
        placeholder: "https://instagram.com/username",
    },
    youtube: {
        name: "YouTube",
        icon: Youtube,
        color: "#FF0000",
        placeholder: "https://youtube.com/c/username",
    },
};

// Author status options
const AUTHOR_STATUS = {
    active: { label: "Active", color: "green" },
    inactive: { label: "Inactive", color: "gray" },
    pending: { label: "Pending", color: "yellow" },
    suspended: { label: "Suspended", color: "red" },
};

// Author roles
const AUTHOR_ROLES = {
    admin: { label: "Admin", color: "red", icon: Crown },
    editor: { label: "Editor", color: "purple", icon: Edit },
    author: { label: "Author", color: "blue", icon: PenTool },
    contributor: { label: "Contributor", color: "green", icon: User },
    guest: { label: "Guest", color: "gray", icon: UserPlus },
};

export default function BlogAuthors({ auth, authors, flash }) {
    // State management
    const [searchQuery, setSearchQuery] = useState("");
    const [statusFilter, setStatusFilter] = useState("all");
    const [roleFilter, setRoleFilter] = useState("all");
    const [sortBy, setSortBy] = useState("name");
    const [sortOrder, setSortOrder] = useState("asc");
    const [viewMode, setViewMode] = useState("table"); // table, grid
    const [selectedAuthors, setSelectedAuthors] = useState([]);
    const [showCreateModal, setShowCreateModal] = useState(false);
    const [showEditModal, setShowEditModal] = useState(false);
    const [showDeleteModal, setShowDeleteModal] = useState(false);
    const [showDetailsModal, setShowDetailsModal] = useState(false);
    const [editingAuthor, setEditingAuthor] = useState(null);
    const [deletingAuthor, setDeletingAuthor] = useState(null);
    const [viewingAuthor, setViewingAuthor] = useState(null);
    const [loading, setLoading] = useState(false);
    const [actionLoading, setActionLoading] = useState({});
    const [avatarPreview, setAvatarPreview] = useState(null);
    const [expandedCard, setExpandedCard] = useState(null);

    // Form for creating/editing authors
    const { data, setData, post, patch, processing, errors, reset } = useForm({
        name: "",
        email: "",
        bio: "",
        avatar: null,
        social_links: {},
        status: "active",
        role: "author",
        password: "",
        password_confirmation: "",
    });

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: {
            opacity: 1,
            y: 0,
            transition: {
                duration: 0.5,
            },
        },
    };

    const modalVariants = {
        hidden: { opacity: 0, scale: 0.95 },
        visible: { opacity: 1, scale: 1 },
        exit: { opacity: 0, scale: 0.95 },
    };

    const cardVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0 },
        hover: { y: -2, transition: { duration: 0.2 } },
    };

    // Filter and sort authors
    const filteredAuthors = useMemo(() => {
        let filtered = authors || [];

        // Search filter
        if (searchQuery) {
            filtered = filtered.filter(
                (author) =>
                    author.name
                        .toLowerCase()
                        .includes(searchQuery.toLowerCase()) ||
                    author.email
                        .toLowerCase()
                        .includes(searchQuery.toLowerCase()) ||
                    (author.bio &&
                        author.bio
                            .toLowerCase()
                            .includes(searchQuery.toLowerCase()))
            );
        }

        // Status filter
        if (statusFilter !== "all") {
            filtered = filtered.filter(
                (author) => author.status === statusFilter
            );
        }

        // Role filter
        if (roleFilter !== "all") {
            filtered = filtered.filter((author) => author.role === roleFilter);
        }

        // Sort authors
        filtered.sort((a, b) => {
            let aValue = a[sortBy];
            let bValue = b[sortBy];

            if (sortBy === "posts_count") {
                aValue = a.posts_count || 0;
                bValue = b.posts_count || 0;
            } else if (sortBy === "created_at") {
                aValue = new Date(a.created_at);
                bValue = new Date(b.created_at);
            }

            if (sortOrder === "asc") {
                return aValue > bValue ? 1 : -1;
            } else {
                return aValue < bValue ? 1 : -1;
            }
        });

        return filtered;
    }, [authors, searchQuery, statusFilter, roleFilter, sortBy, sortOrder]);

    // Handle sort
    const handleSort = useCallback(
        (field) => {
            if (sortBy === field) {
                setSortOrder(sortOrder === "asc" ? "desc" : "asc");
            } else {
                setSortBy(field);
                setSortOrder("asc");
            }
        },
        [sortBy, sortOrder]
    );

    // Handle avatar upload
    const handleAvatarUpload = useCallback(
        (e) => {
            const file = e.target.files[0];
            if (file) {
                setData("avatar", file);

                // Create preview
                const reader = new FileReader();
                reader.onload = (e) => {
                    setAvatarPreview(e.target.result);
                };
                reader.readAsDataURL(file);
            }
        },
        [setData]
    );

    // Handle social links
    const handleSocialLinkChange = useCallback(
        (platform, value) => {
            setData("social_links", {
                ...data.social_links,
                [platform]: value,
            });
        },
        [data.social_links, setData]
    );

    // Handle create author
    const handleCreateAuthor = useCallback(() => {
        const formData = new FormData();
        Object.keys(data).forEach((key) => {
            if (key === "social_links") {
                formData.append(key, JSON.stringify(data[key]));
            } else if (key === "avatar" && data[key]) {
                formData.append(key, data[key]);
            } else if (data[key]) {
                formData.append(key, data[key]);
            }
        });

        post("/admin/blog/authors", {
            onSuccess: () => {
                setShowCreateModal(false);
                setAvatarPreview(null);
                reset();
            },
        });
    }, [data, post, reset]);

    // Handle edit author
    const handleEditAuthor = useCallback(
        (author) => {
            setEditingAuthor(author);
            setData({
                name: author.name,
                email: author.email,
                bio: author.bio || "",
                avatar: null,
                social_links: author.social_links || {},
                status: author.status || "active",
                role: author.role || "author",
                password: "",
                password_confirmation: "",
            });
            setAvatarPreview(author.avatar_url);
            setShowEditModal(true);
        },
        [setData]
    );

    // Handle update author
    const handleUpdateAuthor = useCallback(() => {
        if (!editingAuthor) return;

        const formData = new FormData();
        Object.keys(data).forEach((key) => {
            if (key === "social_links") {
                formData.append(key, JSON.stringify(data[key]));
            } else if (key === "avatar" && data[key]) {
                formData.append(key, data[key]);
            } else if (
                data[key] &&
                key !== "password" &&
                key !== "password_confirmation"
            ) {
                formData.append(key, data[key]);
            }
        });

        // Only add password if it's provided
        if (data.password) {
            formData.append("password", data.password);
            formData.append(
                "password_confirmation",
                data.password_confirmation
            );
        }

        formData.append("_method", "PATCH");

        post(`/admin/blog/authors/${editingAuthor.id}`, {
            onSuccess: () => {
                setShowEditModal(false);
                setEditingAuthor(null);
                setAvatarPreview(null);
                reset();
            },
        });
    }, [editingAuthor, data, post, reset]);

    // Handle delete author
    const handleDeleteAuthor = useCallback((author) => {
        setDeletingAuthor(author);
        setShowDeleteModal(true);
    }, []);

    // Confirm delete author
    const confirmDeleteAuthor = useCallback(() => {
        if (!deletingAuthor) return;

        setActionLoading((prev) => ({ ...prev, [deletingAuthor.id]: true }));

        router.delete(`/admin/blog/authors/${deletingAuthor.id}`, {
            onSuccess: () => {
                setShowDeleteModal(false);
                setDeletingAuthor(null);
            },
            onFinish: () => {
                setActionLoading((prev) => ({
                    ...prev,
                    [deletingAuthor.id]: false,
                }));
            },
        });
    }, [deletingAuthor]);

    // Handle view author details
    const handleViewAuthor = useCallback((author) => {
        setViewingAuthor(author);
        setShowDetailsModal(true);
    }, []);

    // Handle author selection
    const handleAuthorSelect = useCallback((authorId) => {
        setSelectedAuthors((prev) =>
            prev.includes(authorId)
                ? prev.filter((id) => id !== authorId)
                : [...prev, authorId]
        );
    }, []);

    // Handle select all
    const handleSelectAll = useCallback(() => {
        if (selectedAuthors.length === filteredAuthors.length) {
            setSelectedAuthors([]);
        } else {
            setSelectedAuthors(filteredAuthors.map((author) => author.id));
        }
    }, [selectedAuthors, filteredAuthors]);

    // Toggle card expansion
    const toggleCardExpansion = useCallback(
        (authorId) => {
            setExpandedCard(expandedCard === authorId ? null : authorId);
        },
        [expandedCard]
    );

    // Get sort icon
    const getSortIcon = (field) => {
        if (sortBy !== field) return <ArrowUpDown className="w-4 h-4" />;
        return sortOrder === "asc" ? (
            <ArrowUp className="w-4 h-4" />
        ) : (
            <ArrowDown className="w-4 h-4" />
        );
    };

    // Get status badge
    const getStatusBadge = (status) => {
        const config = AUTHOR_STATUS[status] || AUTHOR_STATUS.active;
        return (
            <span
                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-${config.color}-100 text-${config.color}-800 dark:bg-${config.color}-900/30 dark:text-${config.color}-300`}
            >
                {config.label}
            </span>
        );
    };

    // Get role badge
    const getRoleBadge = (role) => {
        const config = AUTHOR_ROLES[role] || AUTHOR_ROLES.author;
        const IconComponent = config.icon;
        return (
            <span
                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-${config.color}-100 text-${config.color}-800 dark:bg-${config.color}-900/30 dark:text-${config.color}-300`}
            >
                <IconComponent className="w-3 h-3 mr-1" />
                {config.label}
            </span>
        );
    };

    // Get social platform icon
    const getSocialIcon = (platform) => {
        const config = SOCIAL_PLATFORMS[platform];
        if (!config) return <LinkIcon className="w-4 h-4" />;
        const IconComponent = config.icon;
        return (
            <IconComponent
                className="w-4 h-4"
                style={{ color: config.color }}
            />
        );
    };

    // Calculate author statistics
    const authorStats = useMemo(() => {
        const totalAuthors = authors?.length || 0;
        const activeAuthors =
            authors?.filter((author) => author.status === "active").length || 0;
        const totalPosts =
            authors?.reduce(
                (sum, author) => sum + (author.posts_count || 0),
                0
            ) || 0;
        const avgPostsPerAuthor =
            totalAuthors > 0 ? Math.round(totalPosts / totalAuthors) : 0;
        const topAuthor =
            authors?.reduce((top, author) =>
                (author.posts_count || 0) > (top?.posts_count || 0)
                    ? author
                    : top
            ) || null;

        return {
            totalAuthors,
            activeAuthors,
            totalPosts,
            avgPostsPerAuthor,
            topAuthor,
        };
    }, [authors]);

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div className="flex items-center justify-between">
                    <h2 className="font-semibold text-xl text-gray-800 dark:text-gray-200 leading-tight">
                        Blog Authors
                    </h2>
                    <button
                        onClick={() => setShowCreateModal(true)}
                        className="inline-flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                    >
                        <Plus className="w-4 h-4 mr-2" />
                        Add Author
                    </button>
                </div>
            }
        >
            <Head title="Blog Authors" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8 space-y-6"
            >
                {/* Statistics Cards */}
                <motion.div
                    variants={itemVariants}
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6"
                >
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Authors
                                </p>
                                <p className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                    {authorStats.totalAuthors}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <Users className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Active Authors
                                </p>
                                <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                    {authorStats.activeAuthors}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                <UserCheck className="w-6 h-6 text-green-600 dark:text-green-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Posts
                                </p>
                                <p className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                    {authorStats.totalPosts}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-purple-100 dark:bg-purple-900/30 rounded-lg flex items-center justify-center">
                                <FileText className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Avg Posts/Author
                                </p>
                                <p className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                    {authorStats.avgPostsPerAuthor}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-orange-100 dark:bg-orange-900/30 rounded-lg flex items-center justify-center">
                                <BarChart3 className="w-6 h-6 text-orange-600 dark:text-orange-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Top Author
                                </p>
                                <p className="text-lg font-bold text-yellow-600 dark:text-yellow-400">
                                    {authorStats.topAuthor?.name || "None"}
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    {authorStats.topAuthor?.posts_count || 0}{" "}
                                    posts
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-yellow-100 dark:bg-yellow-900/30 rounded-lg flex items-center justify-center">
                                <Crown className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Search and Controls */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex flex-col lg:flex-row gap-4 items-start lg:items-center justify-between">
                        <div className="flex-1 max-w-md">
                            <div className="relative">
                                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                                <input
                                    type="text"
                                    placeholder="Search authors..."
                                    value={searchQuery}
                                    onChange={(e) =>
                                        setSearchQuery(e.target.value)
                                    }
                                    className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400"
                                />
                            </div>
                        </div>

                        <div className="flex items-center gap-3">
                            <div className="flex items-center gap-2">
                                <FilterIcon className="w-4 h-4 text-gray-400" />
                                <select
                                    value={statusFilter}
                                    onChange={(e) =>
                                        setStatusFilter(e.target.value)
                                    }
                                    className="border border-gray-300 dark:border-gray-600 rounded-lg px-3 py-2 bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                >
                                    <option value="all">All Status</option>
                                    {Object.entries(AUTHOR_STATUS).map(
                                        ([key, config]) => (
                                            <option key={key} value={key}>
                                                {config.label}
                                            </option>
                                        )
                                    )}
                                </select>
                            </div>

                            <div className="flex items-center gap-2">
                                <select
                                    value={roleFilter}
                                    onChange={(e) =>
                                        setRoleFilter(e.target.value)
                                    }
                                    className="border border-gray-300 dark:border-gray-600 rounded-lg px-3 py-2 bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                >
                                    <option value="all">All Roles</option>
                                    {Object.entries(AUTHOR_ROLES).map(
                                        ([key, config]) => (
                                            <option key={key} value={key}>
                                                {config.label}
                                            </option>
                                        )
                                    )}
                                </select>
                            </div>

                            <div className="flex items-center gap-2 border-l border-gray-300 dark:border-gray-600 pl-3">
                                <button
                                    onClick={() => setViewMode("table")}
                                    className={`p-2 rounded-lg ${
                                        viewMode === "table"
                                            ? "bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400"
                                            : "text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                    }`}
                                >
                                    <List className="w-4 h-4" />
                                </button>
                                <button
                                    onClick={() => setViewMode("grid")}
                                    className={`p-2 rounded-lg ${
                                        viewMode === "grid"
                                            ? "bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400"
                                            : "text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                    }`}
                                >
                                    <Grid className="w-4 h-4" />
                                </button>
                            </div>

                            <div className="text-sm text-gray-600 dark:text-gray-400">
                                {filteredAuthors.length} of{" "}
                                {authors?.length || 0} authors
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Bulk Actions */}
                <AnimatePresence>
                    {selectedAuthors.length > 0 && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4"
                        >
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-3">
                                    <span className="text-sm text-blue-700 dark:text-blue-300">
                                        {selectedAuthors.length} author
                                        {selectedAuthors.length !== 1
                                            ? "s"
                                            : ""}{" "}
                                        selected
                                    </span>
                                    <button
                                        onClick={() => setSelectedAuthors([])}
                                        className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200"
                                    >
                                        Clear selection
                                    </button>
                                </div>

                                <div className="flex items-center gap-2">
                                    <button
                                        onClick={() => {
                                            // Implement bulk status change
                                            console.log(
                                                "Bulk status change for:",
                                                selectedAuthors
                                            );
                                        }}
                                        className="px-3 py-1 bg-yellow-600 hover:bg-yellow-700 text-white rounded-lg text-sm transition-colors"
                                    >
                                        Change Status
                                    </button>
                                    <button
                                        onClick={() => {
                                            // Implement bulk delete
                                            if (
                                                confirm(
                                                    `Are you sure you want to delete ${selectedAuthors.length} authors?`
                                                )
                                            ) {
                                                // Add bulk delete logic here
                                            }
                                        }}
                                        className="px-3 py-1 bg-red-600 hover:bg-red-700 text-white rounded-lg text-sm transition-colors"
                                    >
                                        Delete Selected
                                    </button>
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Authors Display */}
                {viewMode === "table" ? (
                    /* Table View */
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden"
                    >
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead className="bg-gray-50 dark:bg-gray-900 border-b border-gray-200 dark:border-gray-700">
                                    <tr>
                                        <th className="w-12 px-6 py-3 text-left">
                                            <input
                                                type="checkbox"
                                                checked={
                                                    selectedAuthors.length ===
                                                    filteredAuthors.length
                                                }
                                                onChange={handleSelectAll}
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("name")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Author
                                                {getSortIcon("name")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Contact
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Role
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Status
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("posts_count")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Posts
                                                {getSortIcon("posts_count")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("created_at")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Joined
                                                {getSortIcon("created_at")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Actions
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {filteredAuthors.map((author, index) => (
                                        <motion.tr
                                            key={author.id}
                                            initial={{ opacity: 0, y: 20 }}
                                            animate={{ opacity: 1, y: 0 }}
                                            transition={{ delay: index * 0.05 }}
                                            className="hover:bg-gray-50 dark:hover:bg-gray-900 transition-colors"
                                        >
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <input
                                                    type="checkbox"
                                                    checked={selectedAuthors.includes(
                                                        author.id
                                                    )}
                                                    onChange={() =>
                                                        handleAuthorSelect(
                                                            author.id
                                                        )
                                                    }
                                                    className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                                />
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center">
                                                    <div className="flex-shrink-0 w-10 h-10 mr-3">
                                                        {author.avatar_url ? (
                                                            <img
                                                                src={
                                                                    author.avatar_url
                                                                }
                                                                alt={
                                                                    author.name
                                                                }
                                                                className="w-10 h-10 rounded-full object-cover"
                                                            />
                                                        ) : (
                                                            <div className="w-10 h-10 bg-gray-300 dark:bg-gray-600 rounded-full flex items-center justify-center">
                                                                <User className="w-5 h-5 text-gray-500 dark:text-gray-400" />
                                                            </div>
                                                        )}
                                                    </div>
                                                    <div>
                                                        <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                            {author.name}
                                                        </div>
                                                        <div className="text-sm text-gray-500 dark:text-gray-400">
                                                            @
                                                            {author.name
                                                                .toLowerCase()
                                                                .replace(
                                                                    /\s+/g,
                                                                    ""
                                                                )}
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="text-sm text-gray-900 dark:text-white">
                                                    {author.email}
                                                </div>
                                                <div className="flex items-center gap-2 mt-1">
                                                    {author.social_links &&
                                                        Object.entries(
                                                            author.social_links
                                                        ).map(
                                                            ([platform, url]) =>
                                                                url && (
                                                                    <a
                                                                        key={
                                                                            platform
                                                                        }
                                                                        href={
                                                                            url
                                                                        }
                                                                        target="_blank"
                                                                        rel="noopener noreferrer"
                                                                        className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                                                    >
                                                                        {getSocialIcon(
                                                                            platform
                                                                        )}
                                                                    </a>
                                                                )
                                                        )}
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                {getRoleBadge(author.role)}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                {getStatusBadge(author.status)}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center">
                                                    <span className="text-sm text-gray-900 dark:text-white">
                                                        {author.posts_count ||
                                                            0}
                                                    </span>
                                                    {author.posts_count > 0 && (
                                                        <Link
                                                            href={`/admin/blog/posts?author=${author.id}`}
                                                            className="ml-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                        >
                                                            <Eye className="w-4 h-4" />
                                                        </Link>
                                                    )}
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {new Date(
                                                    author.created_at
                                                ).toLocaleDateString()}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <div className="flex items-center gap-2">
                                                    <button
                                                        onClick={() =>
                                                            handleViewAuthor(
                                                                author
                                                            )
                                                        }
                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                    >
                                                        <Eye className="w-4 h-4" />
                                                    </button>
                                                    <button
                                                        onClick={() =>
                                                            handleEditAuthor(
                                                                author
                                                            )
                                                        }
                                                        className="text-yellow-600 hover:text-yellow-800 dark:text-yellow-400 dark:hover:text-yellow-200"
                                                    >
                                                        <Edit className="w-4 h-4" />
                                                    </button>
                                                    <button
                                                        onClick={() =>
                                                            handleDeleteAuthor(
                                                                author
                                                            )
                                                        }
                                                        disabled={
                                                            actionLoading[
                                                                author.id
                                                            ]
                                                        }
                                                        className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-200 disabled:opacity-50"
                                                    >
                                                        {actionLoading[
                                                            author.id
                                                        ] ? (
                                                            <Loader2 className="w-4 h-4 animate-spin" />
                                                        ) : (
                                                            <Trash2 className="w-4 h-4" />
                                                        )}
                                                    </button>
                                                </div>
                                            </td>
                                        </motion.tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    </motion.div>
                ) : (
                    /* Grid View */
                    <motion.div
                        variants={itemVariants}
                        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"
                    >
                        {filteredAuthors.map((author, index) => (
                            <motion.div
                                key={author.id}
                                variants={cardVariants}
                                initial="hidden"
                                animate="visible"
                                whileHover="hover"
                                transition={{ delay: index * 0.1 }}
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden"
                            >
                                <div className="p-6">
                                    <div className="flex items-start justify-between mb-4">
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                checked={selectedAuthors.includes(
                                                    author.id
                                                )}
                                                onChange={() =>
                                                    handleAuthorSelect(
                                                        author.id
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600 mr-3"
                                            />
                                            <div className="flex-shrink-0">
                                                {author.avatar_url ? (
                                                    <img
                                                        src={author.avatar_url}
                                                        alt={author.name}
                                                        className="w-12 h-12 rounded-full object-cover"
                                                    />
                                                ) : (
                                                    <div className="w-12 h-12 bg-gray-300 dark:bg-gray-600 rounded-full flex items-center justify-center">
                                                        <User className="w-6 h-6 text-gray-500 dark:text-gray-400" />
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            {getRoleBadge(author.role)}
                                            <button
                                                onClick={() =>
                                                    toggleCardExpansion(
                                                        author.id
                                                    )
                                                }
                                                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                            >
                                                {expandedCard === author.id ? (
                                                    <Minimize2 className="w-4 h-4" />
                                                ) : (
                                                    <Maximize2 className="w-4 h-4" />
                                                )}
                                            </button>
                                        </div>
                                    </div>

                                    <div className="mb-4">
                                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-1">
                                            {author.name}
                                        </h3>
                                        <p className="text-sm text-gray-500 dark:text-gray-400 mb-2">
                                            {author.email}
                                        </p>
                                        {author.bio && (
                                            <p className="text-sm text-gray-600 dark:text-gray-300 line-clamp-3">
                                                {author.bio}
                                            </p>
                                        )}
                                    </div>

                                    <div className="flex items-center justify-between mb-4">
                                        {getStatusBadge(author.status)}
                                        <span className="text-sm text-gray-500 dark:text-gray-400">
                                            {author.posts_count || 0} posts
                                        </span>
                                    </div>

                                    {author.social_links &&
                                        Object.keys(author.social_links)
                                            .length > 0 && (
                                            <div className="flex items-center gap-2 mb-4">
                                                {Object.entries(
                                                    author.social_links
                                                ).map(
                                                    ([platform, url]) =>
                                                        url && (
                                                            <a
                                                                key={platform}
                                                                href={url}
                                                                target="_blank"
                                                                rel="noopener noreferrer"
                                                                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors"
                                                            >
                                                                {getSocialIcon(
                                                                    platform
                                                                )}
                                                            </a>
                                                        )
                                                )}
                                            </div>
                                        )}

                                    <AnimatePresence>
                                        {expandedCard === author.id && (
                                            <motion.div
                                                initial={{
                                                    opacity: 0,
                                                    height: 0,
                                                }}
                                                animate={{
                                                    opacity: 1,
                                                    height: "auto",
                                                }}
                                                exit={{ opacity: 0, height: 0 }}
                                                className="border-t border-gray-200 dark:border-gray-700 pt-4 mt-4"
                                            >
                                                <div className="space-y-2">
                                                    <div className="flex items-center justify-between text-sm">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Member since:
                                                        </span>
                                                        <span className="text-gray-900 dark:text-white">
                                                            {new Date(
                                                                author.created_at
                                                            ).toLocaleDateString()}
                                                        </span>
                                                    </div>
                                                    <div className="flex items-center justify-between text-sm">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Last active:
                                                        </span>
                                                        <span className="text-gray-900 dark:text-white">
                                                            {author.last_active_at
                                                                ? new Date(
                                                                      author.last_active_at
                                                                  ).toLocaleDateString()
                                                                : "Never"}
                                                        </span>
                                                    </div>
                                                    <div className="flex items-center justify-between text-sm">
                                                        <span className="text-gray-500 dark:text-gray-400">
                                                            Total views:
                                                        </span>
                                                        <span className="text-gray-900 dark:text-white">
                                                            {author.total_views ||
                                                                0}
                                                        </span>
                                                    </div>
                                                </div>
                                            </motion.div>
                                        )}
                                    </AnimatePresence>

                                    <div className="flex items-center justify-between pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center gap-2">
                                            <button
                                                onClick={() =>
                                                    handleViewAuthor(author)
                                                }
                                                className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                title="View details"
                                            >
                                                <Eye className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={() =>
                                                    handleEditAuthor(author)
                                                }
                                                className="text-yellow-600 hover:text-yellow-800 dark:text-yellow-400 dark:hover:text-yellow-200"
                                                title="Edit author"
                                            >
                                                <Edit className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={() =>
                                                    handleDeleteAuthor(author)
                                                }
                                                disabled={
                                                    actionLoading[author.id]
                                                }
                                                className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-200 disabled:opacity-50"
                                                title="Delete author"
                                            >
                                                {actionLoading[author.id] ? (
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                ) : (
                                                    <Trash2 className="w-4 h-4" />
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            ID: {author.id}
                                        </div>
                                    </div>
                                </div>
                            </motion.div>
                        ))}
                    </motion.div>
                )}

                {/* Empty State */}
                {filteredAuthors.length === 0 && (
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-12 text-center"
                    >
                        <div className="w-16 h-16 mx-auto mb-4 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center">
                            <Users className="w-8 h-8 text-gray-400" />
                        </div>
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                            No authors found
                        </h3>
                        <p className="text-gray-500 dark:text-gray-400 mb-6">
                            {searchQuery
                                ? "Try adjusting your search criteria or filters."
                                : "Get started by adding your first author."}
                        </p>
                        <div className="flex items-center justify-center gap-3">
                            {searchQuery ||
                            statusFilter !== "all" ||
                            roleFilter !== "all" ? (
                                <button
                                    onClick={() => {
                                        setSearchQuery("");
                                        setStatusFilter("all");
                                        setRoleFilter("all");
                                    }}
                                    className="flex items-center gap-2 px-4 py-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                >
                                    <RefreshCw className="w-4 h-4" />
                                    Clear filters
                                </button>
                            ) : (
                                <button
                                    onClick={() => setShowCreateModal(true)}
                                    className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                                >
                                    <Plus className="w-4 h-4" />
                                    Add Author
                                </button>
                            )}
                        </div>
                    </motion.div>
                )}

                {/* Create Author Modal */}
                <AnimatePresence>
                    {showCreateModal && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto"
                            >
                                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center justify-between">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            Create New Author
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowCreateModal(false);
                                                setAvatarPreview(null);
                                                reset();
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-6 h-6" />
                                        </button>
                                    </div>
                                </div>

                                <form
                                    onSubmit={(e) => {
                                        e.preventDefault();
                                        handleCreateAuthor();
                                    }}
                                    className="p-6 space-y-6"
                                >
                                    {/* Avatar Upload */}
                                    <div className="flex items-center gap-4">
                                        <div className="flex-shrink-0">
                                            <div className="w-20 h-20 bg-gray-300 dark:bg-gray-600 rounded-full flex items-center justify-center overflow-hidden">
                                                {avatarPreview ? (
                                                    <img
                                                        src={avatarPreview}
                                                        alt="Avatar preview"
                                                        className="w-full h-full object-cover"
                                                    />
                                                ) : (
                                                    <User className="w-8 h-8 text-gray-500 dark:text-gray-400" />
                                                )}
                                            </div>
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Profile Picture
                                            </label>
                                            <input
                                                type="file"
                                                accept="image/*"
                                                onChange={handleAvatarUpload}
                                                className="block w-full text-sm text-gray-500 dark:text-gray-400 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 dark:file:bg-blue-900/30 dark:file:text-blue-400"
                                            />
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Full Name *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.name}
                                                onChange={(e) =>
                                                    setData(
                                                        "name",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter full name"
                                                required
                                            />
                                            {errors.name && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.name}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Email Address *
                                            </label>
                                            <input
                                                type="email"
                                                value={data.email}
                                                onChange={(e) =>
                                                    setData(
                                                        "email",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter email address"
                                                required
                                            />
                                            {errors.email && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.email}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Role
                                            </label>
                                            <select
                                                value={data.role}
                                                onChange={(e) =>
                                                    setData(
                                                        "role",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.entries(
                                                    AUTHOR_ROLES
                                                ).map(([key, config]) => (
                                                    <option
                                                        key={key}
                                                        value={key}
                                                    >
                                                        {config.label}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.role && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.role}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Status
                                            </label>
                                            <select
                                                value={data.status}
                                                onChange={(e) =>
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.entries(
                                                    AUTHOR_STATUS
                                                ).map(([key, config]) => (
                                                    <option
                                                        key={key}
                                                        value={key}
                                                    >
                                                        {config.label}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.status && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.status}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Bio
                                        </label>
                                        <textarea
                                            value={data.bio}
                                            onChange={(e) =>
                                                setData("bio", e.target.value)
                                            }
                                            rows={4}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="Enter author bio"
                                        />
                                        {errors.bio && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.bio}
                                            </p>
                                        )}
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Password *
                                            </label>
                                            <input
                                                type="password"
                                                value={data.password}
                                                onChange={(e) =>
                                                    setData(
                                                        "password",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter password"
                                                required
                                            />
                                            {errors.password && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.password}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Confirm Password *
                                            </label>
                                            <input
                                                type="password"
                                                value={
                                                    data.password_confirmation
                                                }
                                                onChange={(e) =>
                                                    setData(
                                                        "password_confirmation",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Confirm password"
                                                required
                                            />
                                            {errors.password_confirmation && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {
                                                        errors.password_confirmation
                                                    }
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    {/* Social Links */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-4">
                                            Social Links
                                        </label>
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            {Object.entries(
                                                SOCIAL_PLATFORMS
                                            ).map(([platform, config]) => (
                                                <div
                                                    key={platform}
                                                    className="flex items-center"
                                                >
                                                    <div className="flex-shrink-0 w-10 h-10 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-center mr-3">
                                                        <config.icon
                                                            className="w-5 h-5"
                                                            style={{
                                                                color: config.color,
                                                            }}
                                                        />
                                                    </div>
                                                    <div className="flex-1">
                                                        <input
                                                            type="url"
                                                            value={
                                                                data
                                                                    .social_links[
                                                                    platform
                                                                ] || ""
                                                            }
                                                            onChange={(e) =>
                                                                handleSocialLinkChange(
                                                                    platform,
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white text-sm"
                                                            placeholder={
                                                                config.placeholder
                                                            }
                                                        />
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>

                                    <div className="flex items-center justify-end gap-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <button
                                            type="button"
                                            onClick={() => {
                                                setShowCreateModal(false);
                                                setAvatarPreview(null);
                                                reset();
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            disabled={processing}
                                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Creating...
                                                </>
                                            ) : (
                                                <>
                                                    <UserPlus className="w-4 h-4" />
                                                    Create Author
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </form>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Edit Author Modal */}
                <AnimatePresence>
                    {showEditModal && editingAuthor && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto"
                            >
                                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center justify-between">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            Edit Author
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowEditModal(false);
                                                setEditingAuthor(null);
                                                setAvatarPreview(null);
                                                reset();
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-6 h-6" />
                                        </button>
                                    </div>
                                </div>

                                <form
                                    onSubmit={(e) => {
                                        e.preventDefault();
                                        handleUpdateAuthor();
                                    }}
                                    className="p-6 space-y-6"
                                >
                                    {/* Avatar Upload */}
                                    <div className="flex items-center gap-4">
                                        <div className="flex-shrink-0">
                                            <div className="w-20 h-20 bg-gray-300 dark:bg-gray-600 rounded-full flex items-center justify-center overflow-hidden">
                                                {avatarPreview ? (
                                                    <img
                                                        src={avatarPreview}
                                                        alt="Avatar preview"
                                                        className="w-full h-full object-cover"
                                                    />
                                                ) : (
                                                    <User className="w-8 h-8 text-gray-500 dark:text-gray-400" />
                                                )}
                                            </div>
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Profile Picture
                                            </label>
                                            <input
                                                type="file"
                                                accept="image/*"
                                                onChange={handleAvatarUpload}
                                                className="block w-full text-sm text-gray-500 dark:text-gray-400 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 dark:file:bg-blue-900/30 dark:file:text-blue-400"
                                            />
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Full Name *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.name}
                                                onChange={(e) =>
                                                    setData(
                                                        "name",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter full name"
                                                required
                                            />
                                            {errors.name && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.name}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Email Address *
                                            </label>
                                            <input
                                                type="email"
                                                value={data.email}
                                                onChange={(e) =>
                                                    setData(
                                                        "email",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter email address"
                                                required
                                            />
                                            {errors.email && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.email}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Role
                                            </label>
                                            <select
                                                value={data.role}
                                                onChange={(e) =>
                                                    setData(
                                                        "role",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.entries(
                                                    AUTHOR_ROLES
                                                ).map(([key, config]) => (
                                                    <option
                                                        key={key}
                                                        value={key}
                                                    >
                                                        {config.label}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.role && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.role}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Status
                                            </label>
                                            <select
                                                value={data.status}
                                                onChange={(e) =>
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            >
                                                {Object.entries(
                                                    AUTHOR_STATUS
                                                ).map(([key, config]) => (
                                                    <option
                                                        key={key}
                                                        value={key}
                                                    >
                                                        {config.label}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.status && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.status}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Bio
                                        </label>
                                        <textarea
                                            value={data.bio}
                                            onChange={(e) =>
                                                setData("bio", e.target.value)
                                            }
                                            rows={4}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                            placeholder="Enter author bio"
                                        />
                                        {errors.bio && (
                                            <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                {errors.bio}
                                            </p>
                                        )}
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                New Password{" "}
                                                <span className="text-gray-500">
                                                    (leave blank to keep
                                                    current)
                                                </span>
                                            </label>
                                            <input
                                                type="password"
                                                value={data.password}
                                                onChange={(e) =>
                                                    setData(
                                                        "password",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Enter new password"
                                            />
                                            {errors.password && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.password}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Confirm Password
                                            </label>
                                            <input
                                                type="password"
                                                value={
                                                    data.password_confirmation
                                                }
                                                onChange={(e) =>
                                                    setData(
                                                        "password_confirmation",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                placeholder="Confirm new password"
                                            />
                                            {errors.password_confirmation && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {
                                                        errors.password_confirmation
                                                    }
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    {/* Social Links */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-4">
                                            Social Links
                                        </label>
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            {Object.entries(
                                                SOCIAL_PLATFORMS
                                            ).map(([platform, config]) => (
                                                <div
                                                    key={platform}
                                                    className="flex items-center"
                                                >
                                                    <div className="flex-shrink-0 w-10 h-10 bg-gray-100 dark:bg-gray-700 rounded-lg flex items-center justify-center mr-3">
                                                        <config.icon
                                                            className="w-5 h-5"
                                                            style={{
                                                                color: config.color,
                                                            }}
                                                        />
                                                    </div>
                                                    <div className="flex-1">
                                                        <input
                                                            type="url"
                                                            value={
                                                                data
                                                                    .social_links[
                                                                    platform
                                                                ] || ""
                                                            }
                                                            onChange={(e) =>
                                                                handleSocialLinkChange(
                                                                    platform,
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white text-sm"
                                                            placeholder={
                                                                config.placeholder
                                                            }
                                                        />
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>

                                    <div className="flex items-center justify-end gap-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <button
                                            type="button"
                                            onClick={() => {
                                                setShowEditModal(false);
                                                setEditingAuthor(null);
                                                setAvatarPreview(null);
                                                reset();
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            disabled={processing}
                                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {processing ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Updating...
                                                </>
                                            ) : (
                                                <>
                                                    <Save className="w-4 h-4" />
                                                    Update Author
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </form>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Delete Confirmation Modal */}
                <AnimatePresence>
                    {showDeleteModal && deletingAuthor && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full"
                            >
                                <div className="p-6">
                                    <div className="flex items-center justify-center w-12 h-12 mx-auto mb-4 bg-red-100 dark:bg-red-900/30 rounded-full">
                                        <AlertCircle className="w-6 h-6 text-red-600 dark:text-red-400" />
                                    </div>
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white text-center mb-2">
                                        Delete Author
                                    </h3>
                                    <p className="text-sm text-gray-500 dark:text-gray-400 text-center mb-6">
                                        Are you sure you want to delete "
                                        {deletingAuthor.name}"? This action
                                        cannot be undone.
                                        {deletingAuthor.posts_count > 0 && (
                                            <span className="block mt-2 text-yellow-600 dark:text-yellow-400">
                                                This author has{" "}
                                                {deletingAuthor.posts_count}{" "}
                                                published posts that will need
                                                to be reassigned.
                                            </span>
                                        )}
                                    </p>
                                    <div className="flex items-center justify-center gap-3">
                                        <button
                                            onClick={() => {
                                                setShowDeleteModal(false);
                                                setDeletingAuthor(null);
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            onClick={confirmDeleteAuthor}
                                            disabled={
                                                actionLoading[deletingAuthor.id]
                                            }
                                            className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-700 dark:bg-red-500 dark:hover:bg-red-600 text-white rounded-lg transition-colors disabled:opacity-50"
                                        >
                                            {actionLoading[
                                                deletingAuthor.id
                                            ] ? (
                                                <>
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                    Deleting...
                                                </>
                                            ) : (
                                                <>
                                                    <Trash2 className="w-4 h-4" />
                                                    Delete Author
                                                </>
                                            )}
                                        </button>
                                    </div>
                                </div>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>

                {/* Author Details Modal */}
                <AnimatePresence>
                    {showDetailsModal && viewingAuthor && (
                        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                            <motion.div
                                variants={modalVariants}
                                initial="hidden"
                                animate="visible"
                                exit="exit"
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto"
                            >
                                <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center justify-between">
                                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                                            Author Details
                                        </h2>
                                        <button
                                            onClick={() => {
                                                setShowDetailsModal(false);
                                                setViewingAuthor(null);
                                            }}
                                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                        >
                                            <X className="w-6 h-6" />
                                        </button>
                                    </div>
                                </div>

                                <div className="p-6 space-y-6">
                                    {/* Author Header */}
                                    <div className="flex items-center gap-4">
                                        <div className="flex-shrink-0">
                                            {viewingAuthor.avatar_url ? (
                                                <img
                                                    src={
                                                        viewingAuthor.avatar_url
                                                    }
                                                    alt={viewingAuthor.name}
                                                    className="w-16 h-16 rounded-full object-cover"
                                                />
                                            ) : (
                                                <div className="w-16 h-16 bg-gray-300 dark:bg-gray-600 rounded-full flex items-center justify-center">
                                                    <User className="w-8 h-8 text-gray-500 dark:text-gray-400" />
                                                </div>
                                            )}
                                        </div>
                                        <div className="flex-1">
                                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white">
                                                {viewingAuthor.name}
                                            </h3>
                                            <p className="text-gray-500 dark:text-gray-400 mb-2">
                                                {viewingAuthor.email}
                                            </p>
                                            <div className="flex items-center gap-2">
                                                {getRoleBadge(
                                                    viewingAuthor.role
                                                )}
                                                {getStatusBadge(
                                                    viewingAuthor.status
                                                )}
                                            </div>
                                        </div>
                                    </div>

                                    {/* Bio */}
                                    {viewingAuthor.bio && (
                                        <div>
                                            <h4 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                                                Biography
                                            </h4>
                                            <p className="text-gray-600 dark:text-gray-400 leading-relaxed">
                                                {viewingAuthor.bio}
                                            </p>
                                        </div>
                                    )}

                                    {/* Statistics */}
                                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                                        <div className="text-center p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                            <div className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                                {viewingAuthor.posts_count || 0}
                                            </div>
                                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                                Posts
                                            </div>
                                        </div>
                                        <div className="text-center p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                            <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                                                {viewingAuthor.total_views || 0}
                                            </div>
                                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                                Views
                                            </div>
                                        </div>
                                        <div className="text-center p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                            <div className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                                {viewingAuthor.total_comments ||
                                                    0}
                                            </div>
                                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                                Comments
                                            </div>
                                        </div>
                                        <div className="text-center p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                            <div className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                                {viewingAuthor.total_likes || 0}
                                            </div>
                                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                                Likes
                                            </div>
                                        </div>
                                    </div>

                                    {/* Social Links */}
                                    {viewingAuthor.social_links &&
                                        Object.keys(viewingAuthor.social_links)
                                            .length > 0 && (
                                            <div>
                                                <h4 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                                    Social Links
                                                </h4>
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                                    {Object.entries(
                                                        viewingAuthor.social_links
                                                    ).map(
                                                        ([platform, url]) =>
                                                            url && (
                                                                <a
                                                                    key={
                                                                        platform
                                                                    }
                                                                    href={url}
                                                                    target="_blank"
                                                                    rel="noopener noreferrer"
                                                                    className="flex items-center gap-3 p-3 bg-gray-50 dark:bg-gray-900 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors"
                                                                >
                                                                    <div className="w-8 h-8 bg-white dark:bg-gray-700 rounded-lg flex items-center justify-center">
                                                                        {getSocialIcon(
                                                                            platform
                                                                        )}
                                                                    </div>
                                                                    <div className="flex-1">
                                                                        <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                                            {SOCIAL_PLATFORMS[
                                                                                platform
                                                                            ]
                                                                                ?.name ||
                                                                                platform}
                                                                        </div>
                                                                        <div className="text-xs text-gray-500 dark:text-gray-400 truncate">
                                                                            {
                                                                                url
                                                                            }
                                                                        </div>
                                                                    </div>
                                                                    <ExternalLink className="w-4 h-4 text-gray-400" />
                                                                </a>
                                                            )
                                                    )}
                                                </div>
                                            </div>
                                        )}

                                    {/* Account Information */}
                                    <div>
                                        <h4 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                            Account Information
                                        </h4>
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Member Since
                                                </span>
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {new Date(
                                                        viewingAuthor.created_at
                                                    ).toLocaleDateString()}
                                                </span>
                                            </div>
                                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Last Updated
                                                </span>
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {new Date(
                                                        viewingAuthor.updated_at
                                                    ).toLocaleDateString()}
                                                </span>
                                            </div>
                                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Last Active
                                                </span>
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {viewingAuthor.last_active_at
                                                        ? new Date(
                                                              viewingAuthor.last_active_at
                                                          ).toLocaleDateString()
                                                        : "Never"}
                                                </span>
                                            </div>
                                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                                    Author ID
                                                </span>
                                                <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                    #{viewingAuthor.id}
                                                </span>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Recent Posts */}
                                    {viewingAuthor.recent_posts &&
                                        viewingAuthor.recent_posts.length >
                                            0 && (
                                            <div>
                                                <div className="flex items-center justify-between mb-4">
                                                    <h4 className="text-lg font-medium text-gray-900 dark:text-white">
                                                        Recent Posts
                                                    </h4>
                                                    <Link
                                                        href={`/admin/blog/posts?author=${viewingAuthor.id}`}
                                                        className="text-sm text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                    >
                                                        View All Posts
                                                    </Link>
                                                </div>
                                                <div className="space-y-3">
                                                    {viewingAuthor.recent_posts.map(
                                                        (post) => (
                                                            <div
                                                                key={post.id}
                                                                className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 rounded-lg"
                                                            >
                                                                <div className="flex-1">
                                                                    <h5 className="text-sm font-medium text-gray-900 dark:text-white">
                                                                        {
                                                                            post.title
                                                                        }
                                                                    </h5>
                                                                    <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                                        {new Date(
                                                                            post.published_at
                                                                        ).toLocaleDateString()}{" "}
                                                                        •{" "}
                                                                        {post.views ||
                                                                            0}{" "}
                                                                        views
                                                                    </p>
                                                                </div>
                                                                <div className="flex items-center gap-2">
                                                                    <span
                                                                        className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                                                            post.status ===
                                                                            "published"
                                                                                ? "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300"
                                                                                : post.status ===
                                                                                  "draft"
                                                                                ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-300"
                                                                                : "bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-300"
                                                                        }`}
                                                                    >
                                                                        {
                                                                            post.status
                                                                        }
                                                                    </span>
                                                                    <Link
                                                                        href={`/admin/blog/posts/${post.id}`}
                                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                                    >
                                                                        <Eye className="w-4 h-4" />
                                                                    </Link>
                                                                </div>
                                                            </div>
                                                        )
                                                    )}
                                                </div>
                                            </div>
                                        )}

                                    {/* Action Buttons */}
                                    <div className="flex items-center justify-between pt-4 border-t border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center gap-3">
                                            <button
                                                onClick={() => {
                                                    setShowDetailsModal(false);
                                                    setViewingAuthor(null);
                                                    handleEditAuthor(
                                                        viewingAuthor
                                                    );
                                                }}
                                                className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                                            >
                                                <Edit className="w-4 h-4" />
                                                Edit Author
                                            </button>
                                            <button
                                                onClick={() => {
                                                    setShowDetailsModal(false);
                                                    setViewingAuthor(null);
                                                    handleDeleteAuthor(
                                                        viewingAuthor
                                                    );
                                                }}
                                                className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-700 dark:bg-red-500 dark:hover:bg-red-600 text-white rounded-lg transition-colors"
                                            >
                                                <Trash2 className="w-4 h-4" />
                                                Delete Author
                                            </button>
                                        </div>
                                        <button
                                            onClick={() => {
                                                setShowDetailsModal(false);
                                                setViewingAuthor(null);
                                            }}
                                            className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                        >
                                            Close
                                        </button>
                                    </div>
                                </div>
                            </motion.div>
                        </div>
                    )}
                </AnimatePresence>
            </motion.div>
        </AdminLayout>
    );
}
