import { useForm, usePage } from "@inertiajs/react";
import { useEffect, useState } from "react";
import { Link } from "@inertiajs/react";
import RichTextEditor from "@/Components/Admin/RichTextEditor";
import {
    ArrowLeft,
    Save,
    Eye,
    FileText,
    Image,
    Tag,
    Settings,
    Globe,
    Plus,
    X,
    Calendar,
    Clock,
    Star,
    MessageCircle,
    ThumbsUp,
    AlertCircle,
    Database,
    User,
    TrendingUp,
    CheckCircle,
    Upload,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";

// Helper functions for time formatting
const formatDateTime = (dateString) => {
    if (!dateString) return "Not set";
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
        month: 'short',
        day: 'numeric',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
    });
};

const getTimeAgo = (dateString) => {
    if (!dateString) return "Never";
    const date = new Date(dateString);
    const now = new Date();
    const seconds = Math.floor((now - date) / 1000);
    
    if (seconds < 60) return "Just now";
    
    const minutes = Math.floor(seconds / 60);
    if (minutes < 60) return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    
    const hours = Math.floor(minutes / 60);
    if (hours < 24) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    
    const days = Math.floor(hours / 24);
    if (days < 30) return `${days} day${days > 1 ? 's' : ''} ago`;
    
    const months = Math.floor(days / 30);
    if (months < 12) return `${months} month${months > 1 ? 's' : ''} ago`;
    
    const years = Math.floor(days / 365);
    return `${years} year${years > 1 ? 's' : ''} ago`;
};

const getDaysSince = (dateString) => {
    if (!dateString) return 0;
    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now - date);
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return diffDays;
};

export default function EditPost({ post, categories, authors, tags, flash }) {
    
    const [activeTab, setActiveTab] = useState("content");
    const [selectedTags, setSelectedTags] = useState(post?.tags || []);
    const [showTagDropdown, setShowTagDropdown] = useState(false);
    const [tagSearch, setTagSearch] = useState("");
    const [tocEntries, setTocEntries] = useState(post?.table_of_contents || []);
    const [metaFields, setMetaFields] = useState(post?.meta_data || {});
    const [socialFields, setSocialFields] = useState(post?.social_sharing || {});
    const [analyticsFields, setAnalyticsFields] = useState(
        post?.analytics || {}
    );
    const [structuredDataFields, setStructuredDataFields] = useState(
        post?.structured_data || {}
    );
    const [imageUploading, setImageUploading] = useState(false);
    const [seoImageUploading, setSeoImageUploading] = useState(false);
    const [showToast, setShowToast] = useState(false);
    const [toastMessage, setToastMessage] = useState("");
    
    // Get flash messages from Inertia page props as fallback
    const page = usePage();
    const pageFlash = page.props.flash || flash;

    const { data, setData, put, processing, errors } = useForm({
        title: post?.title || "",
        slug: post?.slug || "",
        subtitle: post?.subtitle || "",
        excerpt: post?.excerpt || "",
        content: post?.content || "",
        featured_image: post?.featured_image || "",
        featured_image_alt: post?.featured_image_alt || "",
        category_id: post?.category_id || "",
        author_id: post?.author_id || "",
        status: post?.status || "draft",
        published_at: post?.published_at
            ? new Date(post.published_at).toISOString().slice(0, 16)
            : "",
        read_time: post?.read_time || 5,
        difficulty: post?.difficulty || "beginner",
        featured: post?.featured || false,
        enable_comments:
            post?.enable_comments !== undefined ? post.enable_comments : true,
        enable_ratings:
            post?.enable_ratings !== undefined ? post.enable_ratings : true,
        meta_data: post?.meta_data || {},
        structured_data: post?.structured_data || {},
        social_sharing: post?.social_sharing || {},
        analytics: post?.analytics || {},
        table_of_contents: post?.table_of_contents || [],
    });

    // Set page title directly
    useEffect(() => {
        document.title = `Edit Post: ${post?.title || "Loading..."}`;

        return () => {
            document.title = "Admin Dashboard";
        };
    }, [post?.title]);

    // CRITICAL FIX: Sync separate state fields with useForm data
    // This ensures the useForm's internal data is always up-to-date
    useEffect(() => {
        setData('structured_data', structuredDataFields);
    }, [structuredDataFields]);

    useEffect(() => {
        setData('meta_data', metaFields);
    }, [metaFields]);

    useEffect(() => {
        setData('social_sharing', socialFields);
    }, [socialFields]);

    useEffect(() => {
        setData('analytics', analyticsFields);
    }, [analyticsFields]);

    useEffect(() => {
        setData('table_of_contents', tocEntries);
    }, [tocEntries]);

    // Show toast notification when there's a flash message
    useEffect(() => {
        console.log('Flash check:', pageFlash); // Debug log
        const message = pageFlash?.success || pageFlash?.message;
        if (message) {
            console.log('Success message:', message); // Debug log
            setToastMessage(message);
            setShowToast(true);
            const timer = setTimeout(() => {
                setShowToast(false);
            }, 4000); // Hide after 4 seconds

            return () => clearTimeout(timer);
        }
    }, [pageFlash, pageFlash?.success, pageFlash?.message]);

    // Auto-generate slug from title
    useEffect(() => {
        if (data.title && !data.slug) {
            const slug = data.title
                .toLowerCase()
                .replace(/[^a-z0-9]+/g, "-")
                .replace(/^-+|-+$/g, "");
            setData("slug", slug);
        }
    }, [data.title]);

    // Handle image upload
    const handleImageUpload = async (file) => {
        setImageUploading(true);
        const formData = new FormData();
        formData.append('featured_image', file);

        try {
            const response = await fetch('/admin/blog/upload-featured-image', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                },
                body: formData,
            });

            const result = await response.json();

            if (result.success) {
                setData('featured_image', result.image_url);
                // You can also store the image_path if needed
                // setData('featured_image_path', result.image_path);
            } else {
                alert('Failed to upload image: ' + result.message);
            }
        } catch (error) {
            console.error('Upload error:', error);
            alert('Failed to upload image. Please try again.');
        } finally {
            setImageUploading(false);
        }
    };

    // Handle SEO/OG image upload
    const handleSeoImageUpload = async (e) => {
        const file = e.target.files[0];
        if (!file) return;

        // Validate file type
        const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
        if (!validTypes.includes(file.type)) {
            alert('Please upload a valid image file (JPEG, PNG, GIF, or WebP)');
            return;
        }

        // Validate file size (5MB max)
        if (file.size > 5 * 1024 * 1024) {
            alert('Image size must be less than 5MB');
            return;
        }

        setSeoImageUploading(true);
        const formData = new FormData();
        formData.append('og_image', file);

        try {
            const response = await fetch('/admin/blog/upload-seo-image', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                },
                body: formData,
            });

            const result = await response.json();

            if (result.success) {
                setSocialFields({
                    ...socialFields,
                    og_image: result.image_url,
                    // Also update Twitter image if it's empty or same as OG image
                    twitter_image: socialFields.twitter_image && socialFields.twitter_image !== socialFields.og_image 
                        ? socialFields.twitter_image 
                        : result.image_url,
                });
            } else {
                alert('Failed to upload image: ' + result.message);
            }
        } catch (error) {
            console.error('Upload error:', error);
            alert('Failed to upload image. Please try again.');
        } finally {
            setSeoImageUploading(false);
            // Reset file input
            e.target.value = '';
        }
    };



    const handleSubmit = (e) => {
        e.preventDefault();

        // Convert datetime-local to proper ISO format with timezone
        let publishedAt = data.published_at;
        if (publishedAt && publishedAt.length === 16) {
            const localDate = new Date(publishedAt);
            publishedAt = localDate.toISOString();
        }

        console.log('=== FORM SUBMISSION ===');
        console.log('Using useForm data directly');
        console.log('data.structured_data:', data.structured_data);
        console.log('data.title:', data.title);

        // Use the submit method from useForm with updated published_at
        put(route("admin.blog.posts.update", post.id), {
            ...data,
            published_at: publishedAt,
            tag_ids: selectedTags.map((tag) => tag.id),
        }, {
            preserveScroll: false,
            preserveState: true,
            onSuccess: () => {
                window.location.href = route("admin.blog.posts.edit", post.id);
            },
        });
    };

    // Handle tag selection
    const handleTagToggle = (tag) => {
        const isSelected = selectedTags.some((t) => t.id === tag.id);
        if (isSelected) {
            setSelectedTags(selectedTags.filter((t) => t.id !== tag.id));
        } else {
            setSelectedTags([...selectedTags, tag]);
        }
    };

    // Filter tags based on search
    const filteredTags = tags.filter((tag) =>
        tag.name.toLowerCase().includes(tagSearch.toLowerCase())
    );

    // TOC Management Functions
    const addTocEntry = () => {
        setTocEntries([...tocEntries, { id: "", title: "", level: 1 }]);
    };

    const removeTocEntry = (index) => {
        setTocEntries(tocEntries.filter((_, i) => i !== index));
    };

    const updateTocEntry = (index, field, value) => {
        const updated = [...tocEntries];
        updated[index][field] = value;
        setTocEntries(updated);
    };

    // Tab content configuration
    const tabs = [
        { id: "content", label: "Content", icon: FileText },
        { id: "media", label: "Media", icon: Image },
        { id: "taxonomy", label: "Categories & Tags", icon: Tag },
        { id: "settings", label: "Settings", icon: Settings },
        { id: "seo", label: "SEO & Meta", icon: Globe },
        { id: "structured-data", label: "Structured Data", icon: Database },
        { id: "advanced", label: "Advanced", icon: Settings },
    ];

    return (
        <AdminLayout>
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 sm:py-8">
                {/* Mobile-First Responsive Header */}
                <div className="mb-6 sm:mb-8">
                    {/* Mobile Layout - Stacked */}
                    <div className="flex flex-col space-y-4 sm:hidden">
                        <div className="flex items-center justify-between">
                            <Link
                                href={route("admin.blog.posts")}
                                className="inline-flex items-center px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                            >
                                <ArrowLeft className="w-4 h-4 mr-1" />
                                <span className="hidden xs:inline">Back</span>
                            </Link>
                            <div className="flex items-center space-x-2">
                                <Link
                                    href={`/blog/${post.slug}`}
                                    target="_blank"
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                >
                                    <Eye className="w-4 h-4" />
                                </Link>
                                <button
                                    onClick={handleSubmit}
                                    disabled={processing}
                                    className="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                                >
                                    <Save className="w-4 h-4 mr-1" />
                                    {processing ? "Saving..." : "Save"}
                                </button>
                            </div>
                        </div>
                        <div>
                            <h1 className="text-xl font-bold text-gray-900 dark:text-white truncate">
                                Edit Post
                            </h1>
                            <p className="text-sm text-gray-600 dark:text-gray-400 truncate">
                                {post.title}
                            </p>
                        </div>
                    </div>

                    {/* Desktop Layout */}
                    <div className="hidden sm:flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                            <Link
                                href={route("admin.blog.posts")}
                                className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                            >
                                <ArrowLeft className="w-4 h-4 mr-2" />
                                Back to Posts
                            </Link>
                            <div>
                                <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                    Edit Post
                                </h1>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    {post.title}
                                </p>
                            </div>
                        </div>
                        <div className="flex items-center space-x-3">
                            <a
                                href={`/blog/${post.slug}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                            >
                                <Eye className="w-4 h-4 mr-2" />
                                Preview
                            </a>
                            <button
                                onClick={handleSubmit}
                                disabled={processing}
                                className="inline-flex items-center px-6 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                            >
                                <Save className="w-4 h-4 mr-2" />
                                {processing ? "Saving..." : "Save Changes"}
                            </button>
                        </div>
                    </div>
                </div>

                <form onSubmit={handleSubmit}>
                    {/* Mobile-First Grid: Stack on mobile, side-by-side on desktop */}
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-6 lg:gap-8">
                        {/* Main Content Area */}
                        <div className="order-2 lg:order-1 lg:col-span-8">
                            {/* Tab Navigation */}
                            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 mb-6">
                                <div className="flex flex-wrap gap-2 p-4">
                                    {tabs.map((tab) => {
                                        const Icon = tab.icon;
                                        return (
                                            <button
                                                key={tab.id}
                                                type="button"
                                                onClick={() => setActiveTab(tab.id)}
                                                className={`inline-flex items-center px-4 py-2 rounded-md text-sm font-medium transition-colors ${
                                                    activeTab === tab.id
                                                        ? "bg-blue-100 text-blue-700 dark:bg-blue-900/30 dark:text-blue-400"
                                                        : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                                                }`}
                                            >
                                                <Icon className="w-4 h-4 mr-2" />
                                                {tab.label}
                                            </button>
                                        );
                                    })}
                                </div>
                            </div>

                            {/* Tab Content Panels */}
                            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 sm:p-6">
                                {/* Content Tab */}
                                {activeTab === "content" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Title */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Title *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.title}
                                                onChange={(e) =>
                                                    setData(
                                                        "title",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="Enter post title"
                                                required
                                            />
                                            {errors.title && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.title}
                                                </p>
                                            )}
                                        </div>

                                        {/* Slug */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Slug *
                                            </label>
                                            <input
                                                type="text"
                                                value={data.slug}
                                                onChange={(e) =>
                                                    setData(
                                                        "slug",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="post-slug"
                                                required
                                            />
                                            {errors.slug && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.slug}
                                                </p>
                                            )}
                                        </div>

                                        {/* Subtitle */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Subtitle
                                            </label>
                                            <input
                                                type="text"
                                                value={data.subtitle}
                                                onChange={(e) =>
                                                    setData(
                                                        "subtitle",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="Optional subtitle"
                                            />
                                        </div>

                                        {/* Excerpt */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Excerpt *
                                            </label>
                                            <textarea
                                                value={data.excerpt}
                                                onChange={(e) =>
                                                    setData(
                                                        "excerpt",
                                                        e.target.value
                                                    )
                                                }
                                                rows={3}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="Brief description of the post"
                                                required
                                            />
                                            {errors.excerpt && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.excerpt}
                                                </p>
                                            )}
                                        </div>

                                        {/* Content */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Content *
                                            </label>
                                            <RichTextEditor
                                                value={data.content || ""}
                                                onChange={(content) => setData("content", content)}
                                                height={500}
                                            />
                                            {errors.content && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.content}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                )}

                                {/* Media Tab */}
                                {activeTab === "media" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Featured Image Upload */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Featured Image
                                            </label>
                                            <div className="space-y-4">
                                                {/* File Upload */}
                                                <div className="flex items-center space-x-4">
                                                    <input
                                                        type="file"
                                                        accept="image/*"
                                                        disabled={imageUploading}
                                                        onChange={(e) => {
                                                            const file = e.target.files[0];
                                                            if (file) {
                                                                handleImageUpload(file);
                                                            }
                                                        }}
                                                        className="block w-full text-sm text-gray-500 dark:text-gray-400 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 dark:file:bg-blue-900/30 dark:file:text-blue-300 disabled:opacity-50"
                                                    />
                                                    {imageUploading && (
                                                        <div className="flex items-center text-blue-600 dark:text-blue-400">
                                                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600"></div>
                                                            <span className="ml-2 text-sm">Uploading...</span>
                                                        </div>
                                                    )}
                                                </div>
                                                
                                                {/* URL Input (fallback) */}
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                        Or enter image URL
                                                    </label>
                                                    <input
                                                        type="url"
                                                        value={data.featured_image}
                                                        onChange={(e) =>
                                                            setData(
                                                                "featured_image",
                                                                e.target.value
                                                            )
                                                        }
                                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                        placeholder="https://example.com/image.jpg"
                                                    />
                                                </div>
                                                
                                                {/* Image Preview */}
                                                {data.featured_image && (
                                                    <div className="mt-4">
                                                        <img
                                                            src={data.featured_image}
                                                            alt="Featured image preview"
                                                            className="w-full max-w-md h-48 object-cover rounded-lg border border-gray-200 dark:border-gray-700"
                                                        />
                                                    </div>
                                                )}
                                            </div>
                                        </div>

                                        {/* Featured Image Alt Text */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Featured Image Alt Text
                                            </label>
                                            <input
                                                type="text"
                                                value={data.featured_image_alt}
                                                onChange={(e) =>
                                                    setData(
                                                        "featured_image_alt",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="Describe the image for accessibility"
                                            />
                                        </div>
                                    </div>
                                )}

                                {/* Taxonomy Tab */}
                                {activeTab === "taxonomy" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Category */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Category *
                                            </label>
                                            <select
                                                value={data.category_id}
                                                onChange={(e) =>
                                                    setData(
                                                        "category_id",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                required
                                            >
                                                <option value="">
                                                    Select a category
                                                </option>
                                                {categories.map((category) => (
                                                    <option
                                                        key={category.id}
                                                        value={category.id}
                                                    >
                                                        {category.name}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.category_id && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.category_id}
                                                </p>
                                            )}
                                        </div>

                                        {/* Tags */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Tags
                                            </label>
                                            <div className="relative">
                                                <input
                                                    type="text"
                                                    value={tagSearch}
                                                    onChange={(e) =>
                                                        setTagSearch(
                                                            e.target.value
                                                        )
                                                    }
                                                    onFocus={() =>
                                                        setShowTagDropdown(true)
                                                    }
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                    placeholder="Search tags..."
                                                />
                                                {showTagDropdown && (
                                                    <div className="absolute z-10 w-full mt-1 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md shadow-lg max-h-60 overflow-y-auto">
                                                        {filteredTags.map(
                                                            (tag) => (
                                                                <button
                                                                    key={tag.id}
                                                                    type="button"
                                                                    onClick={() =>
                                                                        handleTagToggle(
                                                                            tag
                                                                        )
                                                                    }
                                                                    className={`w-full text-left px-3 py-2 hover:bg-gray-100 dark:hover:bg-gray-700 text-sm sm:text-base ${
                                                                        selectedTags.some(
                                                                            (
                                                                                t
                                                                            ) =>
                                                                                t.id ===
                                                                                tag.id
                                                                        )
                                                                            ? "bg-blue-50 dark:bg-blue-900/30 text-blue-700 dark:text-blue-400"
                                                                            : "text-gray-700 dark:text-gray-300"
                                                                    }`}
                                                                >
                                                                    {tag.name}
                                                                </button>
                                                            )
                                                        )}
                                                    </div>
                                                )}
                                            </div>
                                            {selectedTags.length > 0 && (
                                                <div className="mt-3 flex flex-wrap gap-2">
                                                    {selectedTags.map((tag) => (
                                                        <span
                                                            key={tag.id}
                                                            className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400"
                                                        >
                                                            {tag.name}
                                                            <button
                                                                type="button"
                                                                onClick={() =>
                                                                    handleTagToggle(
                                                                        tag
                                                                    )
                                                                }
                                                                className="ml-1 inline-flex items-center justify-center w-4 h-4 rounded-full hover:bg-blue-200 dark:hover:bg-blue-800"
                                                            >
                                                                <X className="w-3 h-3" />
                                                            </button>
                                                        </span>
                                                    ))}
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                )}

                                {/* Settings Tab */}
                                {activeTab === "settings" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Author */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Author *
                                            </label>
                                            <select
                                                value={data.author_id}
                                                onChange={(e) =>
                                                    setData(
                                                        "author_id",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                required
                                            >
                                                <option value="">
                                                    Select an author
                                                </option>
                                                {authors.map((author) => (
                                                    <option
                                                        key={author.id}
                                                        value={author.id}
                                                    >
                                                        {author.name}
                                                    </option>
                                                ))}
                                            </select>
                                            {errors.author_id && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.author_id}
                                                </p>
                                            )}
                                        </div>

                                        {/* Status */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Status
                                            </label>
                                            <select
                                                value={data.status}
                                                onChange={(e) =>
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                            >
                                                <option value="draft">
                                                    Draft
                                                </option>
                                                <option value="published">
                                                    Published
                                                </option>
                                                <option value="archived">
                                                    Archived
                                                </option>
                                            </select>
                                        </div>

                                        {/* Published Date */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Published Date
                                            </label>
                                            <input
                                                type="datetime-local"
                                                value={data.published_at}
                                                onChange={(e) =>
                                                    setData(
                                                        "published_at",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                            />
                                        </div>

                                        {/* Read Time */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Read Time (minutes)
                                            </label>
                                            <input
                                                type="number"
                                                value={data.read_time}
                                                onChange={(e) =>
                                                    setData(
                                                        "read_time",
                                                        parseInt(e.target.value)
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                min="1"
                                                max="60"
                                            />
                                        </div>

                                        {/* Difficulty */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Difficulty Level
                                            </label>
                                            <select
                                                value={data.difficulty}
                                                onChange={(e) =>
                                                    setData(
                                                        "difficulty",
                                                        e.target.value
                                                    )
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                            >
                                                <option value="beginner">
                                                    Beginner
                                                </option>
                                                <option value="intermediate">
                                                    Intermediate
                                                </option>
                                                <option value="advanced">
                                                    Advanced
                                                </option>
                                            </select>
                                        </div>

                                        {/* Featured Post */}
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                id="featured"
                                                checked={data.featured}
                                                onChange={(e) =>
                                                    setData(
                                                        "featured",
                                                        e.target.checked
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <label
                                                htmlFor="featured"
                                                className="ml-2 text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Featured Post
                                            </label>
                                        </div>

                                        {/* Enable Comments */}
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                id="enable_comments"
                                                checked={data.enable_comments}
                                                onChange={(e) =>
                                                    setData(
                                                        "enable_comments",
                                                        e.target.checked
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <label
                                                htmlFor="enable_comments"
                                                className="ml-2 text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Enable Comments
                                            </label>
                                        </div>

                                        {/* Enable Ratings */}
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                id="enable_ratings"
                                                checked={data.enable_ratings}
                                                onChange={(e) =>
                                                    setData(
                                                        "enable_ratings",
                                                        e.target.checked
                                                    )
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <label
                                                htmlFor="enable_ratings"
                                                className="ml-2 text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Enable Ratings
                                            </label>
                                        </div>
                                    </div>
                                )}

                                {/* SEO Tab */}
                                {activeTab === "seo" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Meta Title */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Meta Title
                                            </label>
                                            <input
                                                type="text"
                                                value={metaFields.title || ""}
                                                onChange={(e) =>
                                                    setMetaFields({
                                                        ...metaFields,
                                                        title: e.target.value,
                                                    })
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="SEO title for search engines"
                                            />
                                        </div>

                                        {/* Meta Description */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Meta Description
                                            </label>
                                            <textarea
                                                value={
                                                    metaFields.description || ""
                                                }
                                                onChange={(e) =>
                                                    setMetaFields({
                                                        ...metaFields,
                                                        description:
                                                            e.target.value,
                                                    })
                                                }
                                                rows={3}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="SEO description for search engines"
                                            />
                                        </div>

                                        {/* Meta Keywords */}
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Meta Keywords
                                            </label>
                                            <input
                                                type="text"
                                                value={
                                                    metaFields.keywords || ""
                                                }
                                                onChange={(e) =>
                                                    setMetaFields({
                                                        ...metaFields,
                                                        keywords:
                                                            e.target.value,
                                                    })
                                                }
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                placeholder="keyword1, keyword2, keyword3"
                                            />
                                        </div>

                                        {/* Social Sharing / Open Graph */}
                                        <div>
                                            <h4 className="font-medium text-gray-900 dark:text-white mb-4">
                                                Social Sharing & Open Graph
                                            </h4>
                                            <div className="space-y-4">
                                                {/* Open Graph Section */}
                                                <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                                    <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-3">
                                                        Open Graph (Facebook, LinkedIn, etc.)
                                                    </h5>
                                                    <div className="space-y-4">
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                OG Title
                                                            </label>
                                                            <input
                                                                type="text"
                                                                value={
                                                                    socialFields.og_title ||
                                                                    socialFields.facebook_title ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        og_title: e.target.value,
                                                                        facebook_title: e.target.value, // Sync with Facebook title
                                                                    })
                                                                }
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Open Graph title (for social sharing)"
                                                            />
                                                        </div>
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                OG Description
                                                            </label>
                                                            <textarea
                                                                value={
                                                                    socialFields.og_description ||
                                                                    socialFields.facebook_description ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        og_description: e.target.value,
                                                                        facebook_description: e.target.value, // Sync with Facebook description
                                                                    })
                                                                }
                                                                rows={2}
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Open Graph description (for social sharing)"
                                                            />
                                                        </div>
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                OG Image (Used for Twitter too)
                                                            </label>
                                                            <div className="space-y-3">
                                                                {/* Image Upload */}
                                                                <div className="flex items-center gap-3">
                                                                    <label className="flex items-center justify-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors text-sm">
                                                                        <Upload className="h-4 w-4 mr-2" />
                                                                        <span>{seoImageUploading ? 'Uploading...' : 'Upload Image'}</span>
                                                                        <input
                                                                            type="file"
                                                                            accept="image/jpeg,image/png,image/jpg,image/gif,image/webp"
                                                                            className="hidden"
                                                                            onChange={handleSeoImageUpload}
                                                                            disabled={seoImageUploading}
                                                                        />
                                                                    </label>
                                                                    {socialFields.og_image && (
                                                                        <button
                                                                            type="button"
                                                                            onClick={() =>
                                                                                setSocialFields({
                                                                                    ...socialFields,
                                                                                    og_image: '',
                                                                                })
                                                                            }
                                                                            className="px-3 py-2 border border-red-300 dark:border-red-600 rounded-md text-red-600 dark:text-red-400 hover:bg-red-50 dark:hover:bg-red-900/20 transition-colors text-sm"
                                                                        >
                                                                            <X className="h-4 w-4 inline mr-1" />
                                                                            Clear
                                                                        </button>
                                                                    )}
                                                                </div>
                                                                
                                                                {/* Image Preview */}
                                                                {socialFields.og_image && (
                                                                    <div className="relative border border-gray-300 dark:border-gray-600 rounded-md p-3 bg-gray-50 dark:bg-gray-800">
                                                                        <img
                                                                            src={socialFields.og_image.startsWith('http') ? socialFields.og_image : `${window.location.origin}${socialFields.og_image}`}
                                                                            alt="OG Image Preview"
                                                                            className="max-w-full h-auto max-h-48 rounded-md"
                                                                            onError={(e) => {
                                                                                e.target.style.display = 'none';
                                                                            }}
                                                                        />
                                                                    </div>
                                                                )}
                                                                
                                                                {/* URL Input (fallback) */}
                                                                <div>
                                                                    <input
                                                                        type="url"
                                                                        value={
                                                                            socialFields.og_image ||
                                                                            ""
                                                                        }
                                                                        onChange={(e) =>
                                                                            setSocialFields({
                                                                                ...socialFields,
                                                                                og_image: e.target.value,
                                                                            })
                                                                        }
                                                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                        placeholder="Or enter image URL: https://example.com/og-image.jpg"
                                                                    />
                                                                    <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                                        Recommended: 1200x630px. This image will be used for both Open Graph and Twitter Card.
                                                                    </p>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                OG Image Alt Text
                                                            </label>
                                                            <input
                                                                type="text"
                                                                value={
                                                                    socialFields.og_image_alt ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        og_image_alt: e.target.value,
                                                                    })
                                                                }
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Describe the OG image"
                                                            />
                                                        </div>
                                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    OG URL
                                                                </label>
                                                                <input
                                                                    type="url"
                                                                    value={
                                                                        socialFields.og_url ||
                                                                        ""
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            og_url: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                    placeholder="Leave empty to use post URL"
                                                                />
                                                            </div>
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    OG Site Name
                                                                </label>
                                                                <input
                                                                    type="text"
                                                                    value={
                                                                        socialFields.og_site_name ||
                                                                        "Academic Scribe"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            og_site_name: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                    placeholder="Academic Scribe"
                                                                />
                                                            </div>
                                                        </div>
                                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    OG Type
                                                                </label>
                                                                <select
                                                                    value={
                                                                        socialFields.og_type ||
                                                                        "article"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            og_type: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                >
                                                                    <option value="article">Article</option>
                                                                    <option value="website">Website</option>
                                                                    <option value="blog">Blog</option>
                                                                </select>
                                                            </div>
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    OG Locale
                                                                </label>
                                                                <select
                                                                    value={
                                                                        socialFields.og_locale ||
                                                                        "en_US"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            og_locale: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                >
                                                                    <option value="en_US">English (US)</option>
                                                                    <option value="en_GB">English (UK)</option>
                                                                    <option value="es_ES">Spanish</option>
                                                                    <option value="fr_FR">French</option>
                                                                </select>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Facebook Section (Legacy - kept for backward compatibility) */}
                                                <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                                    <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-3">
                                                        Facebook (Legacy - use OG fields above)
                                                    </h5>
                                                    <div className="space-y-4">
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                Facebook Title
                                                            </label>
                                                            <input
                                                                type="text"
                                                                value={
                                                                    socialFields.facebook_title ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        facebook_title: e.target.value,
                                                                        og_title: e.target.value, // Sync with OG title
                                                                    })
                                                                }
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Facebook share title"
                                                            />
                                                        </div>
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                Facebook Description
                                                            </label>
                                                            <textarea
                                                                value={
                                                                    socialFields.facebook_description ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        facebook_description: e.target.value,
                                                                        og_description: e.target.value, // Sync with OG description
                                                                    })
                                                                }
                                                                rows={2}
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Facebook share description"
                                                            />
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Twitter Section */}
                                                <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                                    <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-3">
                                                        Twitter Card
                                                    </h5>
                                                    <div className="space-y-4">
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                Twitter Title
                                                            </label>
                                                            <input
                                                                type="text"
                                                                value={
                                                                    socialFields.twitter_title ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        twitter_title: e.target.value,
                                                                    })
                                                                }
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Twitter share title"
                                                            />
                                                        </div>
                                                        <div>
                                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                Twitter Description
                                                            </label>
                                                            <textarea
                                                                value={
                                                                    socialFields.twitter_description ||
                                                                    ""
                                                                }
                                                                onChange={(e) =>
                                                                    setSocialFields({
                                                                        ...socialFields,
                                                                        twitter_description: e.target.value,
                                                                    })
                                                                }
                                                                rows={2}
                                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                placeholder="Twitter share description"
                                                            />
                                                        </div>
                                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    Twitter Card Type
                                                                </label>
                                                                <select
                                                                    value={
                                                                        socialFields.twitter_card ||
                                                                        "summary_large_image"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            twitter_card: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                >
                                                                    <option value="summary">Summary</option>
                                                                    <option value="summary_large_image">Summary Large Image</option>
                                                                </select>
                                                            </div>
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    Twitter Site Handle
                                                                </label>
                                                                <input
                                                                    type="text"
                                                                    value={
                                                                        socialFields.twitter_site ||
                                                                        "@AcademicScribe"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            twitter_site: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                    placeholder="@AcademicScribe"
                                                                />
                                                            </div>
                                                        </div>
                                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    Twitter Image URL
                                                                </label>
                                                                <input
                                                                    type="url"
                                                                    value={
                                                                        socialFields.twitter_image ||
                                                                        ""
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            twitter_image: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                    placeholder="Leave empty to use OG image"
                                                                />
                                                            </div>
                                                            <div>
                                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                                    Twitter Creator Handle
                                                                </label>
                                                                <input
                                                                    type="text"
                                                                    value={
                                                                        socialFields.twitter_creator ||
                                                                        "@AcademicScribe"
                                                                    }
                                                                    onChange={(e) =>
                                                                        setSocialFields({
                                                                            ...socialFields,
                                                                            twitter_creator: e.target.value,
                                                                        })
                                                                    }
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                                    placeholder="@AcademicScribe"
                                                                />
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}

                                {/* Structured Data Tab */}
                                {activeTab === "structured-data" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        <div>
                                            <div className="flex items-center justify-between mb-4">
                                                <h4 className="font-medium text-gray-900 dark:text-white">
                                                    Structured Data (JSON-LD)
                                                </h4>
                                                <button
                                                    type="button"
                                                    onClick={() => {
                                                        try {
                                                            const formatted = JSON.stringify(
                                                                JSON.parse(JSON.stringify(structuredDataFields) || "{}"),
                                                                null,
                                                                2
                                                            );
                                                            setStructuredDataFields(JSON.parse(formatted));
                                                        } catch (e) {
                                                            // If parsing fails, just continue
                                                        }
                                                    }}
                                                    className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-blue-700 bg-blue-100 hover:bg-blue-200 dark:bg-blue-900/30 dark:text-blue-400 dark:hover:bg-blue-900/50"
                                                >
                                                    Format JSON
                                                </button>
                                            </div>
                                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                                                Add structured data markup to help search engines understand your content better. 
                                                Common types include Article, BlogPosting, FAQ, HowTo, etc.
                                            </p>
                                            
                                            {/* Common Structured Data Templates */}
                                            <div className="mb-4">
                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                    Quick Templates
                                                </label>
                                                <div className="flex flex-wrap gap-2">
                                                    <button
                                                        type="button"
                                                        onClick={() => {
                                                            const authorName = authors.find(a => a.id === data.author_id)?.name || post?.author?.name || "Academic Scribe";
                                                            const publishedDate = data.published_at || new Date().toISOString();
                                                            const imageUrl = data.featured_image ? 
                                                                (data.featured_image.startsWith('http') ? data.featured_image : `https://academicscribe.com${data.featured_image}`) 
                                                                : 'https://academicscribe.com/images/logo.png';
                                                            
                                                            const articleTemplate = {
                                                                "@context": "https://schema.org",
                                                                "@type": "Article",
                                                                "headline": data.title || "Article Headline",
                                                                "description": data.excerpt || "Article description",
                                                                "image": imageUrl,
                                                                "author": {
                                                                    "@type": "Person",
                                                                    "name": authorName
                                                                },
                                                                "datePublished": publishedDate,
                                                                "dateModified": new Date().toISOString(),
                                                                "publisher": {
                                                                    "@type": "Organization",
                                                                    "name": "Academic Scribe",
                                                                    "logo": {
                                                                        "@type": "ImageObject",
                                                                        "url": "https://academicscribe.com/images/logo.png"
                                                                    }
                                                                },
                                                                "mainEntityOfPage": {
                                                                    "@type": "WebPage",
                                                                    "@id": `https://academicscribe.com/blog/${data.slug || 'post'}`
                                                                }
                                                            };
                                                            setStructuredDataFields(articleTemplate);
                                                        }}
                                                        className="inline-flex items-center px-3 py-1 border border-gray-300 dark:border-gray-600 rounded-md text-xs font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                                    >
                                                        Article
                                                    </button>
                                                    <button
                                                        type="button"
                                                        onClick={() => {
                                                            const authorName = authors.find(a => a.id === data.author_id)?.name || post?.author?.name || "Academic Scribe";
                                                            const publishedDate = data.published_at || new Date().toISOString();
                                                            const imageUrl = data.featured_image ? 
                                                                (data.featured_image.startsWith('http') ? data.featured_image : `https://academicscribe.com${data.featured_image}`) 
                                                                : 'https://academicscribe.com/images/logo.png';
                                                            
                                                            const blogPostTemplate = {
                                                                "@context": "https://schema.org",
                                                                "@type": "BlogPosting",
                                                                "headline": data.title || "Blog Post Headline",
                                                                "description": data.excerpt || "Blog post description",
                                                                "image": imageUrl,
                                                                "author": {
                                                                    "@type": "Person",
                                                                    "name": authorName
                                                                },
                                                                "datePublished": publishedDate,
                                                                "dateModified": new Date().toISOString(),
                                                                "publisher": {
                                                                    "@type": "Organization",
                                                                    "name": "Academic Scribe",
                                                                    "logo": {
                                                                        "@type": "ImageObject",
                                                                        "url": "https://academicscribe.com/images/logo.png"
                                                                    }
                                                                },
                                                                "mainEntityOfPage": {
                                                                    "@type": "WebPage",
                                                                    "@id": `https://academicscribe.com/blog/${data.slug || 'post'}`
                                                                }
                                                            };
                                                            setStructuredDataFields(blogPostTemplate);
                                                        }}
                                                        className="inline-flex items-center px-3 py-1 border border-gray-300 dark:border-gray-600 rounded-md text-xs font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                                    >
                                                        BlogPosting
                                                    </button>
                                                    <button
                                                        type="button"
                                                        onClick={() => {
                                                            const faqTemplate = {
                                                                "@context": "https://schema.org",
                                                                "@type": "FAQPage",
                                                                "mainEntity": [
                                                                    {
                                                                        "@type": "Question",
                                                                        "name": "Sample Question?",
                                                                        "acceptedAnswer": {
                                                                            "@type": "Answer",
                                                                            "text": "Sample answer text here."
                                                                        }
                                                                    }
                                                                ]
                                                            };
                                                            setStructuredDataFields(faqTemplate);
                                                        }}
                                                        className="inline-flex items-center px-3 py-1 border border-gray-300 dark:border-gray-600 rounded-md text-xs font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                                                    >
                                                        FAQ
                                                    </button>
                                                    <button
                                                        type="button"
                                                        onClick={() => {
                                                            setStructuredDataFields({});
                                                        }}
                                                        className="inline-flex items-center px-3 py-1 border border-red-300 dark:border-red-600 rounded-md text-xs font-medium text-red-700 dark:text-red-400 bg-red-50 dark:bg-red-900/30 hover:bg-red-100 dark:hover:bg-red-900/50 transition-colors"
                                                    >
                                                        Clear
                                                    </button>
                                                </div>
                                            </div>

                                            {/* JSON Editor */}
                                            <div>
                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                    Structured Data JSON
                                                </label>
                                                <textarea
                                                    value={typeof structuredDataFields === 'string' 
                                                        ? structuredDataFields 
                                                        : JSON.stringify(structuredDataFields, null, 2)}
                                                    onChange={(e) => {
                                                        try {
                                                            const parsed = JSON.parse(e.target.value);
                                                            setStructuredDataFields(parsed);
                                                        } catch (error) {
                                                            // Allow invalid JSON while typing (keep as string temporarily)
                                                            setStructuredDataFields(e.target.value);
                                                        }
                                                    }}
                                                    rows={20}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm font-mono"
                                                    placeholder='{\n  "@context": "https://schema.org",\n  "@type": "Article",\n  "headline": "Your Article Title"\n}'
                                                />
                                                <div className="mt-2 flex items-center space-x-4">
                                                    <div className="flex items-center">
                                                        {(() => {
                                                            try {
                                                                // Handle both string and object types
                                                                const dataToValidate = typeof structuredDataFields === 'string' 
                                                                    ? structuredDataFields 
                                                                    : JSON.stringify(structuredDataFields);
                                                                JSON.parse(dataToValidate);
                                                                return (
                                                                    <div className="flex items-center text-green-600 dark:text-green-400">
                                                                        <div className="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                                                                        <span className="text-xs">Valid JSON</span>
                                                                    </div>
                                                                );
                                                            } catch (e) {
                                                                return (
                                                                    <div className="flex items-center text-red-600 dark:text-red-400">
                                                                        <AlertCircle className="w-4 h-4 mr-1" />
                                                                        <span className="text-xs">Invalid JSON</span>
                                                                    </div>
                                                                );
                                                            }
                                                        })()}
                                                    </div>
                                                    <a
                                                        href="https://developers.google.com/search/docs/appearance/structured-data/intro-structured-data"
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                        className="text-xs text-blue-600 dark:text-blue-400 hover:underline"
                                                    >
                                                        Learn about structured data
                                                    </a>
                                                    <a
                                                        href="https://search.google.com/test/rich-results"
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                        className="text-xs text-blue-600 dark:text-blue-400 hover:underline"
                                                    >
                                                        Test with Google
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}

                                {/* Advanced Tab */}
                                {activeTab === "advanced" && (
                                    <div className="space-y-4 sm:space-y-6">
                                        {/* Table of Contents */}
                                        <div>
                                            <div className="flex items-center justify-between mb-4">
                                                <h4 className="font-medium text-gray-900 dark:text-white">
                                                    Table of Contents
                                                </h4>
                                                <button
                                                    type="button"
                                                    onClick={addTocEntry}
                                                    className="inline-flex items-center px-3 py-1 border border-transparent text-xs font-medium rounded-md text-blue-700 bg-blue-100 hover:bg-blue-200 dark:bg-blue-900/30 dark:text-blue-400 dark:hover:bg-blue-900/50"
                                                >
                                                    <Plus className="w-3 h-3 mr-1" />
                                                    Add Entry
                                                </button>
                                            </div>
                                            <div className="space-y-3">
                                                {tocEntries.map(
                                                    (entry, index) => (
                                                        <div
                                                            key={index}
                                                            className="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-3 p-3 border border-gray-200 dark:border-gray-700 rounded-md"
                                                        >
                                                            <select
                                                                value={
                                                                    entry.level
                                                                }
                                                                onChange={(e) =>
                                                                    updateTocEntry(
                                                                        index,
                                                                        "level",
                                                                        parseInt(
                                                                            e
                                                                                .target
                                                                                .value
                                                                        )
                                                                    )
                                                                }
                                                                className="px-2 py-1 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm"
                                                            >
                                                                <option
                                                                    value={1}
                                                                >
                                                                    H1
                                                                </option>
                                                                <option
                                                                    value={2}
                                                                >
                                                                    H2
                                                                </option>
                                                                <option
                                                                    value={3}
                                                                >
                                                                    H3
                                                                </option>
                                                                <option
                                                                    value={4}
                                                                >
                                                                    H4
                                                                </option>
                                                            </select>
                                                            <input
                                                                type="text"
                                                                value={
                                                                    entry.title
                                                                }
                                                                onChange={(e) =>
                                                                    updateTocEntry(
                                                                        index,
                                                                        "title",
                                                                        e.target
                                                                            .value
                                                                    )
                                                                }
                                                                className="flex-1 px-3 py-1 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm"
                                                                placeholder="Section title"
                                                            />
                                                            <input
                                                                type="text"
                                                                value={entry.id}
                                                                onChange={(e) =>
                                                                    updateTocEntry(
                                                                        index,
                                                                        "id",
                                                                        e.target
                                                                            .value
                                                                    )
                                                                }
                                                                className="w-full sm:w-32 px-3 py-1 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm"
                                                                placeholder="anchor-id"
                                                            />
                                                            <button
                                                                type="button"
                                                                onClick={() =>
                                                                    removeTocEntry(
                                                                        index
                                                                    )
                                                                }
                                                                className="p-1 text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300 self-end sm:self-center"
                                                            >
                                                                <X className="w-4 h-4" />
                                                            </button>
                                                        </div>
                                                    )
                                                )}
                                            </div>
                                        </div>

                                        {/* Analytics */}
                                        <div>
                                            <h4 className="font-medium text-gray-900 dark:text-white mb-4">
                                                Analytics
                                            </h4>
                                            <div className="space-y-4">
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                        Google Analytics Event
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={
                                                            analyticsFields.ga_event ||
                                                            ""
                                                        }
                                                        onChange={(e) =>
                                                            setAnalyticsFields({
                                                                ...analyticsFields,
                                                                ga_event:
                                                                    e.target
                                                                        .value,
                                                            })
                                                        }
                                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                        placeholder="page_view"
                                                    />
                                                </div>
                                                <div>
                                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                        Custom Tracking ID{" "}
                                                    </label>
                                                    <input
                                                        type="text"
                                                        value={
                                                            analyticsFields.tracking_id ||
                                                            ""
                                                        }
                                                        onChange={(e) =>
                                                            setAnalyticsFields({
                                                                ...analyticsFields,
                                                                tracking_id:
                                                                    e.target
                                                                        .value,
                                                            })
                                                        }
                                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white text-sm sm:text-base"
                                                        placeholder="Custom tracking identifier"
                                                    />
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Sidebar - Publish Panel, Quick Actions, Stats */}
                        <div className="order-1 lg:order-2 lg:col-span-4">
                            <div className="space-y-4 lg:space-y-6 lg:sticky lg:top-4">
                                {/* Publish Panel - Enhanced Timeline */}
                                <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 sm:p-6">
                                    <div className="flex items-center justify-between mb-4">
                                        <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                            Publish Timeline
                                        </h3>
                                        <span
                                            className={`inline-flex items-center px-2.5 py-1 rounded-full text-xs font-semibold ${
                                                post.status === "published"
                                                    ? "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400"
                                                    : post.status === "draft"
                                                    ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400"
                                                    : "bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400"
                                            }`}
                                        >
                                            {post.status.charAt(0).toUpperCase() + post.status.slice(1)}
                                        </span>
                                    </div>
                                    
                                    <div className="space-y-4">
                                        {/* Published Information */}
                                        {post.published_at && (
                                            <div className="pb-4 border-b border-gray-200 dark:border-gray-700">
                                                <div className="flex items-start">
                                                    <div className="flex-shrink-0">
                                                        <div className="flex items-center justify-center w-8 h-8 rounded-full bg-green-100 dark:bg-green-900/30">
                                                            <TrendingUp className="w-4 h-4 text-green-600 dark:text-green-400" />
                                                        </div>
                                                    </div>
                                                    <div className="ml-3 flex-1">
                                                        <p className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                            Published
                                                        </p>
                                                        <p className="mt-1 text-sm font-medium text-gray-900 dark:text-white">
                                                            {formatDateTime(post.published_at)}
                                                        </p>
                                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                            {getDaysSince(post.published_at)} days ago
                                                        </p>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                        
                                        {/* Last Updated Information */}
                                        <div className="pb-4 border-b border-gray-200 dark:border-gray-700">
                                            <div className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="flex items-center justify-center w-8 h-8 rounded-full bg-blue-100 dark:bg-blue-900/30">
                                                        <Clock className="w-4 h-4 text-blue-600 dark:text-blue-400" />
                                                    </div>
                                                </div>
                                                <div className="ml-3 flex-1">
                                                    <p className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                        Last Updated
                                                    </p>
                                                    <p className="mt-1 text-sm font-medium text-gray-900 dark:text-white">
                                                        {formatDateTime(post.updated_at)}
                                                    </p>
                                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                        {getTimeAgo(post.updated_at)}
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        {/* Author Information */}
                                        <div className="pb-4 border-b border-gray-200 dark:border-gray-700">
                                            <div className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="flex items-center justify-center w-8 h-8 rounded-full bg-purple-100 dark:bg-purple-900/30">
                                                        <User className="w-4 h-4 text-purple-600 dark:text-purple-400" />
                                                    </div>
                                                </div>
                                                <div className="ml-3 flex-1">
                                                    <p className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                        Author
                                                    </p>
                                                    <p className="mt-1 text-sm font-medium text-gray-900 dark:text-white">
                                                        {post.author?.name || "Unknown"}
                                                    </p>
                                                    {post.author?.email && (
                                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                            {post.author.email}
                                                        </p>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                        
                                        {/* Created Date */}
                                        <div>
                                            <div className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className="flex items-center justify-center w-8 h-8 rounded-full bg-gray-100 dark:bg-gray-700">
                                                        <Calendar className="w-4 h-4 text-gray-600 dark:text-gray-400" />
                                                    </div>
                                                </div>
                                                <div className="ml-3 flex-1">
                                                    <p className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wide">
                                                        Created
                                                    </p>
                                                    <p className="mt-1 text-sm font-medium text-gray-900 dark:text-white">
                                                        {formatDateTime(post.created_at)}
                                                    </p>
                                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                        {getTimeAgo(post.created_at)}
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {/* Quick Actions */}
                                <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 sm:p-6">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                        Quick Actions
                                    </h3>
                                    <div className="space-y-3">
                                        <Link
                                            href={`/blog/${post.slug}`}
                                            target="_blank"
                                            className="flex items-center text-sm text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                        >
                                            <Eye className="w-4 h-4 mr-2" />
                                            View Post
                                        </Link>
                                        <button
                                            type="button"
                                            onClick={() => setActiveTab("seo")}
                                            className="flex items-center text-sm text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-300"
                                        >
                                            <Globe className="w-4 h-4 mr-2" />
                                            SEO Settings
                                        </button>
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setActiveTab("advanced")
                                            }
                                            className="flex items-center text-sm text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-300"
                                        >
                                            <Settings className="w-4 h-4 mr-2" />
                                            Advanced Options
                                        </button>
                                    </div>
                                </div>

                                {/* Post Statistics */}
                                <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 sm:p-6">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                        Post Stats
                                    </h3>
                                    <div className="space-y-3">
                                        <div className="flex items-center justify-between text-sm">
                                            <span className="text-gray-600 dark:text-gray-400">
                                                Read Time:
                                            </span>
                                            <span className="text-gray-900 dark:text-white font-medium">
                                                {data.read_time} min
                                            </span>
                                        </div>
                                        <div className="flex items-center justify-between text-sm">
                                            <span className="text-gray-600 dark:text-gray-400">
                                                Difficulty:
                                            </span>
                                            <span className="text-gray-900 dark:text-white font-medium capitalize">
                                                {data.difficulty}
                                            </span>
                                        </div>
                                        <div className="flex items-center justify-between text-sm">
                                            <span className="text-gray-600 dark:text-gray-400">
                                                Featured:
                                            </span>
                                            <span
                                                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                                    data.featured
                                                        ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400"
                                                        : "bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400"
                                                }`}
                                            >
                                                {data.featured ? (
                                                    <>
                                                        <Star className="w-3 h-3 mr-1" />
                                                        Yes
                                                    </>
                                                ) : (
                                                    "No"
                                                )}
                                            </span>
                                        </div>
                                        <div className="flex items-center justify-between text-sm">
                                            <span className="text-gray-600 dark:text-gray-400">
                                                Comments:
                                            </span>
                                            <span
                                                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                                    data.enable_comments
                                                        ? "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400"
                                                        : "bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400"
                                                }`}
                                            >
                                                {data.enable_comments ? (
                                                    <>
                                                        <MessageCircle className="w-3 h-3 mr-1" />
                                                        Enabled
                                                    </>
                                                ) : (
                                                    "Disabled"
                                                )}
                                            </span>
                                        </div>
                                        <div className="flex items-center justify-between text-sm">
                                            <span className="text-gray-600 dark:text-gray-400">
                                                Ratings:
                                            </span>
                                            <span
                                                className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                                                    data.enable_ratings
                                                        ? "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400"
                                                        : "bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400"
                                                }`}
                                            >
                                                {data.enable_ratings ? (
                                                    <>
                                                        <ThumbsUp className="w-3 h-3 mr-1" />
                                                        Enabled
                                                    </>
                                                ) : (
                                                    "Disabled"
                                                )}
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                {/* Validation Errors */}
                                {Object.keys(errors).length > 0 && (
                                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
                                        <div className="flex items-center mb-2">
                                            <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400 mr-2" />
                                            <h3 className="text-sm font-medium text-red-800 dark:text-red-400">
                                                Validation Errors
                                            </h3>
                                        </div>
                                        <ul className="text-sm text-red-700 dark:text-red-300 space-y-1">
                                            {Object.entries(errors).map(
                                                ([key, error]) => (
                                                    <li
                                                        key={key}
                                                        className="flex items-start"
                                                    >
                                                        <span className="font-medium capitalize mr-2">
                                                            {key.replace(
                                                                "_",
                                                                " "
                                                            )}
                                                            :
                                                        </span>
                                                        <span>{error}</span>
                                                    </li>
                                                )
                                            )}
                                        </ul>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </form>
            </div>

            {/* Click outside handler for tag dropdown */}
            {showTagDropdown && (
                <div
                    className="fixed inset-0 z-0"
                    onClick={() => setShowTagDropdown(false)}
                />
            )}

            {/* Toast Notification */}
            {showToast && toastMessage && (
                <div className="fixed top-8 left-8 z-50 animate-in slide-in-from-top-5 duration-300">
                    <div className="bg-green-600 text-white px-6 py-4 rounded-lg shadow-lg flex items-center gap-3 min-w-[300px] max-w-md">
                        <CheckCircle className="w-5 h-5 flex-shrink-0" />
                        <div className="flex-1">
                            <p className="font-medium">Success!</p>
                            <p className="text-sm text-green-100">{toastMessage}</p>
                        </div>
                        <button
                            onClick={() => setShowToast(false)}
                            className="text-green-100 hover:text-white transition-colors"
                        >
                            <X className="w-5 h-5" />
                        </button>
                    </div>
                </div>
            )}
        </AdminLayout>
    );
}
