import React, { useState, useCallback, useEffect } from "react";
import { Head, Link, router } from "@inertiajs/react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Search,
    Filter,
    Plus,
    Edit,
    Trash2,
    Eye,
    EyeOff,
    Calendar,
    User,
    Tag,
    FolderOpen,
    Clock,
    TrendingUp,
    MoreHorizontal,
    ArrowUpDown,
    ArrowUp,
    ArrowDown,
    Download,
    Upload,
    Settings,
    RefreshCw,
    CheckCircle,
    XCircle,
    AlertCircle,
    Archive,
    Copy,
    ExternalLink,
    BookOpen,
    Hash,
    Users,
    BarChart3,
    FileText,
    Globe,
    Heart,
    MessageCircle,
    Share2,
    Star,
    Loader,
	Loader2,
} from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";

export default function BlogPosts({
    auth,
    posts,
    categories,
    tags,
    authors,
    filters = {},
    flash,
}) {
    // Add safety checks for all props
    const safeFilters = filters || {};
    const safePosts = posts || {
        data: [],
        links: [],
        total: 0,
        from: 0,
        to: 0,
    };
    const safeCategories = categories || [];
    const safeTags = tags || [];
    const safeAuthors = authors || [];

    // State management with safe defaults
    const [selectedPosts, setSelectedPosts] = useState([]);
    const [searchQuery, setSearchQuery] = useState(safeFilters.search || "");
    const [statusFilter, setStatusFilter] = useState(
        safeFilters.status || "all"
    );
    const [categoryFilter, setCategoryFilter] = useState(
        safeFilters.category || "all"
    );
    const [authorFilter, setAuthorFilter] = useState(
        safeFilters.author || "all"
    );
    const [difficultyFilter, setDifficultyFilter] = useState(
        safeFilters.difficulty || "all"
    );
    const [sortBy, setSortBy] = useState(safeFilters.sort || "created_at");
    const [sortOrder, setSortOrder] = useState(safeFilters.direction || "desc");
    const [viewMode, setViewMode] = useState("grid");
    const [showFilters, setShowFilters] = useState(false);
    const [showBulkActions, setShowBulkActions] = useState(false);
    const [loading, setLoading] = useState(false);
    const [actionLoading, setActionLoading] = useState({});

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: {
            opacity: 1,
            y: 0,
            transition: {
                duration: 0.5,
            },
        },
    };

    // Apply filters by making server request
    const applyFilters = useCallback(() => {
        router.get(
            '/admin/blog/posts',
            {
                search: searchQuery || undefined,
                status: statusFilter !== 'all' ? statusFilter : undefined,
                category: categoryFilter !== 'all' ? categoryFilter : undefined,
                author: authorFilter !== 'all' ? authorFilter : undefined,
                difficulty: difficultyFilter !== 'all' ? difficultyFilter : undefined,
                sort: sortBy,
                direction: sortOrder,
            },
            {
                preserveState: true,
                preserveScroll: true,
                only: ['posts', 'filters'],
            }
        );
    }, [searchQuery, statusFilter, categoryFilter, authorFilter, difficultyFilter, sortBy, sortOrder]);

    // Handle search with debounce
    const handleSearch = useCallback((query) => {
        setSearchQuery(query);
    }, []);

    // Apply filters when they change
    React.useEffect(() => {
        const timer = setTimeout(() => {
            applyFilters();
        }, 500); // Debounce for search
        
        return () => clearTimeout(timer);
    }, [searchQuery]);

    // Apply filters immediately for non-search filters
    React.useEffect(() => {
        applyFilters();
    }, [statusFilter, categoryFilter, authorFilter, difficultyFilter, sortBy, sortOrder]);

    // Handle sort
    const handleSort = useCallback(
        (field) => {
            if (sortBy === field) {
                setSortOrder(sortOrder === "asc" ? "desc" : "asc");
            } else {
                setSortBy(field);
                setSortOrder("desc");
            }
        },
        [sortBy, sortOrder]
    );

    // Handle post selection
    const handlePostSelect = useCallback((postId) => {
        setSelectedPosts((prev) =>
            prev.includes(postId)
                ? prev.filter((id) => id !== postId)
                : [...prev, postId]
        );
    }, []);

    // Handle select all
    const handleSelectAll = useCallback(() => {
        if (selectedPosts.length === safePosts.data?.length) {
            setSelectedPosts([]);
        } else {
            setSelectedPosts(safePosts.data?.map((post) => post.id) || []);
        }
    }, [selectedPosts, safePosts.data]);

    // Handle post action
    const handlePostAction = useCallback(async (action, postId) => {
        setActionLoading((prev) => ({ ...prev, [postId]: true }));

        try {
            switch (action) {
                case "publish":
                    await router.patch(`/admin/blog/posts/${postId}/publish`);
                    break;
                case "draft":
                    await router.patch(`/admin/blog/posts/${postId}/draft`);
                    break;
                case "archive":
                    await router.patch(`/admin/blog/posts/${postId}/archive`);
                    break;
                case "delete":
                    if (confirm("Are you sure you want to delete this post?")) {
                        await router.delete(`/admin/blog/posts/${postId}`);
                    }
                    break;
                default:
                    break;
            }
        } catch (error) {
            console.error("Action failed:", error);
        } finally {
            setActionLoading((prev) => ({ ...prev, [postId]: false }));
        }
    }, []);

    // Handle bulk actions
    const handleBulkAction = useCallback(
        async (action) => {
            if (selectedPosts.length === 0) return;

            setLoading(true);
            try {
                switch (action) {
                    case "publish":
                        await router.patch("/admin/blog/posts/bulk-publish", {
                            post_ids: selectedPosts,
                        });
                        break;
                    case "draft":
                        await router.patch("/admin/blog/posts/bulk-draft", {
                            post_ids: selectedPosts,
                        });
                        break;
                    case "archive":
                        await router.patch("/admin/blog/posts/bulk-archive", {
                            post_ids: selectedPosts,
                        });
                        break;
                    case "delete":
                        if (
                            confirm(
                                `Are you sure you want to delete ${selectedPosts.length} posts?`
                            )
                        ) {
                            await router.delete(
                                "/admin/blog/posts/bulk-delete",
                                {
                                    data: { post_ids: selectedPosts },
                                }
                            );
                        }
                        break;
                    default:
                        break;
                }
                setSelectedPosts([]);
            } catch (error) {
                console.error("Bulk action failed:", error);
            } finally {
                setLoading(false);
            }
        },
        [selectedPosts]
    );

    // Get status color
    const getStatusColor = (status) => {
        switch (status) {
            case "published":
                return "text-green-600 bg-green-100 dark:text-green-400 dark:bg-green-900/30";
            case "draft":
                return "text-yellow-600 bg-yellow-100 dark:text-yellow-400 dark:bg-yellow-900/30";
            case "archived":
                return "text-gray-600 bg-gray-100 dark:text-gray-400 dark:bg-gray-900/30";
            default:
                return "text-gray-600 bg-gray-100 dark:text-gray-400 dark:bg-gray-900/30";
        }
    };

    // Get difficulty color
    const getDifficultyColor = (difficulty) => {
        switch (difficulty) {
            case "beginner":
                return "text-green-600 bg-green-100 dark:text-green-400 dark:bg-green-900/30";
            case "intermediate":
                return "text-yellow-600 bg-yellow-100 dark:text-yellow-400 dark:bg-yellow-900/30";
            case "advanced":
                return "text-red-600 bg-red-100 dark:text-red-400 dark:bg-red-900/30";
            default:
                return "text-gray-600 bg-gray-100 dark:text-gray-400 dark:bg-gray-900/30";
        }
    };

    // Format date
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "short",
            day: "numeric",
        });
    };

    // Format read time
    const formatReadTime = (minutes) => {
        return minutes === 1 ? "1 min" : `${minutes} mins`;
    };

    // Get sort icon
    const getSortIcon = (field) => {
        if (sortBy !== field) return <ArrowUpDown className="w-4 h-4" />;
        return sortOrder === "asc" ? (
            <ArrowUp className="w-4 h-4" />
        ) : (
            <ArrowDown className="w-4 h-4" />
        );
    };

    return (
        <AdminLayout>
            <Head title="Blog Posts" />

            <motion.div
                variants={containerVariants}
                initial="hidden"
                animate="visible"
                className="max-w-7xl mx-auto space-y-6"
            >
                {/* Header */}
                <motion.div
                    variants={itemVariants}
                    className="flex items-center justify-between"
                >
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                            Blog Posts
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400 mt-1">
                            Manage your blog posts, categories, and content
                        </p>
                    </div>

                    <div className="flex items-center gap-3">
                        <Link
                            href="/admin/blog/upload"
                            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                        >
                            <Upload className="w-4 h-4" />
                            <span>Upload Files</span>
                        </Link>

                        <Link
                            href="/admin/blog/posts/create"
                            className="flex items-center gap-2 px-4 py-2 bg-green-600 hover:bg-green-700 dark:bg-green-500 dark:hover:bg-green-600 text-white rounded-lg transition-colors"
                        >
                            <Plus className="w-4 h-4" />
                            <span>New Post</span>
                        </Link>
                    </div>
                </motion.div>

                {/* Stats Cards */}
                <motion.div
                    variants={itemVariants}
                    className="grid grid-cols-1 md:grid-cols-4 gap-4"
                >
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Total Posts
                                </p>
                                <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                    {safePosts.total || 0}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <FileText className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Published
                                </p>
                                <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                    {safePosts.data?.filter(
                                        (p) => p.status === "published"
                                    ).length || 0}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-green-100 dark:bg-green-900/30 rounded-lg flex items-center justify-center">
                                <CheckCircle className="w-6 h-6 text-green-600 dark:text-green-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Drafts
                                </p>
                                <p className="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                                    {safePosts.data?.filter(
                                        (p) => p.status === "draft"
                                    ).length || 0}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-yellow-100 dark:bg-yellow-900/30 rounded-lg flex items-center justify-center">
                                <AlertCircle className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">
                                    Categories
                                </p>
                                <p className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                    {safeCategories.length || 0}
                                </p>
                            </div>
                            <div className="w-12 h-12 bg-purple-100 dark:bg-purple-900/30 rounded-lg flex items-center justify-center">
                                <FolderOpen className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                            </div>
                        </div>
                    </div>
                </motion.div>

                {/* Search and Filters */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700"
                >
                    <div className="flex flex-col lg:flex-row gap-4 items-start lg:items-center justify-between">
                        <div className="flex-1 max-w-md">
                            <div className="relative">
                                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                                <input
                                    type="text"
                                    placeholder="Search posts..."
                                    value={searchQuery}
                                    onChange={(e) =>
                                        handleSearch(e.target.value)
                                    }
                                    className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400"
                                />
                            </div>
                        </div>

                        <div className="flex items-center gap-3">
                            <button
                                onClick={() => setShowFilters(!showFilters)}
                                className={`flex items-center gap-2 px-4 py-2 border rounded-lg transition-colors ${
                                    showFilters
                                        ? "bg-blue-50 border-blue-200 text-blue-700 dark:bg-blue-900/30 dark:border-blue-700 dark:text-blue-300"
                                        : "bg-white border-gray-300 text-gray-700 hover:bg-gray-50 dark:bg-gray-700 dark:border-gray-600 dark:text-gray-300 dark:hover:bg-gray-600"
                                }`}
                            >
                                <Filter className="w-4 h-4" />
                                <span>Filters</span>
                            </button>

                            <div className="flex items-center gap-2">
                                <span className="text-sm text-gray-600 dark:text-gray-400">
                                    View:
                                </span>
                                <div className="flex border border-gray-300 dark:border-gray-600 rounded-lg overflow-hidden">
                                    <button
                                        onClick={() => setViewMode("table")}
                                        className={`px-3 py-1 text-sm ${
                                            viewMode === "table"
                                                ? "bg-blue-600 text-white"
                                                : "bg-white text-gray-700 hover:bg-gray-50 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600"
                                        }`}
                                    >
                                        Table
                                    </button>
                                    <button
                                        onClick={() => setViewMode("grid")}
                                        className={`px-3 py-1 text-sm ${
                                            viewMode === "grid"
                                                ? "bg-blue-600 text-white"
                                                : "bg-white text-gray-700 hover:bg-gray-50 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600"
                                        }`}
                                    >
                                        Grid
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Filters Panel */}
                    <AnimatePresence>
                        {showFilters && (
                            <motion.div
                                initial={{ opacity: 0, height: 0 }}
                                animate={{ opacity: 1, height: "auto" }}
                                exit={{ opacity: 0, height: 0 }}
                                className="mt-4 pt-4 border-t border-gray-200 dark:border-gray-700"
                            >
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Status
                                        </label>
                                        <select
                                            value={statusFilter}
                                            onChange={(e) =>
                                                setStatusFilter(e.target.value)
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                        >
                                            <option value="all">
                                                All Status
                                            </option>
                                            <option value="published">
                                                Published
                                            </option>
                                            <option value="draft">Draft</option>
                                            <option value="archived">
                                                Archived
                                            </option>
                                        </select>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Category
                                        </label>
                                        <select
                                            value={categoryFilter}
                                            onChange={(e) =>
                                                setCategoryFilter(
                                                    e.target.value
                                                )
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                        >
                                            <option value="all">
                                                All Categories
                                            </option>
                                            {safeCategories.map((category) => (
                                                <option
                                                    key={category.id}
                                                    value={category.slug}
                                                >
                                                    {category.name}
                                                </option>
                                            ))}
                                        </select>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Author
                                        </label>
                                        <select
                                            value={authorFilter}
                                            onChange={(e) =>
                                                setAuthorFilter(e.target.value)
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                        >
                                            <option value="all">
                                                All Authors
                                            </option>
                                            {safeAuthors.map((author) => (
                                                <option
                                                    key={author.id}
                                                    value={author.id}
                                                >
                                                    {author.name}
                                                </option>
                                            ))}
                                        </select>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Difficulty
                                        </label>
                                        <select
                                            value={difficultyFilter}
                                            onChange={(e) =>
                                                setDifficultyFilter(
                                                    e.target.value
                                                )
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                        >
                                            <option value="all">
                                                All Difficulty
                                            </option>
                                            <option value="beginner">
                                                Beginner
                                            </option>
                                            <option value="intermediate">
                                                Intermediate
                                            </option>
                                            <option value="advanced">
                                                Advanced
                                            </option>
                                        </select>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Sort By
                                        </label>
                                        <select
                                            value={sortBy}
                                            onChange={(e) =>
                                                setSortBy(e.target.value)
                                            }
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                        >
                                            <option value="created_at">
                                                Date Created
                                            </option>
                                            <option value="published_at">
                                                Date Published
                                            </option>
                                            <option value="title">Title</option>
                                            <option value="author">
                                                Author
                                            </option>
                                            <option value="category">
                                                Category
                                            </option>
                                            <option value="read_time">
                                                Read Time
                                            </option>
                                        </select>
                                    </div>
                                </div>
                            </motion.div>
                        )}
                    </AnimatePresence>
                </motion.div>

                {/* Bulk Actions */}
                <AnimatePresence>
                    {selectedPosts.length > 0 && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4"
                        >
                            <div className="flex items-center justify-between">
                                <div className="flex items-center gap-3">
                                    <span className="text-sm text-blue-700 dark:text-blue-300">
                                        {selectedPosts.length} post
                                        {selectedPosts.length !== 1
                                            ? "s"
                                            : ""}{" "}
                                        selected
                                    </span>
                                    <button
                                        onClick={() => setSelectedPosts([])}
                                        className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200"
                                    >
                                        Clear selection
                                    </button>
                                </div>

                                <div className="flex items-center gap-2">
                                    <button
                                        onClick={() =>
                                            handleBulkAction("publish")
                                        }
                                        disabled={loading}
                                        className="px-3 py-1 bg-green-600 hover:bg-green-700 text-white rounded-lg text-sm transition-colors disabled:opacity-50"
                                    >
                                        Publish
                                    </button>
                                    <button
                                        onClick={() =>
                                            handleBulkAction("draft")
                                        }
                                        disabled={loading}
                                        className="px-3 py-1 bg-yellow-600 hover:bg-yellow-700 text-white rounded-lg text-sm transition-colors disabled:opacity-50"
                                    >
                                        Draft
                                    </button>
                                    <button
                                        onClick={() =>
                                            handleBulkAction("archive")
                                        }
                                        disabled={loading}
                                        className="px-3 py-1 bg-gray-600 hover:bg-gray-700 text-white rounded-lg text-sm transition-colors disabled:opacity-50"
                                    >
                                        Archive
                                    </button>
                                    <button
                                        onClick={() =>
                                            handleBulkAction("delete")
                                        }
                                        disabled={loading}
                                        className="px-3 py-1 bg-red-600 hover:bg-red-700 text-white rounded-lg text-sm transition-colors disabled:opacity-50"
                                    >
                                        Delete
                                    </button>
                                </div>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Posts Table */}
                <motion.div
                    variants={itemVariants}
                    className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden"
                >
                    {viewMode === "table" ? (
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead className="bg-gray-50 dark:bg-gray-900 border-b border-gray-200 dark:border-gray-700">
                                    <tr>
                                        <th className="w-12 px-6 py-3 text-left">
                                            <input
                                                type="checkbox"
                                                checked={
                                                    selectedPosts.length ===
                                                    safePosts.data?.length &&
                                                    safePosts.data?.length > 0
                                                }
                                                onChange={handleSelectAll}
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("title")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Title {getSortIcon("title")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("status")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Status
                                                {getSortIcon("status")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("category")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Category
                                                {getSortIcon("category")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("author")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Author
                                                {getSortIcon("author")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            <button
                                                onClick={() =>
                                                    handleSort("published_at")
                                                }
                                                className="flex items-center gap-2 hover:text-gray-700 dark:hover:text-gray-300"
                                            >
                                                Published
                                                {getSortIcon("published_at")}
                                            </button>
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Read Time
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Actions
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {safePosts.data?.map((post, index) => (
                                        <motion.tr
                                            key={post.id}
                                            initial={{ opacity: 0, y: 20 }}
                                            animate={{ opacity: 1, y: 0 }}
                                            transition={{ delay: index * 0.05 }}
                                            className="hover:bg-gray-50 dark:hover:bg-gray-900 transition-colors"
                                        >
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <input
                                                    type="checkbox"
                                                    checked={selectedPosts.includes(
                                                        post.id
                                                    )}
                                                    onChange={() =>
                                                        handlePostSelect(
                                                            post.id
                                                        )
                                                    }
                                                    className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                                />
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center">
                                                    <div>
                                                        <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                            {post.title}
                                                        </div>
                                                        <div className="text-sm text-gray-500 dark:text-gray-400">
                                                            {post.excerpt?.substring(
                                                                0,
                                                                60
                                                            )}
                                                            ...
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <span
                                                    className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(
                                                        post.status
                                                    )}`}
                                                >
                                                    {post.status}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center">
                                                    <span
                                                        className="w-3 h-3 rounded-full mr-2"
                                                        style={{
                                                            backgroundColor:
                                                                post.category
                                                                    ?.color ||
                                                                "#6366f1",
                                                        }}
                                                    />
                                                    <span className="text-sm text-gray-900 dark:text-white">
                                                        {post.category?.name ||
                                                            "Uncategorized"}
                                                    </span>
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center">
                                                    <div className="flex-shrink-0 h-8 w-8">
                                                        <div className="h-8 w-8 rounded-full bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                                            <User className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                                                        </div>
                                                    </div>
                                                    <div className="ml-3">
                                                        <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                            {post.author
                                                                ?.name ||
                                                                "Unknown"}
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                {formatDate(post.published_at)}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                                    <Clock className="w-4 h-4 mr-1" />
                                                    {formatReadTime(
                                                        post.read_time
                                                    )}
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <div className="flex items-center gap-2">
                                                    <a
                                                        href={`/blog/${post.slug}`}
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                                    >
                                                        <Eye className="w-4 h-4" />
                                                    </a>

                                                    {/* <Link
                                                        href={route(
                                                            "admin.blog.posts.edit",
                                                            post.slug
                                                        )}
                                                        className="text-yellow-600 hover:text-yellow-800"
                                                    >
                                                        <Edit className="w-4 h-4" />
                                                    </Link> */}

                                                    <Link
                                                        href={`/admin/blog/posts/${post.id}/edit`}
                                                        className="p-1 text-gray-400 hover:text-yellow-600 dark:hover:text-yellow-400 transition-colors"
                                                    >
                                                        <Edit className="w-4 h-4" />
                                                    </Link>
                                                    <button
                                                        onClick={() =>
                                                            handlePostAction(
                                                                "delete",
                                                                post.id
                                                            )
                                                        }
                                                        disabled={
                                                            actionLoading[
                                                                post.id
                                                            ]
                                                        }
                                                        className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-200 disabled:opacity-50"
                                                    >
                                                        {actionLoading[
                                                            post.id
                                                        ] ? (
                                                            <Loader className="w-4 h-4 animate-spin" />
                                                        ) : (
                                                            <Trash2 className="w-4 h-4" />
                                                        )}
                                                    </button>
                                                </div>
                                            </td>
                                        </motion.tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    ) : (
                        /* Grid View */
                        <div className="p-6 grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {safePosts.data?.map((post, index) => (
                                <motion.div
                                    key={post.id}
                                    initial={{ opacity: 0, scale: 0.9 }}
                                    animate={{ opacity: 1, scale: 1 }}
                                    transition={{ delay: index * 0.05 }}
                                    className="bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-lg p-4 hover:shadow-md transition-shadow"
                                >
                                    <div className="flex items-start justify-between mb-3">
                                        <div className="flex items-center gap-2">
                                            <input
                                                type="checkbox"
                                                checked={selectedPosts.includes(
                                                    post.id
                                                )}
                                                onChange={() =>
                                                    handlePostSelect(post.id)
                                                }
                                                className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                                            />
                                            <span
                                                className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(
                                                    post.status
                                                )}`}
                                            >
                                                {post.status}
                                            </span>
                                        </div>
                                        <div className="flex items-center gap-1">
                                            <a
                                                href={`/blog/${post.slug}`}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="p-1 text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                            >
                                                <Eye className="w-4 h-4" />
                                            </a>
                                            <Link
                                                href={`/admin/blog/posts/${post.id}/edit`}
                                                className="p-1 text-gray-400 hover:text-yellow-600 dark:hover:text-yellow-400 transition-colors"
                                            >
                                                <Edit className="w-4 h-4" />
                                            </Link>
                                            <button
                                                onClick={() =>
                                                    handlePostAction(
                                                        "delete",
                                                        post.id
                                                    )
                                                }
                                                disabled={
                                                    actionLoading[post.id]
                                                }
                                                className="p-1 text-gray-400 hover:text-red-600 dark:hover:text-red-400 transition-colors disabled:opacity-50"
                                            >
                                                {actionLoading[post.id] ? (
                                                    <Loader2 className="w-4 h-4 animate-spin" />
                                                ) : (
                                                    <Trash2 className="w-4 h-4" />
                                                )}
                                            </button>
                                        </div>
                                    </div>

                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                                        {post.title}
                                    </h3>

                                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-3">
                                        {post.excerpt?.substring(0, 100)}...
                                    </p>

                                    <div className="flex items-center justify-between mb-3">
                                        <div className="flex items-center gap-2">
                                            <span
                                                className="w-3 h-3 rounded-full"
                                                style={{
                                                    backgroundColor:
                                                        post.category?.color ||
                                                        "#6366f1",
                                                }}
                                            />
                                            <span className="text-sm text-gray-600 dark:text-gray-400">
                                                {post.category?.name ||
                                                    "Uncategorized"}
                                            </span>
                                        </div>
                                        <span
                                            className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${getDifficultyColor(
                                                post.difficulty
                                            )}`}
                                        >
                                            {post.difficulty}
                                        </span>
                                    </div>

                                    <div className="flex items-center justify-between text-sm text-gray-500 dark:text-gray-400">
                                        <div className="flex items-center gap-1">
                                            <User className="w-4 h-4" />
                                            <span>
                                                {post.author?.name || "Unknown"}
                                            </span>
                                        </div>
                                        <div className="flex items-center gap-1">
                                            <Clock className="w-4 h-4" />
                                            <span>
                                                {formatReadTime(post.read_time)}
                                            </span>
                                        </div>
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    )}
                </motion.div>

                {/* Pagination */}
                {safePosts.links && (
                    <motion.div
                        variants={itemVariants}
                        className="flex items-center justify-between bg-white dark:bg-gray-800 px-6 py-3 border border-gray-200 dark:border-gray-700 rounded-lg"
                    >
                        <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                            Showing {safePosts.from || 0} to {safePosts.to || 0}{" "}
                            of {safePosts.total || 0} results
                        </div>
                        <div className="flex items-center gap-2">
                            {safePosts.links.map((link, index) => {
                                // Extract page number from URL if available
                                let pageNum = null;
                                if (link.url) {
                                    const urlParams = new URLSearchParams(link.url.split('?')[1]);
                                    pageNum = urlParams.get('page');
                                }
                                
                                return (
                                    <button
                                        key={index}
                                        onClick={() => {
                                            if (link.url && !link.active) {
                                                router.get(
                                                    '/admin/blog/posts',
                                                    {
                                                        search: searchQuery || undefined,
                                                        status: statusFilter !== 'all' ? statusFilter : undefined,
                                                        category: categoryFilter !== 'all' ? categoryFilter : undefined,
                                                        author: authorFilter !== 'all' ? authorFilter : undefined,
                                                        difficulty: difficultyFilter !== 'all' ? difficultyFilter : undefined,
                                                        sort: sortBy,
                                                        direction: sortOrder,
                                                        page: pageNum,
                                                    },
                                                    {
                                                        preserveState: true,
                                                        preserveScroll: true,
                                                        only: ['posts', 'filters'],
                                                    }
                                                );
                                            }
                                        }}
                                        disabled={!link.url || link.active}
                                        className={`px-3 py-1 rounded-lg text-sm ${
                                            link.active
                                                ? "bg-blue-600 text-white"
                                                : link.url
                                                ? "text-blue-600 hover:bg-blue-50 dark:text-blue-400 dark:hover:bg-blue-900/20 cursor-pointer"
                                                : "text-gray-400 cursor-not-allowed"
                                        }`}
                                        dangerouslySetInnerHTML={{
                                            __html: link.label,
                                        }}
                                    />
                                );
                            })}
                        </div>
                    </motion.div>
                )}

                {/* Empty State */}
                {(!safePosts.data || safePosts.data.length === 0) && (
                    <motion.div
                        variants={itemVariants}
                        className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-12 text-center"
                    >
                        <div className="w-16 h-16 mx-auto mb-4 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center">
                            <FileText className="w-8 h-8 text-gray-400" />
                        </div>
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                            No posts found
                        </h3>
                        <p className="text-gray-500 dark:text-gray-400 mb-6">
                            {searchQuery ||
                            statusFilter !== "all" ||
                            categoryFilter !== "all"
                                ? "Try adjusting your search or filter criteria."
                                : "Get started by creating your first blog post."}
                        </p>
                        <div className="flex items-center justify-center gap-3">
                            {searchQuery ||
                            statusFilter !== "all" ||
                            categoryFilter !== "all" ? (
                                <button
                                    onClick={() => {
                                        setSearchQuery("");
                                        setStatusFilter("all");
                                        setCategoryFilter("all");
                                        setAuthorFilter("all");
                                        setDifficultyFilter("all");
                                    }}
                                    className="flex items-center gap-2 px-4 py-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-200"
                                >
                                    <RefreshCw className="w-4 h-4" />
                                    Clear filters
                                </button>
                            ) : (
                                <>
                                    <Link
                                        href="/admin/blog/upload"
                                        className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white rounded-lg transition-colors"
                                    >
                                        <Upload className="w-4 h-4" />
                                        Upload Files
                                    </Link>
                                    <Link
                                        href="/admin/blog/posts/create"
                                        className="flex items-center gap-2 px-4 py-2 bg-green-600 hover:bg-green-700 dark:bg-green-500 dark:hover:bg-green-600 text-white rounded-lg transition-colors"
                                    >
                                        <Plus className="w-4 h-4" />
                                        Create Post
                                    </Link>
                                </>
                            )}
                        </div>
                    </motion.div>
                )}

                {/* Flash Messages */}
                <AnimatePresence>
                    {flash?.success && (
                        <motion.div
                            initial={{ opacity: 0, y: 50 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -50 }}
                            className="fixed bottom-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-lg z-50"
                        >
                            <div className="flex items-center gap-2">
                                <CheckCircle className="w-5 h-5" />
                                <span>{flash.success}</span>
                            </div>
                        </motion.div>
                    )}
                    {flash?.error && (
                        <motion.div
                            initial={{ opacity: 0, y: 50 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -50 }}
                            className="fixed bottom-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg z-50"
                        >
                            <div className="flex items-center gap-2">
                                <XCircle className="w-5 h-5" />
                                <span>{flash.error}</span>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </motion.div>
        </AdminLayout>
    );
}
