import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import { Badge } from '@/Components/ui/Badge';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import GuaranteesGridBlockEditor from '@/Components/Admin/GuaranteesGridBlockEditor';
import QualityStandardsBlockEditor from '@/Components/Admin/QualityStandardsBlockEditor';
import PricingTiersBlockEditor from '@/Components/Admin/PricingTiersBlockEditor';
import ServicePricingGridBlockEditor from '@/Components/Admin/ServicePricingGridBlockEditor';
import PricingReasonsBlockEditor from '@/Components/Admin/PricingReasonsBlockEditor';
import PricingTestimonialsBlockEditor from '@/Components/Admin/PricingTestimonialsBlockEditor';
import PricingResourcesBlockEditor from '@/Components/Admin/PricingResourcesBlockEditor';
import PricingSecurityBlockEditor from '@/Components/Admin/PricingSecurityBlockEditor';
import ServiceLongformBlockEditor from './ServiceEditors/ServiceLongformBlockEditor';
import ServiceLinksBlockEditor from './ServiceEditors/ServiceLinksBlockEditor';
import PriceCalculatorBlockEditor from '@/Components/Admin/PriceCalculatorBlockEditor';
import { 
    GripVertical, 
    Trash2, 
    Eye, 
    EyeOff,
    Plus,
    Edit,
    Save,
    X,
    Move,
    ChevronUp,
    ChevronDown,
    Settings
} from 'lucide-react';

export default function ContentBlockEditor({ blocks, onUpdate, onDelete, onReorder, lastSavedBlockId, onToggleActive = () => {} }) {
    const [editingBlock, setEditingBlock] = useState(null);
    const [draggedIndex, setDraggedIndex] = useState(null);

    const blockTypeLabels = {
        hero: 'Hero Section',
        overview: 'Overview',
        benefits: 'Benefits',
        process: 'Process',
        testimonials: 'Testimonials',
        faqs: 'FAQs',
        contact_form: 'Contact Form',
        cta: 'Call to Action',
        features: 'Features',
        pricing: 'Pricing',
        gallery: 'Gallery',
        text_content: 'Text Content',
        video: 'Video',
        custom: 'Custom',
        service_hero: 'Service Hero',
        service_overview: 'Service Overview',
        service_process: 'Service Process',
        service_benefits: 'Service Benefits',
        service_testimonials: 'Service Testimonials',
        service_faqs: 'Service FAQs',
        service_cta: 'Service CTA',
        internal_links: 'Internal Links',
        about: 'About Section',
        services: 'Services Section',
        stats: 'Statistics Section',
        contact: 'Contact Section',
        guarantees_hero: 'Guarantees Hero',
        main_guarantees: 'Guarantees Grid',
        quality_standards: 'Quality Standards',
        guarantee_process: 'How Guarantees Work',
        service_guarantees: 'Service Guarantees',
        guarantees_testimonials: 'Guarantees Testimonials',
        guarantees_faqs: 'Guarantees FAQs',
        trust_indicators: 'Trust Indicators',
        guarantee_comparison: 'Guarantee Comparison',
        contact_support: 'Contact Support',
        service_links: 'Service Links Footer',
        guarantees_cta: 'Guarantees CTA',
        reviews_hero: 'Reviews Hero',
        rating_overview: 'Rating Overview',
        featured_testimonials: 'Featured Testimonials',
        service_filters: 'Service Filters',
        customer_reviews: 'Customer Reviews',
        reviews_by_service: 'Reviews by Service',
        review_trust: 'Review Trust Indicators',
        reviews_faqs: 'Reviews FAQs',
        reviews_cta: 'Reviews CTA',
        review_service_links: 'Review Service Links',
        writers_hero: 'Writers Hero',
        selection_process: 'Selection Process',
        writer_superpowers: 'Writer Superpowers',
        quality_control: 'Quality Control',
        client_benefits: 'Client Benefits',
        featured_writers: 'Featured Writers',
        writer_expertise: 'Writer Expertise',
        writers_faqs: 'Writers FAQs',
        writers_cta: 'Writers CTA',
        about_hero: 'About Hero',
        about_story: 'About Story',
        about_achievements: 'About Achievements',
        about_values: 'About Values',
        about_services: 'About Services',
        about_why_choose: 'About Why Choose',
        about_process: 'About Process',
        about_testimonials: 'About Testimonials',
        about_faqs: 'About FAQs',
        about_cta: 'About CTA',
        pricing_hero: 'Pricing Hero',
        pricing_tiers: 'Pricing Tiers',
        service_pricing_grid: 'Service Pricing Grid',
        service_image: 'Service Image',
        pricing_reasons: 'Pricing Reasons',
        pricing_faqs: 'Pricing FAQs',
        pricing_cta: 'Pricing CTA',
        pricing_keywords: 'Pricing Keywords',
        pricing_testimonials: 'Pricing Testimonials',
        pricing_resources: 'Pricing Resources',
        pricing_security: 'Pricing Security',
        pricing_contact: 'Pricing Contact',
        service_scrollable: 'Scrollable Section',
        price_calculator: 'Price Calculator'
    };

    const blockTypeIcons = {
        hero: '🎯',
        overview: '📋',
        benefits: '✨',
        process: '⚡',
        testimonials: '💬',
        faqs: '❓',
        contact_form: '📝',
        cta: '📢',
        features: '🔧',
        pricing: '💰',
        gallery: '🖼️',
        text_content: '📄',
        video: '🎥',
        custom: '⚙️',
        service_hero: '🎯',
        service_overview: '📋',
        service_process: '⚡',
        service_benefits: '✨',
        service_testimonials: '💬',
        service_faqs: '❓',
        service_cta: '📢',
        internal_links: '🔗',
        about: '👥',
        services: '🛠️',
        stats: '📊',
        contact: '📧',
        guarantees_hero: '🛡️',
        main_guarantees: '✅',
        quality_standards: '⭐',
        guarantee_process: '🔄',
        service_guarantees: '🎯',
        guarantees_testimonials: '💬',
        guarantees_faqs: '❓',
        trust_indicators: '📈',
        guarantee_comparison: '📊',
        contact_support: '📞',
        service_links: '🔗',
        guarantees_cta: '📢',
        reviews_hero: '⭐',
        rating_overview: '📊',
        featured_testimonials: '💬',
        service_filters: '🔍',
        customer_reviews: '📝',
        reviews_by_service: '🎯',
        review_trust: '🛡️',
        reviews_faqs: '❓',
        reviews_cta: '📢',
        review_service_links: '🔗',
        writers_hero: '✍️',
        selection_process: '🎯',
        writer_superpowers: '⚡',
        quality_control: '🛡️',
        client_benefits: '💎',
        featured_writers: '👨‍🎓',
        writer_expertise: '🌟',
        writers_faqs: '❓',
        writers_cta: '📢',
        about_hero: '🏢',
        about_story: '📖',
        about_achievements: '🏆',
        about_values: '💎',
        about_services: '📝',
        about_why_choose: '⭐',
        about_process: '⚙️',
        about_testimonials: '💬',
        about_faqs: '❓',
        about_cta: '📢',
        pricing_hero: '💰',
        pricing_tiers: '📊',
        service_pricing_grid: '🛠️',
        service_image: '🖼️',
        pricing_reasons: '✨',
        pricing_faqs: '❓',
        pricing_cta: '📢',
        pricing_keywords: '🔍',
        pricing_testimonials: '💬',
        pricing_resources: '📚',
        pricing_security: '🛡️',
        pricing_contact: '📞',
        service_scrollable: '📝',
        price_calculator: '💰'
    };

    const handleDragStart = (e, index) => {
        setDraggedIndex(index);
        e.dataTransfer.effectAllowed = 'move';
    };

    const handleDragOver = (e) => {
        e.preventDefault();
        e.dataTransfer.dropEffect = 'move';
    };

    const handleDrop = (e, dropIndex) => {
        e.preventDefault();
        if (draggedIndex !== null && draggedIndex !== dropIndex) {
            onReorder(draggedIndex, dropIndex);
        }
        setDraggedIndex(null);
    };

    const toggleBlockStatus = (block) => {
        onToggleActive(block.id, !block.is_active);
    };

    const renderBlockEditor = (block) => {
        switch (block.block_type) {
            case 'hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'overview':
                return <OverviewBlockEditor block={block} onUpdate={onUpdate} />;
            case 'benefits':
                return <BenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'process':
                return <ProcessBlockEditor block={block} onUpdate={onUpdate} />;
            case 'testimonials':
                return <TestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'faqs':
                return <FAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'contact_form':
                return <ContactFormBlockEditor block={block} onUpdate={onUpdate} />;
            case 'cta':
                return <CTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_hero':
                return <ServiceHeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_overview':
                return <ServiceOverviewBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_process':
                return <ServiceProcessBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_benefits':
                return <ServiceBenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_testimonials':
                return <ServiceTestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'internal_links':
                return <InternalLinksBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about':
                return <AboutBlockEditor block={block} onUpdate={onUpdate} />;
            case 'services':
                return <ServicesBlockEditor block={block} onUpdate={onUpdate} />;
            case 'stats':
                return <StatsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'contact':
                return <ContactBlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantees_hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'main_guarantees':
                return <GuaranteesGridBlockEditor block={block} onUpdate={onUpdate} />;
            case 'quality_standards':
                return <QualityStandardsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantees_testimonials':
                return <ServiceTestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantees_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantees_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantee_process':
                return <ServiceProcessBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_guarantees':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'trust_indicators':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'guarantee_comparison':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'contact_support':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_links':
                return <ServiceLinksBlockEditor block={block} onUpdate={onUpdate} />;
            case 'reviews_hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'rating_overview':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'featured_testimonials':
                return <ServiceTestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_filters':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'customer_reviews':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'reviews_by_service':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'review_trust':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'reviews_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'reviews_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'review_service_links':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'writers_hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'selection_process':
                return <ServiceProcessBlockEditor block={block} onUpdate={onUpdate} />;
            case 'writer_superpowers':
                return <ServiceBenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'quality_control':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'client_benefits':
                return <ServiceBenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'featured_writers':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'writer_expertise':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'writers_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'writers_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_story':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_achievements':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_values':
                return <ServiceBenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_services':
                return <ServiceBenefitsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_why_choose':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_process':
                return <ServiceProcessBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_testimonials':
                return <ServiceTestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'about_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_hero':
                return <HeroBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_tiers':
                return <PricingTiersBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_pricing_grid':
                return <ServicePricingGridBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_image':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_reasons':
                return <PricingReasonsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_faqs':
                return <ServiceFAQsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_cta':
                return <ServiceCTABlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_keywords':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_testimonials':
                return <PricingTestimonialsBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_resources':
                return <PricingResourcesBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_security':
                return <PricingSecurityBlockEditor block={block} onUpdate={onUpdate} />;
            case 'pricing_contact':
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
            case 'service_scrollable':
                return <ServiceLongformBlockEditor block={block} onUpdate={onUpdate} />;
            case 'price_calculator':
                return <PriceCalculatorBlockEditor block={block} onUpdate={onUpdate} />;
            default:
                return <GenericBlockEditor block={block} onUpdate={onUpdate} />;
        }
    };

    if (blocks.length === 0) {
        return (
            <div className="text-center py-12 border-2 border-dashed border-gray-300 rounded-lg">
                <div className="text-gray-400 mb-4">
                    <Settings className="h-12 w-12 mx-auto" />
                </div>
                <h3 className="text-lg font-medium text-gray-900 mb-2">No content blocks</h3>
                <p className="text-gray-500">Add content blocks to build your page</p>
            </div>
        );
    }

    return (
        <div className="space-y-4">
            {blocks.map((block, index) => {
                const contentTitle = typeof block.content === 'object' && block.content !== null
                    ? (block.content.title || block.content.heading || block.content.label || '').toString().trim()
                    : '';
                const displayTitle = contentTitle || blockTypeLabels[block.block_type] || block.block_type;
                const subtitle = contentTitle ? (blockTypeLabels[block.block_type] || block.block_type) : '';

                return (
                <Card 
                    className={`transition-all duration-200 ${
                        block.is_active ? 'opacity-100' : 'opacity-60'
                    } ${draggedIndex === index ? 'shadow-lg' : ''} ${
                        lastSavedBlockId === block.id ? 'ring-2 ring-green-400 shadow-green-100' : ''
                    }`}
                    draggable
                    onDragStart={(e) => handleDragStart(e, index)}
                    onDragOver={handleDragOver}
                    onDrop={(e) => handleDrop(e, index)}
                >
                    <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                                <GripVertical className="h-5 w-5 text-gray-400 cursor-move" />
                                <div className="text-2xl">
                                    {blockTypeIcons[block.block_type]}
                                </div>
                                <div>
                                    <CardTitle className="text-lg">
                                        {displayTitle}
                                    </CardTitle>
                                    {subtitle && (
                                        <p className="text-xs uppercase tracking-wide text-gray-500 dark:text-gray-400">
                                            {subtitle}
                                        </p>
                                    )}
                                    <div className="flex items-center space-x-2 mt-1">
                                        <Badge variant={block.is_active ? 'default' : 'secondary'}>
                                            {block.is_active ? 'Active' : 'Inactive'}
                                        </Badge>
                                        {lastSavedBlockId === block.id && (
                                            <Badge className="bg-green-500 text-white animate-pulse">
                                                ✓ Saved in Memory
                                            </Badge>
                                        )}
                                        <span className="text-sm text-gray-500">
                                            Order: {block.order}
                                        </span>
                                    </div>
                                </div>
                            </div>
                            
                            <div className="flex items-center space-x-2">
                                <button
                                    type="button"
                                    className="p-2 text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-200 rounded-md hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                                    onClick={() => toggleBlockStatus(block)}
                                    title={block.is_active ? 'Deactivate Block (Hide on Frontend)' : 'Activate Block (Show on Frontend)'}
                                >
                                    {block.is_active ? (
                                        <Eye className="h-4 w-4" />
                                    ) : (
                                        <EyeOff className="h-4 w-4" />
                                    )}
                                </button>
                                
                                <button
                                    type="button"
                                    className="p-2 text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-200 rounded-md hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                                    onClick={() => setEditingBlock(editingBlock === block.id ? null : block.id)}
                                >
                                    <Edit className="h-4 w-4" />
                                </button>
                                
                                <button
                                    type="button"
                                    className="p-2 text-red-600 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300 rounded-md hover:bg-red-50 dark:hover:bg-red-900/20 transition-colors"
                                    onClick={() => {
                                        if (confirm('Are you sure you want to delete this content block?')) {
                                            onDelete(block.id);
                                        }
                                    }}
                                >
                                    <Trash2 className="h-4 w-4" />
                                </button>
                            </div>
                        </div>
                    </CardHeader>
                    
                    {editingBlock === block.id && (
                        <CardContent>
                            <div className="border-t pt-4">
                                {renderBlockEditor(block)}
                            </div>
                        </CardContent>
                    )}
                    
                    {!editingBlock && (
                        <CardContent>
                            <div className="text-sm text-gray-600">
                                {block.block_type === 'hero' && (
                                    <div>
                                        <strong>Title:</strong> {block.content.title || 'Not set'}<br />
                                        <strong>Subtitle:</strong> {block.content.subtitle || 'Not set'}
                                    </div>
                                )}
                                {block.block_type === 'overview' && (
                                    <div>
                                        <strong>Title:</strong> {block.content.title || 'Not set'}<br />
                                        <strong>Description:</strong> {block.content.description ? `${block.content.description.substring(0, 100)}...` : 'Not set'}
                                    </div>
                                )}
                                {block.block_type === 'benefits' && (
                                    <div>
                                        <strong>Title:</strong> {block.content.title || 'Not set'}<br />
                                        <strong>Benefits:</strong> {block.content.benefits?.length || 0} items
                                    </div>
                                )}
                                {block.block_type === 'faqs' && (
                                    <div>
                                        <strong>Title:</strong> {block.content.title || 'Not set'}<br />
                                        <strong>FAQs:</strong> {block.content.faqs?.length || 0} items
                                    </div>
                                )}
                                {block.block_type === 'service_scrollable' && (
                                    <div>
                                        <strong>Section:</strong> {contentTitle || 'Not set'}
                                    </div>
                                )}
                            </div>
                        </CardContent>
                    )}
                </Card>
            )})}
        </div>
    );
}

// Individual Block Editors
function HeroBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [activeSection, setActiveSection] = useState('headline');

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    const sections = [
        { id: 'headline', label: 'Headline', icon: '📝' },
        { id: 'description', label: 'Description', icon: '📄' },
        { id: 'bullets', label: 'Key Points', icon: '✓' },
        { id: 'ctas', label: 'Call to Actions', icon: '🔘' },
        { id: 'stats', label: 'Statistics', icon: '📊' },
        { id: 'media', label: 'Images', icon: '🖼️' },
    ];

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {sections.map((section) => (
                    <button
                        key={section.id}
                        type="button"
                        onClick={() => setActiveSection(section.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeSection === section.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {section.icon} {section.label}
                    </button>
                ))}
            </div>

            {/* Headline Section */}
            {activeSection === 'headline' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Headline Components</h3>
                    <div className="space-y-4">
                        <div>
                            <Label htmlFor="headlinePrimary">Headline Primary Text</Label>
                            <Input
                                id="headlinePrimary"
                                value={content.headlinePrimary || ''}
                                onChange={(e) => setContent({...content, headlinePrimary: e.target.value})}
                                placeholder="e.g., Professional Academic Writing"
                            />
                        </div>
                        <div>
                            <Label htmlFor="headlineHighlight">Headline Highlight (Colored Text)</Label>
                            <Input
                                id="headlineHighlight"
                                value={content.headlineHighlight || ''}
                                onChange={(e) => setContent({...content, headlineHighlight: e.target.value})}
                                placeholder="e.g., Services"
                                className="border-l-4 border-cyan-400"
                            />
                            <p className="text-xs text-gray-500 mt-1">This text will appear in cyan color</p>
                        </div>
                        <div>
                            <Label htmlFor="headlineSuffix">Headline Suffix Text</Label>
                            <Input
                                id="headlineSuffix"
                                value={content.headlineSuffix || ''}
                                onChange={(e) => setContent({...content, headlineSuffix: e.target.value})}
                                placeholder="e.g., for Students Worldwide"
                            />
                        </div>
                        <div>
                            <Label htmlFor="trustText">Trust Badge Text</Label>
                            <Input
                                id="trustText"
                                value={content.trustText || ''}
                                onChange={(e) => setContent({...content, trustText: e.target.value})}
                                placeholder="e.g., Trusted by 50,000+ Students Worldwide"
                            />
                            <p className="text-xs text-gray-500 mt-1">Shows above headline with 5 stars</p>
                        </div>
                    </div>
                </div>
            )}

            {/* Description Section */}
            {activeSection === 'description' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Hero Description</h3>
                    <div>
                        <Label htmlFor="paragraph">Paragraph Content (Rich Text)</Label>
                        <RichTextEditor
                            value={content.paragraph || ''}
                            onChange={(value) => setContent({...content, paragraph: value})}
                            height={250}
                        />
                        <p className="text-xs text-gray-500 mt-1">Use bold, italic, links, etc.</p>
                    </div>
                </div>
            )}

            {/* Bullets/Key Points Section */}
            {activeSection === 'bullets' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Key Benefit Points</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">These appear as checkmark bullet points below the description</p>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="bullet1">✓ Bullet Point 1</Label>
                            <Input
                                id="bullet1"
                                value={content.bullet1 || ''}
                                onChange={(e) => setContent({...content, bullet1: e.target.value})}
                                placeholder="e.g., 24/7 Support"
                            />
                        </div>
                        <div>
                            <Label htmlFor="bullet2">✓ Bullet Point 2</Label>
                            <Input
                                id="bullet2"
                                value={content.bullet2 || ''}
                                onChange={(e) => setContent({...content, bullet2: e.target.value})}
                                placeholder="e.g., Expert Writers"
                            />
                        </div>
                        <div>
                            <Label htmlFor="bullet3">✓ Bullet Point 3</Label>
                            <Input
                                id="bullet3"
                                value={content.bullet3 || ''}
                                onChange={(e) => setContent({...content, bullet3: e.target.value})}
                                placeholder="e.g., Plagiarism Free"
                            />
                        </div>
                        <div>
                            <Label htmlFor="bullet4">✓ Bullet Point 4</Label>
                            <Input
                                id="bullet4"
                                value={content.bullet4 || ''}
                                onChange={(e) => setContent({...content, bullet4: e.target.value})}
                                placeholder="e.g., Money Back Guarantee"
                            />
                        </div>
                    </div>
                </div>
            )}

            {/* CTAs Section */}
            {activeSection === 'ctas' && (
                <div className="space-y-6">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Call to Action Buttons</h3>
                    
                    {/* Primary CTA */}
                    <div className="border border-blue-200 dark:border-blue-800 rounded-lg p-4 bg-blue-50 dark:bg-blue-900/20">
                        <h4 className="font-medium text-blue-900 dark:text-blue-200 mb-3">Primary CTA Button</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="ctaPrimaryText">Button Text</Label>
                                <Input
                                    id="ctaPrimaryText"
                                    value={content.ctaPrimaryText || ''}
                                    onChange={(e) => setContent({...content, ctaPrimaryText: e.target.value})}
                                    placeholder="e.g., Get Started Now"
                                />
                            </div>
                            <div>
                                <Label htmlFor="ctaPrimaryHref">Button Link</Label>
                                <Input
                                    id="ctaPrimaryHref"
                                    value={content.ctaPrimaryHref || ''}
                                    onChange={(e) => setContent({...content, ctaPrimaryHref: e.target.value})}
                                    placeholder="/place-order"
                                />
                            </div>
                        </div>
                    </div>

                    {/* Secondary CTA */}
                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-gray-50 dark:bg-gray-800/50">
                        <h4 className="font-medium text-gray-900 dark:text-gray-200 mb-3">Secondary CTA Button</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="ctaSecondaryText">Button Text</Label>
                                <Input
                                    id="ctaSecondaryText"
                                    value={content.ctaSecondaryText || ''}
                                    onChange={(e) => setContent({...content, ctaSecondaryText: e.target.value})}
                                    placeholder="e.g., View Our Services"
                                />
                            </div>
                            <div>
                                <Label htmlFor="ctaSecondaryHref">Button Link</Label>
                                <Input
                                    id="ctaSecondaryHref"
                                    value={content.ctaSecondaryHref || ''}
                                    onChange={(e) => setContent({...content, ctaSecondaryHref: e.target.value})}
                                    placeholder="/services"
                                />
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Statistics Section */}
            {activeSection === 'stats' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Quick Statistics</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">These appear at the bottom of the hero section</p>
                    
                    {/* Stat 1 */}
                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4">
                        <h4 className="font-medium text-gray-900 dark:text-gray-200 mb-3">📊 Statistic 1</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="stat1Value">Value</Label>
                                <Input
                                    id="stat1Value"
                                    value={content.stat1Value || ''}
                                    onChange={(e) => setContent({...content, stat1Value: e.target.value})}
                                    placeholder="e.g., 50,000+"
                                />
                            </div>
                            <div>
                                <Label htmlFor="stat1Label">Label</Label>
                                <Input
                                    id="stat1Label"
                                    value={content.stat1Label || ''}
                                    onChange={(e) => setContent({...content, stat1Label: e.target.value})}
                                    placeholder="e.g., Happy Students"
                                />
                            </div>
                        </div>
                    </div>

                    {/* Stat 2 */}
                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4">
                        <h4 className="font-medium text-gray-900 dark:text-gray-200 mb-3">🏆 Statistic 2</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="stat2Value">Value</Label>
                                <Input
                                    id="stat2Value"
                                    value={content.stat2Value || ''}
                                    onChange={(e) => setContent({...content, stat2Value: e.target.value})}
                                    placeholder="e.g., 98%"
                                />
                            </div>
                            <div>
                                <Label htmlFor="stat2Label">Label</Label>
                                <Input
                                    id="stat2Label"
                                    value={content.stat2Label || ''}
                                    onChange={(e) => setContent({...content, stat2Label: e.target.value})}
                                    placeholder="e.g., Satisfaction Rate"
                                />
                            </div>
                        </div>
                    </div>

                    {/* Stat 3 */}
                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4">
                        <h4 className="font-medium text-gray-900 dark:text-gray-200 mb-3">⏱️ Statistic 3</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="stat3Value">Value</Label>
                                <Input
                                    id="stat3Value"
                                    value={content.stat3Value || ''}
                                    onChange={(e) => setContent({...content, stat3Value: e.target.value})}
                                    placeholder="e.g., 24/7"
                                />
                            </div>
                            <div>
                                <Label htmlFor="stat3Label">Label</Label>
                                <Input
                                    id="stat3Label"
                                    value={content.stat3Label || ''}
                                    onChange={(e) => setContent({...content, stat3Label: e.target.value})}
                                    placeholder="e.g., Support Available"
                                />
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Images Section */}
            {activeSection === 'media' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Hero Images</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="heroImage">Hero Image URL</Label>
                            <Input
                                id="heroImage"
                                value={content.heroImage || ''}
                                onChange={(e) => setContent({...content, heroImage: e.target.value})}
                                placeholder="/images/hero.webp"
                            />
                        </div>
                        <div>
                            <Label htmlFor="heroAlt">Image Alt Text</Label>
                            <Input
                                id="heroAlt"
                                value={content.heroAlt || ''}
                                onChange={(e) => setContent({...content, heroAlt: e.target.value})}
                                placeholder="Describe the image for accessibility"
                            />
                        </div>
                    </div>
                    {content.heroImage && (
                        <div className="mt-4">
                            <Label>Preview</Label>
                            <img 
                                src={content.heroImage} 
                                alt={content.heroAlt || 'Hero preview'} 
                                className="max-w-md rounded-lg shadow-lg border border-gray-200 dark:border-gray-700"
                                onError={(e) => e.target.src = 'https://via.placeholder.com/400x300?text=Image+Not+Found'}
                            />
                        </div>
                    )}
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Hero Changes
                </button>
            </div>
        </div>
    );
}

function AboutBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [features, setFeatures] = useState(content.features || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        onUpdate(block.id, {...content, features});
    };

    const addFeature = () => {
        setFeatures([...features, { title: '', description: '' }]);
    };

    const updateFeature = (index, field, value) => {
        const updated = [...features];
        updated[index][field] = value;
        setFeatures(updated);
    };

    const removeFeature = (index) => {
        setFeatures(features.filter((_, i) => i !== index));
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'mission', label: 'Mission', icon: '🎯' },
        { id: 'features', label: 'Features', icon: '✨' },
        { id: 'media', label: 'Images', icon: '🖼️' },
    ];

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Main About Content</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., About Academic Scribe"
                        />
                    </div>

                    <div>
                        <Label htmlFor="description">Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={300}
                        />
                        <p className="text-xs text-gray-500 mt-1">Main description about your service</p>
                    </div>
                </div>
            )}

            {/* Mission Tab */}
            {activeTab === 'mission' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Mission Section</h3>
                    <div className="border border-blue-200 dark:border-blue-800 rounded-lg p-4 bg-blue-50 dark:bg-blue-900/20">
                        <p className="text-sm text-blue-800 dark:text-blue-200 mb-4">
                            This appears in a highlighted gradient box on the homepage
                        </p>
                        
                        <div className="space-y-4">
                            <div>
                                <Label htmlFor="mission_title">Mission Title</Label>
                                <Input
                                    id="mission_title"
                                    value={content.mission_title || ''}
                                    onChange={(e) => setContent({...content, mission_title: e.target.value})}
                                    placeholder="e.g., Our Mission"
                                />
                            </div>

                            <div>
                                <Label htmlFor="mission_description">Mission Description (Rich Text)</Label>
                                <RichTextEditor
                                    value={content.mission_description || ''}
                                    onChange={(value) => setContent({...content, mission_description: value})}
                                    height={250}
                                />
                                <p className="text-xs text-gray-500 mt-1">Your mission statement or core values</p>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Features Tab */}
            {activeTab === 'features' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Key Features</h3>
                        <button
                            type="button"
                            onClick={addFeature}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Feature
                        </button>
                    </div>

                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Features appear in a 2-column grid below the mission section
                    </p>

                    {features.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No features added yet</p>
                            <button
                                type="button"
                                onClick={addFeature}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Your First Feature
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {features.map((feature, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">
                                            Feature {index + 1}
                                        </h4>
                                        <button
                                            type="button"
                                            onClick={() => removeFeature(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="space-y-3">
                                        <div>
                                            <Label htmlFor={`feature-title-${index}`}>Feature Title</Label>
                                            <Input
                                                id={`feature-title-${index}`}
                                                value={feature.title || ''}
                                                onChange={(e) => updateFeature(index, 'title', e.target.value)}
                                                placeholder="e.g., Expert Writers"
                                            />
                                            <p className="text-xs text-gray-500 mt-1">Can include HTML for formatting</p>
                                        </div>

                                        <div>
                                            <Label htmlFor={`feature-description-${index}`}>Feature Description</Label>
                                            <textarea
                                                id={`feature-description-${index}`}
                                                value={feature.description || ''}
                                                onChange={(e) => updateFeature(index, 'description', e.target.value)}
                                                rows={3}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                                placeholder="Brief description of this feature"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Images Tab */}
            {activeTab === 'media' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">About Section Images</h3>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="about_image">About Image URL</Label>
                            <Input
                                id="about_image"
                                value={content.about_image || ''}
                                onChange={(e) => setContent({...content, about_image: e.target.value})}
                                placeholder="/images/about.webp"
                            />
                        </div>
                        <div>
                            <Label htmlFor="about_image_alt">Image Alt Text</Label>
                            <Input
                                id="about_image_alt"
                                value={content.about_image_alt || ''}
                                onChange={(e) => setContent({...content, about_image_alt: e.target.value})}
                                placeholder="Describe the image for accessibility"
                            />
                        </div>
                    </div>

                    {content.about_image && (
                        <div className="mt-4">
                            <Label>Preview</Label>
                            <img 
                                src={content.about_image} 
                                alt={content.about_image_alt || 'About preview'} 
                                className="max-w-md rounded-lg shadow-lg border border-gray-200 dark:border-gray-700"
                                onError={(e) => e.target.src = 'https://via.placeholder.com/400x300?text=Image+Not+Found'}
                            />
                        </div>
                    )}
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save About Section
                </button>
            </div>
        </div>
    );
}

function OverviewBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>
            
            <div>
                <Label htmlFor="description">Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={250}
                />
            </div>
            
            <div>
                <Label htmlFor="image">Image URL</Label>
                <Input
                    id="image"
                    value={content.image || ''}
                    onChange={(e) => setContent({...content, image: e.target.value})}
                />
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function BenefitsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [benefits, setBenefits] = useState(content.benefits || []);

    const handleSave = () => {
        onUpdate(block.id, {...content, benefits});
    };

    const addBenefit = () => {
        setBenefits([...benefits, { title: '', description: '', icon: '' }]);
    };

    const updateBenefit = (index, field, value) => {
        const updated = [...benefits];
        updated[index][field] = value;
        setBenefits(updated);
    };

    const removeBenefit = (index) => {
        setBenefits(benefits.filter((_, i) => i !== index));
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>
            
            <div>
                <Label htmlFor="subtitle">Subtitle</Label>
                <Input
                    id="subtitle"
                    value={content.subtitle || ''}
                    onChange={(e) => setContent({...content, subtitle: e.target.value})}
                />
            </div>
            
            <div>
                <div className="flex justify-between items-center mb-2">
                    <Label>Benefits</Label>
                    <Button variant="outline" size="sm" onClick={addBenefit}>
                        <Plus className="h-4 w-4 mr-2" />
                        Add Benefit
                    </Button>
                </div>
                
                <div className="space-y-3">
                    {benefits.map((benefit, index) => (
                        <Card key={index} className="p-4">
                            <div className="flex justify-between items-start mb-3">
                                <h4 className="font-medium">Benefit {index + 1}</h4>
                                <Button
                                    variant="ghost"
                                    size="sm"
                                    onClick={() => removeBenefit(index)}
                                    className="text-red-600"
                                >
                                    <X className="h-4 w-4" />
                                </Button>
                            </div>
                            
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                <div>
                                    <Label>Title</Label>
                                    <Input
                                        value={benefit.title || ''}
                                        onChange={(e) => updateBenefit(index, 'title', e.target.value)}
                                    />
                                </div>
                                <div>
                                    <Label>Icon</Label>
                                    <Input
                                        value={benefit.icon || ''}
                                        onChange={(e) => updateBenefit(index, 'icon', e.target.value)}
                                        placeholder="Icon name or emoji"
                                    />
                                </div>
                            </div>
                            <div className="mt-3">
                                <Label>Description</Label>
                                <RichTextEditor
                                    value={benefit.description || ''}
                                    onChange={(value) => updateBenefit(index, 'description', value)}
                                    height={150}
                                />
                            </div>
                        </Card>
                    ))}
                </div>
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function ProcessBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [steps, setSteps] = useState(content.steps || []);

    const handleSave = () => {
        onUpdate(block.id, {...content, steps});
    };

    const addStep = () => {
        setSteps([...steps, { title: '', description: '', number: steps.length + 1 }]);
    };

    const updateStep = (index, field, value) => {
        const updated = [...steps];
        updated[index][field] = value;
        setSteps(updated);
    };

    const removeStep = (index) => {
        setSteps(steps.filter((_, i) => i !== index));
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>
            
            <div>
                <Label htmlFor="subtitle">Subtitle</Label>
                <Input
                    id="subtitle"
                    value={content.subtitle || ''}
                    onChange={(e) => setContent({...content, subtitle: e.target.value})}
                />
            </div>
            
            <div>
                <div className="flex justify-between items-center mb-2">
                    <Label>Process Steps</Label>
                    <Button variant="outline" size="sm" onClick={addStep}>
                        <Plus className="h-4 w-4 mr-2" />
                        Add Step
                    </Button>
                </div>
                
                <div className="space-y-3">
                    {steps.map((step, index) => (
                        <Card key={index} className="p-4">
                            <div className="flex justify-between items-start mb-3">
                                <h4 className="font-medium">Step {step.number || index + 1}</h4>
                                <Button
                                    variant="ghost"
                                    size="sm"
                                    onClick={() => removeStep(index)}
                                    className="text-red-600"
                                >
                                    <X className="h-4 w-4" />
                                </Button>
                            </div>
                            
                            <div className="space-y-3">
                                <div>
                                    <Label>Title</Label>
                                    <Input
                                        value={step.title || ''}
                                        onChange={(e) => updateStep(index, 'title', e.target.value)}
                                    />
                                </div>
                                <div>
                                    <Label>Description</Label>
                                    <RichTextEditor
                                        value={step.description || ''}
                                        onChange={(value) => updateStep(index, 'description', value)}
                                        height={150}
                                    />
                                </div>
                            </div>
                        </Card>
                    ))}
                </div>
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function TestimonialsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [categories, setCategories] = useState(content.categories || []);
    const [overallStats, setOverallStats] = useState(content.overallStats || {});
    const [serviceHighlights, setServiceHighlights] = useState(content.serviceHighlights || []);
    const [trustIndicators, setTrustIndicators] = useState(content.trustIndicators || []);
    const [activeTab, setActiveTab] = useState('header');
    const [activeCategory, setActiveCategory] = useState(0);

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            categories,
            overallStats,
            serviceHighlights,
            trustIndicators,
        });
    };

    const tabs = [
        { id: 'header', label: 'Header', icon: '📝' },
        { id: 'stats', label: 'Overall Stats', icon: '📊' },
        { id: 'categories', label: 'Categories', icon: '🗂️' },
        { id: 'highlights', label: 'Service Highlights', icon: '⭐' },
        { id: 'trust', label: 'Trust Indicators', icon: '✓' },
    ];

    // Category management
    const addCategory = () => {
        setCategories([...categories, { name: '', description: '', testimonials: [] }]);
        setActiveCategory(categories.length);
    };

    const updateCategory = (index, field, value) => {
        const updated = [...categories];
        updated[index][field] = value;
        setCategories(updated);
    };

    const removeCategory = (index) => {
        setCategories(categories.filter((_, i) => i !== index));
        if (activeCategory >= categories.length - 1) {
            setActiveCategory(Math.max(0, categories.length - 2));
        }
    };

    // Testimonial management within category
    const addTestimonial = (categoryIndex) => {
        const updated = [...categories];
        if (!updated[categoryIndex].testimonials) {
            updated[categoryIndex].testimonials = [];
        }
        updated[categoryIndex].testimonials.push({
            text: '',
            author: '',
            role: '',
            university: '',
            grade: '',
            service: '',
            rating: 5,
            date: '',
            verified: true,
            image: '',
        });
        setCategories(updated);
    };

    const updateTestimonial = (categoryIndex, testimonialIndex, field, value) => {
        const updated = [...categories];
        updated[categoryIndex].testimonials[testimonialIndex][field] = value;
        setCategories(updated);
    };

    const removeTestimonial = (categoryIndex, testimonialIndex) => {
        const updated = [...categories];
        updated[categoryIndex].testimonials = updated[categoryIndex].testimonials.filter((_, i) => i !== testimonialIndex);
        setCategories(updated);
    };

    // Service Highlights management
    const addServiceHighlight = () => {
        setServiceHighlights([...serviceHighlights, {
            service: '',
            satisfaction: '',
            avgGrade: '',
            completionRate: '',
            color: 'blue',
            testimonialCount: '',
        }]);
    };

    const updateServiceHighlight = (index, field, value) => {
        const updated = [...serviceHighlights];
        updated[index][field] = value;
        setServiceHighlights(updated);
    };

    const removeServiceHighlight = (index) => {
        setServiceHighlights(serviceHighlights.filter((_, i) => i !== index));
    };

    // Trust Indicators management
    const addTrustIndicator = () => {
        setTrustIndicators([...trustIndicators, { title: '', description: '' }]);
    };

    const updateTrustIndicator = (index, field, value) => {
        const updated = [...trustIndicators];
        updated[index][field] = value;
        setTrustIndicators(updated);
    };

    const removeTrustIndicator = (index) => {
        setTrustIndicators(trustIndicators.filter((_, i) => i !== index));
    };

    const colorOptions = [
        { value: 'blue', label: '🔵 Blue' },
        { value: 'green', label: '🟢 Green' },
        { value: 'purple', label: '🟣 Purple' },
        { value: 'red', label: '🔴 Red' },
        { value: 'yellow', label: '🟡 Yellow' },
        { value: 'orange', label: '🟠 Orange' },
    ];

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Header Tab */}
            {activeTab === 'header' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Testimonials Section Header</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., What Our Clients Say"
                        />
                    </div>

                    <div>
                        <Label htmlFor="subtitle">Subtitle (Rich Text)</Label>
                        <RichTextEditor
                            value={content.subtitle || ''}
                            onChange={(value) => setContent({...content, subtitle: value})}
                            height={150}
                        />
                    </div>
                </div>
            )}

            {/* Overall Stats Tab */}
            {activeTab === 'stats' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Overall Statistics</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">4-column stat cards shown above testimonials</p>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label>Average Rating (e.g., 4.9)</Label>
                            <Input
                                value={overallStats.averageRating || ''}
                                onChange={(e) => setOverallStats({...overallStats, averageRating: e.target.value})}
                                placeholder="4.9"
                            />
                        </div>

                        <div>
                            <Label>Total Reviews (number)</Label>
                            <Input
                                type="number"
                                value={overallStats.totalReviews || ''}
                                onChange={(e) => setOverallStats({...overallStats, totalReviews: e.target.value})}
                                placeholder="10000"
                            />
                        </div>

                        <div>
                            <Label>Recommendation Rate (%)</Label>
                            <Input
                                type="number"
                                min="0"
                                max="100"
                                value={overallStats.recommendationRate || ''}
                                onChange={(e) => setOverallStats({...overallStats, recommendationRate: e.target.value})}
                                placeholder="98"
                            />
                        </div>

                        <div>
                            <Label>Repeat Customers (%)</Label>
                            <Input
                                type="number"
                                min="0"
                                max="100"
                                value={overallStats.repeatCustomers || ''}
                                onChange={(e) => setOverallStats({...overallStats, repeatCustomers: e.target.value})}
                                placeholder="85"
                            />
                        </div>
                    </div>
                </div>
            )}

            {/* Categories Tab */}
            {activeTab === 'categories' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                            Testimonial Categories ({categories.length})
                        </h3>
                        <button
                            type="button"
                            onClick={addCategory}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Category
                        </button>
                    </div>

                    {categories.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No categories yet</p>
                            <button
                                type="button"
                                onClick={addCategory}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Your First Category
                            </button>
                        </div>
                    ) : (
                        <>
                            {/* Category Tabs */}
                            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                                {categories.map((category, index) => (
                                    <button
                                        key={index}
                                        type="button"
                                        onClick={() => setActiveCategory(index)}
                                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                                            activeCategory === index
                                                ? 'bg-purple-600 text-white'
                                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                                        }`}
                                    >
                                        {category.name || `Category ${index + 1}`}
                                        <span className="ml-2 text-xs opacity-75">
                                            ({(category.testimonials || []).length})
                                        </span>
                                    </button>
                                ))}
                            </div>

                            {/* Active Category Editor */}
                            {categories[activeCategory] && (
                                <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-6 bg-white dark:bg-gray-800">
                                    {/* Category Details */}
                                    <div className="space-y-4 mb-6">
                                        <div className="flex justify-between items-center">
                                            <h4 className="font-semibold text-gray-900 dark:text-white">Category Details</h4>
                                            <button
                                                type="button"
                                                onClick={() => removeCategory(activeCategory)}
                                                className="text-red-600 hover:text-red-700 dark:text-red-400 text-sm"
                                            >
                                                <Trash2 className="h-4 w-4 inline mr-1" />
                                                Delete Category
                                            </button>
                                        </div>

                                        <div>
                                            <Label>Category Name</Label>
                                            <Input
                                                value={categories[activeCategory].name || ''}
                                                onChange={(e) => updateCategory(activeCategory, 'name', e.target.value)}
                                                placeholder="e.g., Essay Writing"
                                            />
                                        </div>

                                        <div>
                                            <Label>Category Description</Label>
                                            <Input
                                                value={categories[activeCategory].description || ''}
                                                onChange={(e) => updateCategory(activeCategory, 'description', e.target.value)}
                                                placeholder="Brief description"
                                            />
                                        </div>
                                    </div>

                                    {/* Testimonials in Category */}
                                    <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                        <div className="flex justify-between items-center mb-4">
                                            <h4 className="font-semibold text-gray-900 dark:text-white">
                                                Testimonials ({(categories[activeCategory].testimonials || []).length})
                                            </h4>
                                            <button
                                                type="button"
                                                onClick={() => addTestimonial(activeCategory)}
                                                className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                                            >
                                                <Plus className="h-4 w-4 mr-1" />
                                                Add Testimonial
                                            </button>
                                        </div>

                                        {(!categories[activeCategory].testimonials || categories[activeCategory].testimonials.length === 0) ? (
                                            <div className="text-center py-8 text-gray-500 dark:text-gray-400 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-900">
                                                No testimonials yet. Click "Add Testimonial" to start.
                                            </div>
                                        ) : (
                                            <div className="space-y-6">
                                                {categories[activeCategory].testimonials.map((testimonial, testimonialIndex) => (
                                                    <div key={testimonialIndex} className="border border-gray-300 dark:border-gray-600 rounded-lg p-4 bg-gray-50 dark:bg-gray-900">
                                                        <div className="flex justify-between items-center mb-3">
                                                            <span className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                                                                Testimonial #{testimonialIndex + 1}
                                                            </span>
                                                            <button
                                                                type="button"
                                                                onClick={() => removeTestimonial(activeCategory, testimonialIndex)}
                                                                className="text-red-600 hover:text-red-700 dark:text-red-400"
                                                            >
                                                                <Trash2 className="h-4 w-4" />
                                                            </button>
                                                        </div>

                                                        <div className="space-y-3">
                                                            <div>
                                                                <Label>Testimonial Text</Label>
                                                                <textarea
                                                                    value={testimonial.text || ''}
                                                                    onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'text', e.target.value)}
                                                                    rows={4}
                                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                                                    placeholder="The full testimonial text..."
                                                                />
                                                            </div>

                                                            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                                                <div>
                                                                    <Label>Author Name</Label>
                                                                    <Input
                                                                        value={testimonial.author || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'author', e.target.value)}
                                                                        placeholder="e.g., John Doe"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Role/Year</Label>
                                                                    <Input
                                                                        value={testimonial.role || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'role', e.target.value)}
                                                                        placeholder="e.g., Senior Student"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>University</Label>
                                                                    <Input
                                                                        value={testimonial.university || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'university', e.target.value)}
                                                                        placeholder="e.g., Harvard"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Grade Received</Label>
                                                                    <Input
                                                                        value={testimonial.grade || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'grade', e.target.value)}
                                                                        placeholder="e.g., A+"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Service Type</Label>
                                                                    <Input
                                                                        value={testimonial.service || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'service', e.target.value)}
                                                                        placeholder="e.g., Essay Writing"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Rating (1-5)</Label>
                                                                    <Input
                                                                        type="number"
                                                                        min="1"
                                                                        max="5"
                                                                        value={testimonial.rating || 5}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'rating', parseInt(e.target.value))}
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Date</Label>
                                                                    <Input
                                                                        value={testimonial.date || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'date', e.target.value)}
                                                                        placeholder="e.g., March 2024"
                                                                    />
                                                                </div>

                                                                <div>
                                                                    <Label>Image URL (optional)</Label>
                                                                    <Input
                                                                        value={testimonial.image || ''}
                                                                        onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'image', e.target.value)}
                                                                        placeholder="/images/client.jpg"
                                                                    />
                                                                </div>
                                                            </div>

                                                            <div className="flex items-center">
                                                                <input
                                                                    type="checkbox"
                                                                    checked={testimonial.verified || false}
                                                                    onChange={(e) => updateTestimonial(activeCategory, testimonialIndex, 'verified', e.target.checked)}
                                                                    className="mr-2"
                                                                />
                                                                <Label>Verified Review (shows green checkmark)</Label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>
                                        )}
                                    </div>
                                </div>
                            )}
                        </>
                    )}
                </div>
            )}

            {/* Service Highlights Tab */}
            {activeTab === 'highlights' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Highlights</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Service-specific statistics</p>
                        </div>
                        <button
                            type="button"
                            onClick={addServiceHighlight}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Highlight
                        </button>
                    </div>

                    {serviceHighlights.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No service highlights yet</p>
                            <button
                                type="button"
                                onClick={addServiceHighlight}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add First Highlight
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {serviceHighlights.map((highlight, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Highlight {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeServiceHighlight(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                        <div>
                                            <Label>Service Name</Label>
                                            <Input
                                                value={highlight.service || ''}
                                                onChange={(e) => updateServiceHighlight(index, 'service', e.target.value)}
                                                placeholder="e.g., Dissertation Writing"
                                            />
                                        </div>

                                        <div>
                                            <Label>Satisfaction Rate (%)</Label>
                                            <Input
                                                value={highlight.satisfaction || ''}
                                                onChange={(e) => updateServiceHighlight(index, 'satisfaction', e.target.value)}
                                                placeholder="97"
                                            />
                                        </div>

                                        <div>
                                            <Label>Average Grade</Label>
                                            <Input
                                                value={highlight.avgGrade || ''}
                                                onChange={(e) => updateServiceHighlight(index, 'avgGrade', e.target.value)}
                                                placeholder="A+"
                                            />
                                        </div>

                                        <div>
                                            <Label>Completion Rate (%)</Label>
                                            <Input
                                                value={highlight.completionRate || ''}
                                                onChange={(e) => updateServiceHighlight(index, 'completionRate', e.target.value)}
                                                placeholder="98"
                                            />
                                        </div>

                                        <div>
                                            <Label>Color</Label>
                                            <select
                                                value={highlight.color || 'blue'}
                                                onChange={(e) => updateServiceHighlight(index, 'color', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {colorOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Testimonial Count</Label>
                                            <Input
                                                type="number"
                                                value={highlight.testimonialCount || ''}
                                                onChange={(e) => updateServiceHighlight(index, 'testimonialCount', e.target.value)}
                                                placeholder="150"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Trust Indicators Tab */}
            {activeTab === 'trust' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Trust Indicators</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Trust badges and guarantees</p>
                        </div>
                        <button
                            type="button"
                            onClick={addTrustIndicator}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Indicator
                        </button>
                    </div>

                    {trustIndicators.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No trust indicators yet</p>
                            <button
                                type="button"
                                onClick={addTrustIndicator}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add First Indicator
                            </button>
                        </div>
                    ) : (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            {trustIndicators.map((indicator, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Indicator {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeTrustIndicator(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="space-y-3">
                                        <div>
                                            <Label>Title</Label>
                                            <Input
                                                value={indicator.title || ''}
                                                onChange={(e) => updateTrustIndicator(index, 'title', e.target.value)}
                                                placeholder="e.g., Money Back Guarantee"
                                            />
                                        </div>

                                        <div>
                                            <Label>Description</Label>
                                            <textarea
                                                value={indicator.description || ''}
                                                onChange={(e) => updateTrustIndicator(index, 'description', e.target.value)}
                                                rows={3}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                                placeholder="Brief description of this guarantee"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Testimonials Section
                </button>
            </div>
        </div>
    );
}

function FAQsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [categories, setCategories] = useState(content.categories || []);
    const [activeCategory, setActiveCategory] = useState(0);

    const handleSave = () => {
        onUpdate(block.id, { ...content, categories });
    };

    // Category management
    const addCategory = () => {
        const newCategory = { name: '', description: '', faqs: [] };
        setCategories([...categories, newCategory]);
        setActiveCategory(categories.length);
    };

    const updateCategory = (index, field, value) => {
        const updated = [...categories];
        updated[index][field] = value;
        setCategories(updated);
    };

    const removeCategory = (index) => {
        setCategories(categories.filter((_, i) => i !== index));
        if (activeCategory >= categories.length - 1) {
            setActiveCategory(Math.max(0, categories.length - 2));
        }
    };

    // FAQ management within a category
    const addFAQ = (categoryIndex) => {
        const updated = [...categories];
        if (!updated[categoryIndex].faqs) {
            updated[categoryIndex].faqs = [];
        }
        updated[categoryIndex].faqs.push({ question: '', answer: '', keywords: [] });
        setCategories(updated);
    };

    const updateFAQ = (categoryIndex, faqIndex, field, value) => {
        const updated = [...categories];
        updated[categoryIndex].faqs[faqIndex][field] = value;
        setCategories(updated);
    };

    const removeFAQ = (categoryIndex, faqIndex) => {
        const updated = [...categories];
        updated[categoryIndex].faqs = updated[categoryIndex].faqs.filter((_, i) => i !== faqIndex);
        setCategories(updated);
    };

    const moveFaqUp = (categoryIndex, faqIndex) => {
        if (faqIndex === 0) return;
        const updated = [...categories];
        const faqs = updated[categoryIndex].faqs;
        [faqs[faqIndex - 1], faqs[faqIndex]] = [faqs[faqIndex], faqs[faqIndex - 1]];
        setCategories(updated);
    };

    const moveFaqDown = (categoryIndex, faqIndex) => {
        const updated = [...categories];
        const faqs = updated[categoryIndex].faqs;
        if (faqIndex === faqs.length - 1) return;
        [faqs[faqIndex], faqs[faqIndex + 1]] = [faqs[faqIndex + 1], faqs[faqIndex]];
        setCategories(updated);
    };

    // Keywords management
    const addKeyword = (categoryIndex, faqIndex) => {
        const updated = [...categories];
        if (!updated[categoryIndex].faqs[faqIndex].keywords) {
            updated[categoryIndex].faqs[faqIndex].keywords = [];
        }
        updated[categoryIndex].faqs[faqIndex].keywords.push('');
        setCategories(updated);
    };

    const updateKeyword = (categoryIndex, faqIndex, keywordIndex, value) => {
        const updated = [...categories];
        updated[categoryIndex].faqs[faqIndex].keywords[keywordIndex] = value;
        setCategories(updated);
    };

    const removeKeyword = (categoryIndex, faqIndex, keywordIndex) => {
        const updated = [...categories];
        updated[categoryIndex].faqs[faqIndex].keywords = 
            updated[categoryIndex].faqs[faqIndex].keywords.filter((_, i) => i !== keywordIndex);
        setCategories(updated);
    };

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="space-y-4 pb-4 border-b border-gray-200 dark:border-gray-700">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">FAQs Section Header</h3>
                
                <div>
                    <Label htmlFor="title">Section Title</Label>
                    <Input
                        id="title"
                        value={content.title || ''}
                        onChange={(e) => setContent({...content, title: e.target.value})}
                        placeholder="e.g., Frequently Asked Questions"
                    />
                </div>
                
                <div>
                    <Label htmlFor="subtitle">Subtitle</Label>
                    <Input
                        id="subtitle"
                        value={content.subtitle || ''}
                        onChange={(e) => setContent({...content, subtitle: e.target.value})}
                        placeholder="Brief description"
                    />
                </div>
            </div>

            {/* Categories Management */}
            <div className="space-y-4">
                <div className="flex justify-between items-center">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                        FAQ Categories ({categories.length})
                    </h3>
                    <button
                        type="button"
                        onClick={addCategory}
                        className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                    >
                        <Plus className="h-4 w-4 mr-2" />
                        Add Category
                    </button>
                </div>

                {categories.length === 0 ? (
                    <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                        <p className="text-gray-500 dark:text-gray-400 mb-3">No FAQ categories yet</p>
                        <button
                            type="button"
                            onClick={addCategory}
                            className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Your First Category
                        </button>
                    </div>
                ) : (
                    <>
                        {/* Category Tabs */}
                        <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                            {categories.map((category, index) => (
                                <button
                                    key={index}
                                    type="button"
                                    onClick={() => setActiveCategory(index)}
                                    className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                                        activeCategory === index
                                            ? 'bg-blue-600 text-white'
                                            : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                                    }`}
                                >
                                    {category.name || `Category ${index + 1}`}
                                    <span className="ml-2 text-xs opacity-75">
                                        ({(category.faqs || []).length} FAQs)
                                    </span>
                                </button>
                            ))}
                        </div>

                        {/* Active Category Editor */}
                        {categories[activeCategory] && (
                            <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-6 bg-white dark:bg-gray-800">
                                {/* Category Details */}
                                <div className="space-y-4 mb-6">
                                    <div className="flex justify-between items-center">
                                        <h4 className="font-semibold text-gray-900 dark:text-white">
                                            Category Details
                                        </h4>
                                        <button
                                            type="button"
                                            onClick={() => removeCategory(activeCategory)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 transition-colors text-sm"
                                        >
                                            <Trash2 className="h-4 w-4 inline mr-1" />
                                            Delete Category
                                        </button>
                                    </div>

                                    <div>
                                        <Label>Category Name</Label>
                                        <Input
                                            value={categories[activeCategory].name || ''}
                                            onChange={(e) => updateCategory(activeCategory, 'name', e.target.value)}
                                            placeholder="e.g., General Questions"
                                        />
                                    </div>

                                    <div>
                                        <Label>Category Description</Label>
                                        <Input
                                            value={categories[activeCategory].description || ''}
                                            onChange={(e) => updateCategory(activeCategory, 'description', e.target.value)}
                                            placeholder="Brief description of this category"
                                        />
                                    </div>
                                </div>

                                {/* FAQs in Category */}
                                <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                    <div className="flex justify-between items-center mb-4">
                                        <h4 className="font-semibold text-gray-900 dark:text-white">
                                            FAQs ({(categories[activeCategory].faqs || []).length})
                                        </h4>
                                        <button
                                            type="button"
                                            onClick={() => addFAQ(activeCategory)}
                                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                                        >
                                            <Plus className="h-4 w-4 mr-1" />
                                            Add FAQ
                                        </button>
                                    </div>

                                    {(!categories[activeCategory].faqs || categories[activeCategory].faqs.length === 0) ? (
                                        <div className="text-center py-8 text-gray-500 dark:text-gray-400 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-900">
                                            No FAQs in this category yet. Click "Add FAQ" to start.
                                        </div>
                                    ) : (
                                        <div className="space-y-6">
                                            {categories[activeCategory].faqs.map((faq, faqIndex) => (
                                                <div key={faqIndex} className="border border-gray-300 dark:border-gray-600 rounded-lg p-4 bg-gray-50 dark:bg-gray-900">
                                                    <div className="flex justify-between items-center mb-3">
                                                        <span className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                                                            FAQ #{faqIndex + 1}
                                                        </span>
                                                        <div className="flex gap-2">
                                                            <button
                                                                type="button"
                                                                onClick={() => moveFaqUp(activeCategory, faqIndex)}
                                                                disabled={faqIndex === 0}
                                                                className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                                                title="Move up"
                                                            >
                                                                <ChevronUp className="h-4 w-4" />
                                                            </button>
                                                            <button
                                                                type="button"
                                                                onClick={() => moveFaqDown(activeCategory, faqIndex)}
                                                                disabled={faqIndex === categories[activeCategory].faqs.length - 1}
                                                                className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                                                title="Move down"
                                                            >
                                                                <ChevronDown className="h-4 w-4" />
                                                            </button>
                                                            <button
                                                                type="button"
                                                                onClick={() => removeFAQ(activeCategory, faqIndex)}
                                                                className="p-1 text-red-500 hover:text-red-700"
                                                                title="Remove FAQ"
                                                            >
                                                                <X className="h-4 w-4" />
                                                            </button>
                                                        </div>
                                                    </div>

                                                    <div className="space-y-3">
                                                        <div>
                                                            <Label>Question</Label>
                                                            <Input
                                                                value={faq.question || ''}
                                                                onChange={(e) => updateFAQ(activeCategory, faqIndex, 'question', e.target.value)}
                                                                placeholder="Enter the FAQ question..."
                                                            />
                                                        </div>

                                                        <div>
                                                            <Label>Answer (TinyMCE)</Label>
                                                            <RichTextEditor
                                                                value={faq.answer || ''}
                                                                onChange={(value) => updateFAQ(activeCategory, faqIndex, 'answer', value)}
                                                                height={200}
                                                            />
                                                        </div>

                                                        <div>
                                                            <div className="flex justify-between items-center mb-2">
                                                                <Label>Keywords (for search)</Label>
                                                                <button
                                                                    type="button"
                                                                    onClick={() => addKeyword(activeCategory, faqIndex)}
                                                                    className="text-sm text-blue-600 hover:text-blue-700 dark:text-blue-400"
                                                                >
                                                                    <Plus className="h-3 w-3 inline mr-1" />
                                                                    Add Keyword
                                                                </button>
                                                            </div>
                                                            <div className="flex flex-wrap gap-2">
                                                                {(faq.keywords || []).map((keyword, kwIndex) => (
                                                                    <div key={kwIndex} className="flex items-center gap-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 px-2 py-1 rounded-full text-sm">
                                                                        <input
                                                                            type="text"
                                                                            value={keyword}
                                                                            onChange={(e) => updateKeyword(activeCategory, faqIndex, kwIndex, e.target.value)}
                                                                            className="bg-transparent border-none focus:outline-none w-24 text-sm"
                                                                            placeholder="keyword"
                                                                        />
                                                                        <button
                                                                            type="button"
                                                                            onClick={() => removeKeyword(activeCategory, faqIndex, kwIndex)}
                                                                            className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-200"
                                                                        >
                                                                            <X className="h-3 w-3" />
                                                                        </button>
                                                                    </div>
                                                                ))}
                                                                {(!faq.keywords || faq.keywords.length === 0) && (
                                                                    <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                        No keywords yet - click "Add Keyword" to add searchable terms
                                                                    </span>
                                                                )}
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>
                            </div>
                        )}
                    </>
                )}
            </div>

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save FAQs Section
                </button>
            </div>
        </div>
    );
}

function ContactFormBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>
            
            <div>
                <Label htmlFor="description">Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={180}
                />
            </div>
            
            <div>
                <Label htmlFor="submit_text">Submit Button Text</Label>
                <Input
                    id="submit_text"
                    value={content.submit_text || 'Submit'}
                    onChange={(e) => setContent({...content, submit_text: e.target.value})}
                />
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function CTABlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [benefits, setBenefits] = useState(content.benefits || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        onUpdate(block.id, {...content, benefits});
    };

    const addBenefit = () => {
        setBenefits([...benefits, { text: '', icon: 'CheckCircle' }]);
    };

    const updateBenefit = (index, field, value) => {
        const updated = [...benefits];
        updated[index][field] = value;
        setBenefits(updated);
    };

    const removeBenefit = (index) => {
        setBenefits(benefits.filter((_, i) => i !== index));
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'benefits', label: 'Benefits', icon: '✨' },
        { id: 'buttons', label: 'Buttons', icon: '🔘' },
        { id: 'styling', label: 'Styling', icon: '🎨' },
    ];

    const iconOptions = [
        { value: 'Award', label: '🏆 Award' },
        { value: 'Clock', label: '⏰ Clock' },
        { value: 'CheckCircle', label: '✓ CheckCircle' },
    ];

    const backgroundOptions = [
        { value: 'blue', label: '🔵 Blue Gradient', preview: 'from-blue-600 to-indigo-700' },
        { value: 'gray', label: '⚫ Gray Gradient', preview: 'from-gray-100 to-gray-200' },
        { value: 'green', label: '🟢 Green Gradient', preview: 'from-green-500 to-teal-600' },
    ];

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">CTA Main Content</h3>
                    
                    <div>
                        <Label htmlFor="title">CTA Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Ready to Get Started?"
                        />
                    </div>

                    <div>
                        <Label htmlFor="description">Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={200}
                        />
                        <p className="text-xs text-gray-500 mt-1">Main call-to-action message</p>
                    </div>
                </div>
            )}

            {/* Benefits Tab */}
            {activeTab === 'benefits' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Key Benefits</h3>
                        <button
                            type="button"
                            onClick={addBenefit}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Benefit
                        </button>
                    </div>

                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Benefits appear as icons with text between the description and buttons
                    </p>

                    {benefits.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No benefits added yet</p>
                            <button
                                type="button"
                                onClick={addBenefit}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Your First Benefit
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {benefits.map((benefit, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">
                                            Benefit {index + 1}
                                        </h4>
                                        <button
                                            type="button"
                                            onClick={() => removeBenefit(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                        <div>
                                            <Label htmlFor={`benefit-icon-${index}`}>Icon</Label>
                                            <select
                                                id={`benefit-icon-${index}`}
                                                value={benefit.icon || 'CheckCircle'}
                                                onChange={(e) => updateBenefit(index, 'icon', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {iconOptions.map((option) => (
                                                    <option key={option.value} value={option.value}>
                                                        {option.label}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label htmlFor={`benefit-text-${index}`}>Benefit Text</Label>
                                            <Input
                                                id={`benefit-text-${index}`}
                                                value={benefit.text || ''}
                                                onChange={(e) => updateBenefit(index, 'text', e.target.value)}
                                                placeholder="e.g., On-time delivery guaranteed"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Buttons Tab */}
            {activeTab === 'buttons' && (
                <div className="space-y-6">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">CTA Buttons</h3>
                    
                    {/* Primary Button */}
                    <div className="border border-blue-200 dark:border-blue-800 rounded-lg p-4 bg-blue-50 dark:bg-blue-900/20">
                        <h4 className="font-medium text-blue-900 dark:text-blue-200 mb-3">Primary Button (Solid)</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="primary_button_text">Button Text</Label>
                                <Input
                                    id="primary_button_text"
                                    value={content.primary_button_text || ''}
                                    onChange={(e) => setContent({...content, primary_button_text: e.target.value})}
                                    placeholder="e.g., Place an Order"
                                />
                            </div>
                            <div>
                                <Label htmlFor="primary_button_link">Button Link</Label>
                                <Input
                                    id="primary_button_link"
                                    value={content.primary_button_link || ''}
                                    onChange={(e) => setContent({...content, primary_button_link: e.target.value})}
                                    placeholder="/place-order"
                                />
                            </div>
                        </div>
                    </div>

                    {/* Secondary Button */}
                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-gray-50 dark:bg-gray-800/50">
                        <h4 className="font-medium text-gray-900 dark:text-gray-200 mb-3">Secondary Button (Outline)</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <Label htmlFor="secondary_button_text">Button Text</Label>
                                <Input
                                    id="secondary_button_text"
                                    value={content.secondary_button_text || ''}
                                    onChange={(e) => setContent({...content, secondary_button_text: e.target.value})}
                                    placeholder="e.g., Contact Us"
                                />
                            </div>
                            <div>
                                <Label htmlFor="secondary_button_link">Button Link</Label>
                                <Input
                                    id="secondary_button_link"
                                    value={content.secondary_button_link || ''}
                                    onChange={(e) => setContent({...content, secondary_button_link: e.target.value})}
                                    placeholder="#contact"
                                />
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Styling Tab */}
            {activeTab === 'styling' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">CTA Styling</h3>
                    
                    <div>
                        <Label htmlFor="background_color">Background Color Theme</Label>
                        <div className="space-y-2 mt-2">
                            {backgroundOptions.map((option) => (
                                <label
                                    key={option.value}
                                    className={`flex items-center p-3 border-2 rounded-lg cursor-pointer transition-all ${
                                        content.background_color === option.value
                                            ? 'border-blue-600 bg-blue-50 dark:bg-blue-900/20'
                                            : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600'
                                    }`}
                                >
                                    <input
                                        type="radio"
                                        name="background_color"
                                        value={option.value}
                                        checked={content.background_color === option.value}
                                        onChange={(e) => setContent({...content, background_color: e.target.value})}
                                        className="mr-3"
                                    />
                                    <div className="flex items-center flex-1">
                                        <span className="font-medium text-gray-900 dark:text-white mr-3">
                                            {option.label}
                                        </span>
                                        <div className={`w-24 h-8 rounded bg-gradient-to-r ${option.preview} ml-auto`}></div>
                                    </div>
                                </label>
                            ))}
                        </div>
                        <p className="text-xs text-gray-500 mt-2">
                            Choose the background gradient for the CTA section
                        </p>
                    </div>
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save CTA Section
                </button>
            </div>
        </div>
    );
}

function StatsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [mainStats, setMainStats] = useState(content.main_stats || []);
    const [writerStats, setWriterStats] = useState(content.writer_stats || []);
    const [additionalStats, setAdditionalStats] = useState(content.additional_stats || []);
    const [expertiseDistribution, setExpertiseDistribution] = useState(content.expertise_distribution || []);
    const [activeTab, setActiveTab] = useState('header');

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            main_stats: mainStats,
            writer_stats: writerStats,
            additional_stats: additionalStats,
            expertise_distribution: expertiseDistribution,
        });
    };

    const tabs = [
        { id: 'header', label: 'Header', icon: '📝' },
        { id: 'main', label: 'Main Stats', icon: '📊' },
        { id: 'additional', label: 'Additional', icon: '➕' },
        { id: 'writers', label: 'Writer Stats', icon: '✍️' },
        { id: 'expertise', label: 'Expertise', icon: '🎓' },
    ];

    const iconOptions = [
        { value: 'heart', label: '❤️ Heart' },
        { value: 'book-open', label: '📖 Book' },
        { value: 'users', label: '👥 Users' },
        { value: 'target', label: '🎯 Target' },
        { value: 'clock', label: '⏰ Clock' },
        { value: 'globe', label: '🌐 Globe' },
        { value: 'shield', label: '🛡️ Shield' },
        { value: 'award', label: '🏆 Award' },
        { value: 'star', label: '⭐ Star' },
        { value: 'zap', label: '⚡ Zap' },
        { value: 'graduation-cap', label: '🎓 Graduation' },
        { value: 'file-text', label: '📄 File' },
    ];

    const colorOptions = [
        { value: 'red', label: '🔴 Red' },
        { value: 'blue', label: '🔵 Blue' },
        { value: 'green', label: '🟢 Green' },
        { value: 'purple', label: '🟣 Purple' },
        { value: 'yellow', label: '🟡 Yellow' },
        { value: 'orange', label: '🟠 Orange' },
    ];

    // Main Stats handlers
    const addMainStat = () => {
        setMainStats([...mainStats, { icon: 'book-open', color: 'blue', label: '', number: '', description: '', detail: '' }]);
    };

    const updateMainStat = (index, field, value) => {
        const updated = [...mainStats];
        updated[index][field] = value;
        setMainStats(updated);
    };

    const removeMainStat = (index) => {
        setMainStats(mainStats.filter((_, i) => i !== index));
    };

    // Writer Stats handlers
    const addWriterStat = () => {
        setWriterStats([...writerStats, { count: '', label: '', percentage: '0', description: '' }]);
    };

    const updateWriterStat = (index, field, value) => {
        const updated = [...writerStats];
        updated[index][field] = value;
        setWriterStats(updated);
    };

    const removeWriterStat = (index) => {
        setWriterStats(writerStats.filter((_, i) => i !== index));
    };

    // Additional Stats handlers
    const addAdditionalStat = () => {
        setAdditionalStats([...additionalStats, { icon: 'star', label: '', number: '', description: '' }]);
    };

    const updateAdditionalStat = (index, field, value) => {
        const updated = [...additionalStats];
        updated[index][field] = value;
        setAdditionalStats(updated);
    };

    const removeAdditionalStat = (index) => {
        setAdditionalStats(additionalStats.filter((_, i) => i !== index));
    };

    // Expertise Distribution handlers
    const addExpertise = () => {
        setExpertiseDistribution([...expertiseDistribution, { color: 'blue', count: '', field: '' }]);
    };

    const updateExpertise = (index, field, value) => {
        const updated = [...expertiseDistribution];
        updated[index][field] = value;
        setExpertiseDistribution(updated);
    };

    const removeExpertise = (index) => {
        setExpertiseDistribution(expertiseDistribution.filter((_, i) => i !== index));
    };

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Header Tab */}
            {activeTab === 'header' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Statistics Section Header</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Our Track Record"
                        />
                    </div>

                    <div>
                        <Label htmlFor="subtitle">Subtitle (Rich Text)</Label>
                        <RichTextEditor
                            value={content.subtitle || ''}
                            onChange={(value) => setContent({...content, subtitle: value})}
                            height={200}
                        />
                        <p className="text-xs text-gray-500 mt-1">Brief description below the title</p>
                    </div>
                </div>
            )}

            {/* Main Stats Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Main Statistics</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">4-column grid of primary stats</p>
                        </div>
                        <button
                            type="button"
                            onClick={addMainStat}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Stat
                        </button>
                    </div>

                    {mainStats.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No main stats yet</p>
                            <button
                                type="button"
                                onClick={addMainStat}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Your First Stat
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {mainStats.map((stat, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Main Stat {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeMainStat(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                        <div>
                                            <Label>Icon</Label>
                                            <select
                                                value={stat.icon || 'book-open'}
                                                onChange={(e) => updateMainStat(index, 'icon', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {iconOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Color</Label>
                                            <select
                                                value={stat.color || 'blue'}
                                                onChange={(e) => updateMainStat(index, 'color', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {colorOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Number/Value</Label>
                                            <Input
                                                value={stat.number || ''}
                                                onChange={(e) => updateMainStat(index, 'number', e.target.value)}
                                                placeholder="e.g., 50,000+"
                                            />
                                        </div>

                                        <div>
                                            <Label>Label</Label>
                                            <Input
                                                value={stat.label || ''}
                                                onChange={(e) => updateMainStat(index, 'label', e.target.value)}
                                                placeholder="e.g., Happy Students"
                                            />
                                        </div>

                                        <div className="md:col-span-2">
                                            <Label>Description (Rich Text)</Label>
                                            <RichTextEditor
                                                value={stat.description || ''}
                                                onChange={(value) => updateMainStat(index, 'description', value)}
                                                height={150}
                                            />
                                        </div>

                                        <div className="md:col-span-2">
                                            <Label>Detail Text (Optional, Rich Text)</Label>
                                            <RichTextEditor
                                                value={stat.detail || ''}
                                                onChange={(value) => updateMainStat(index, 'detail', value)}
                                                height={100}
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Additional Stats Tab */}
            {activeTab === 'additional' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Additional Statistics</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">6-column grid of smaller stats</p>
                        </div>
                        <button
                            type="button"
                            onClick={addAdditionalStat}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Stat
                        </button>
                    </div>

                    {additionalStats.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No additional stats yet</p>
                            <button
                                type="button"
                                onClick={addAdditionalStat}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Additional Stat
                            </button>
                        </div>
                    ) : (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            {additionalStats.map((stat, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Additional Stat {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeAdditionalStat(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="space-y-3">
                                        <div>
                                            <Label>Icon</Label>
                                            <select
                                                value={stat.icon || 'star'}
                                                onChange={(e) => updateAdditionalStat(index, 'icon', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {iconOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Number</Label>
                                            <Input
                                                value={stat.number || ''}
                                                onChange={(e) => updateAdditionalStat(index, 'number', e.target.value)}
                                                placeholder="e.g., 98%"
                                            />
                                        </div>

                                        <div>
                                            <Label>Label</Label>
                                            <Input
                                                value={stat.label || ''}
                                                onChange={(e) => updateAdditionalStat(index, 'label', e.target.value)}
                                                placeholder="e.g., Success Rate"
                                            />
                                        </div>

                                        <div>
                                            <Label>Description (Rich Text)</Label>
                                            <RichTextEditor
                                                value={stat.description || ''}
                                                onChange={(value) => updateAdditionalStat(index, 'description', value)}
                                                height={100}
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Writer Stats Tab */}
            {activeTab === 'writers' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Writer Qualification Stats</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Progress bars showing writer qualifications</p>
                        </div>
                        <button
                            type="button"
                            onClick={addWriterStat}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Stat
                        </button>
                    </div>

                    {writerStats.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No writer stats yet</p>
                            <button
                                type="button"
                                onClick={addWriterStat}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Writer Stat
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {writerStats.map((stat, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Writer Stat {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeWriterStat(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                                        <div>
                                            <Label>Count</Label>
                                            <Input
                                                value={stat.count || ''}
                                                onChange={(e) => updateWriterStat(index, 'count', e.target.value)}
                                                placeholder="e.g., 500+"
                                            />
                                        </div>

                                        <div>
                                            <Label>Label</Label>
                                            <Input
                                                value={stat.label || ''}
                                                onChange={(e) => updateWriterStat(index, 'label', e.target.value)}
                                                placeholder="e.g., PhD Writers"
                                            />
                                        </div>

                                        <div>
                                            <Label>Percentage (0-100)</Label>
                                            <Input
                                                type="number"
                                                min="0"
                                                max="100"
                                                value={stat.percentage || '0'}
                                                onChange={(e) => updateWriterStat(index, 'percentage', e.target.value)}
                                                placeholder="e.g., 85"
                                            />
                                        </div>

                                        <div className="md:col-span-3">
                                            <Label>Description (Rich Text)</Label>
                                            <RichTextEditor
                                                value={stat.description || ''}
                                                onChange={(value) => updateWriterStat(index, 'description', value)}
                                                height={100}
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Expertise Distribution Tab */}
            {activeTab === 'expertise' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Writer Expertise Distribution</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">2-column grid of expertise fields</p>
                        </div>
                        <button
                            type="button"
                            onClick={addExpertise}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Field
                        </button>
                    </div>

                    {expertiseDistribution.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No expertise fields yet</p>
                            <button
                                type="button"
                                onClick={addExpertise}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Expertise Field
                            </button>
                        </div>
                    ) : (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            {expertiseDistribution.map((item, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Field {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeExpertise(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400 transition-colors"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="space-y-3">
                                        <div>
                                            <Label>Color</Label>
                                            <select
                                                value={item.color || 'blue'}
                                                onChange={(e) => updateExpertise(index, 'color', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {colorOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Count</Label>
                                            <Input
                                                value={item.count || ''}
                                                onChange={(e) => updateExpertise(index, 'count', e.target.value)}
                                                placeholder="e.g., 80"
                                            />
                                        </div>

                                        <div>
                                            <Label>Field Name</Label>
                                            <Input
                                                value={item.field || ''}
                                                onChange={(e) => updateExpertise(index, 'field', e.target.value)}
                                                placeholder="e.g., Business"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Statistics Section
                </button>
            </div>
        </div>
    );
}

function ContactBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [contactForm, setContactForm] = useState(content.contact_form || { title: '', fields: [], submit_text: '' });
    const [contactInfo, setContactInfo] = useState(content.contact_info || []);
    const [socialLinks, setSocialLinks] = useState(content.social_links || []);
    const [activeTab, setActiveTab] = useState('header');

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            contact_form: contactForm,
            contact_info: contactInfo,
            social_links: socialLinks,
        });
    };

    const tabs = [
        { id: 'header', label: 'Header', icon: '📝' },
        { id: 'form', label: 'Contact Form', icon: '📧' },
        { id: 'info', label: 'Contact Info', icon: '📞' },
        { id: 'social', label: 'Social Links', icon: '🔗' },
    ];

    const iconOptions = [
        { value: 'mail', label: '📧 Mail' },
        { value: 'phone', label: '📞 Phone' },
        { value: 'message-square', label: '💬 Message' },
        { value: 'linkedin', label: '🔗 LinkedIn' },
        { value: 'globe', label: '🌐 Globe' },
        { value: 'clock', label: '⏰ Clock' },
    ];

    const fieldTypeOptions = [
        { value: 'text', label: 'Text' },
        { value: 'email', label: 'Email' },
        { value: 'tel', label: 'Phone' },
        { value: 'textarea', label: 'Textarea' },
        { value: 'select', label: 'Dropdown' },
    ];

    // Contact Info management
    const addContactInfo = () => {
        setContactInfo([...contactInfo, { icon: 'mail', title: '', value: '', action: '', description: '' }]);
    };

    const updateContactInfo = (index, field, value) => {
        const updated = [...contactInfo];
        updated[index][field] = value;
        setContactInfo(updated);
    };

    const removeContactInfo = (index) => {
        setContactInfo(contactInfo.filter((_, i) => i !== index));
    };

    // Form Fields management
    const addFormField = () => {
        const updated = { ...contactForm };
        if (!updated.fields) updated.fields = [];
        updated.fields.push({ name: '', type: 'text', label: '', required: false, placeholder: '' });
        setContactForm(updated);
    };

    const updateFormField = (index, field, value) => {
        const updated = { ...contactForm };
        updated.fields[index][field] = value;
        setContactForm(updated);
    };

    const removeFormField = (index) => {
        const updated = { ...contactForm };
        updated.fields = updated.fields.filter((_, i) => i !== index);
        setContactForm(updated);
    };

    const moveFieldUp = (index) => {
        if (index === 0) return;
        const updated = { ...contactForm };
        [updated.fields[index - 1], updated.fields[index]] = [updated.fields[index], updated.fields[index - 1]];
        setContactForm(updated);
    };

    const moveFieldDown = (index) => {
        const updated = { ...contactForm };
        if (index === updated.fields.length - 1) return;
        [updated.fields[index], updated.fields[index + 1]] = [updated.fields[index + 1], updated.fields[index]];
        setContactForm(updated);
    };

    // Social Links management
    const addSocialLink = () => {
        setSocialLinks([...socialLinks, { url: '', icon: 'linkedin', platform: '' }]);
    };

    const updateSocialLink = (index, field, value) => {
        const updated = [...socialLinks];
        updated[index][field] = value;
        setSocialLinks(updated);
    };

    const removeSocialLink = (index) => {
        setSocialLinks(socialLinks.filter((_, i) => i !== index));
    };

    return (
        <div className="space-y-6">
            {/* Section Tabs */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Header Tab */}
            {activeTab === 'header' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Contact Section Header</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Get in Touch"
                        />
                    </div>

                    <div>
                        <Label htmlFor="subtitle">Subtitle</Label>
                        <RichTextEditor
                            value={content.subtitle || ''}
                            onChange={(value) => setContent({...content, subtitle: value})}
                            height={150}
                        />
                    </div>
                </div>
            )}

            {/* Contact Form Tab */}
            {activeTab === 'form' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Contact Form Configuration</h3>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label>Form Title</Label>
                            <Input
                                value={contactForm.title || ''}
                                onChange={(e) => setContactForm({...contactForm, title: e.target.value})}
                                placeholder="e.g., Send Us a Message"
                            />
                        </div>

                        <div>
                            <Label>Submit Button Text</Label>
                            <Input
                                value={contactForm.submit_text || ''}
                                onChange={(e) => setContactForm({...contactForm, submit_text: e.target.value})}
                                placeholder="e.g., Send Message"
                            />
                        </div>
                    </div>

                    <div className="border-t border-gray-200 dark:border-gray-700 pt-4 mt-6">
                        <div className="flex justify-between items-center mb-4">
                            <h4 className="font-semibold text-gray-900 dark:text-white">
                                Form Fields ({(contactForm.fields || []).length})
                            </h4>
                            <button
                                type="button"
                                onClick={addFormField}
                                className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Field
                            </button>
                        </div>

                        {(!contactForm.fields || contactForm.fields.length === 0) ? (
                            <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                                <p className="text-gray-500 dark:text-gray-400 mb-3">No form fields yet</p>
                                <button
                                    type="button"
                                    onClick={addFormField}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                                >
                                    <Plus className="h-4 w-4 mr-2" />
                                    Add First Field
                                </button>
                            </div>
                        ) : (
                            <div className="space-y-4">
                                {contactForm.fields.map((field, index) => (
                                    <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                        <div className="flex justify-between items-center mb-3">
                                            <span className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                                                Field #{index + 1}
                                            </span>
                                            <div className="flex gap-2">
                                                <button
                                                    type="button"
                                                    onClick={() => moveFieldUp(index)}
                                                    disabled={index === 0}
                                                    className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                                    title="Move up"
                                                >
                                                    <ChevronUp className="h-4 w-4" />
                                                </button>
                                                <button
                                                    type="button"
                                                    onClick={() => moveFieldDown(index)}
                                                    disabled={index === contactForm.fields.length - 1}
                                                    className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                                    title="Move down"
                                                >
                                                    <ChevronDown className="h-4 w-4" />
                                                </button>
                                                <button
                                                    type="button"
                                                    onClick={() => removeFormField(index)}
                                                    className="p-1 text-red-500 hover:text-red-700"
                                                    title="Remove field"
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </button>
                                            </div>
                                        </div>

                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                            <div>
                                                <Label>Field Name (e.g., name, email)</Label>
                                                <Input
                                                    value={field.name || ''}
                                                    onChange={(e) => updateFormField(index, 'name', e.target.value)}
                                                    placeholder="name"
                                                />
                                            </div>

                                            <div>
                                                <Label>Field Type</Label>
                                                <select
                                                    value={field.type || 'text'}
                                                    onChange={(e) => updateFormField(index, 'type', e.target.value)}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                                >
                                                    {fieldTypeOptions.map((opt) => (
                                                        <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                    ))}
                                                </select>
                                            </div>

                                            <div>
                                                <Label>Label (shown to user)</Label>
                                                <Input
                                                    value={field.label || ''}
                                                    onChange={(e) => updateFormField(index, 'label', e.target.value)}
                                                    placeholder="e.g., Full Name"
                                                />
                                            </div>

                                            <div>
                                                <Label>Placeholder</Label>
                                                <Input
                                                    value={field.placeholder || ''}
                                                    onChange={(e) => updateFormField(index, 'placeholder', e.target.value)}
                                                    placeholder="e.g., Your name"
                                                />
                                            </div>

                                            <div className="flex items-center md:col-span-2">
                                                <input
                                                    type="checkbox"
                                                    checked={field.required || false}
                                                    onChange={(e) => updateFormField(index, 'required', e.target.checked)}
                                                    className="mr-2"
                                                />
                                                <Label>Required Field</Label>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                </div>
            )}

            {/* Contact Info Tab */}
            {activeTab === 'info' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Contact Information</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Email, phone, address details</p>
                        </div>
                        <button
                            type="button"
                            onClick={addContactInfo}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Info
                        </button>
                    </div>

                    {contactInfo.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No contact info yet</p>
                            <button
                                type="button"
                                onClick={addContactInfo}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add Contact Info
                            </button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {contactInfo.map((info, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Info Item {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeContactInfo(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                        <div>
                                            <Label>Icon</Label>
                                            <select
                                                value={info.icon || 'mail'}
                                                onChange={(e) => updateContactInfo(index, 'icon', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {iconOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>Title</Label>
                                            <Input
                                                value={info.title || ''}
                                                onChange={(e) => updateContactInfo(index, 'title', e.target.value)}
                                                placeholder="e.g., Email Us"
                                            />
                                        </div>

                                        <div>
                                            <Label>Value/Content</Label>
                                            <Input
                                                value={info.value || ''}
                                                onChange={(e) => updateContactInfo(index, 'value', e.target.value)}
                                                placeholder="e.g., support@academicscribe.com"
                                            />
                                        </div>

                                        <div>
                                            <Label>Action Link (optional)</Label>
                                            <Input
                                                value={info.action || ''}
                                                onChange={(e) => updateContactInfo(index, 'action', e.target.value)}
                                                placeholder="e.g., mailto:support@academicscribe.com"
                                            />
                                        </div>

                                        <div className="md:col-span-2">
                                            <Label>Description (optional)</Label>
                                            <textarea
                                                value={info.description || ''}
                                                onChange={(e) => updateContactInfo(index, 'description', e.target.value)}
                                                rows={2}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                                placeholder="e.g., Available 24/7"
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Social Links Tab */}
            {activeTab === 'social' && (
                <div className="space-y-4">
                    <div className="flex justify-between items-center">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Social Media Links</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Connect with us on social platforms</p>
                        </div>
                        <button
                            type="button"
                            onClick={addSocialLink}
                            className="inline-flex items-center px-3 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Link
                        </button>
                    </div>

                    {socialLinks.length === 0 ? (
                        <div className="text-center py-8 bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600">
                            <p className="text-gray-500 dark:text-gray-400 mb-3">No social links yet</p>
                            <button
                                type="button"
                                onClick={addSocialLink}
                                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 transition-colors"
                            >
                                <Plus className="h-4 w-4 mr-2" />
                                Add First Link
                            </button>
                        </div>
                    ) : (
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            {socialLinks.map((link, index) => (
                                <div key={index} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 bg-white dark:bg-gray-800">
                                    <div className="flex justify-between items-center mb-3">
                                        <h4 className="font-medium text-gray-900 dark:text-white">Link {index + 1}</h4>
                                        <button
                                            type="button"
                                            onClick={() => removeSocialLink(index)}
                                            className="text-red-600 hover:text-red-700 dark:text-red-400"
                                        >
                                            <Trash2 className="h-4 w-4" />
                                        </button>
                                    </div>

                                    <div className="space-y-3">
                                        <div>
                                            <Label>Platform</Label>
                                            <Input
                                                value={link.platform || ''}
                                                onChange={(e) => updateSocialLink(index, 'platform', e.target.value)}
                                                placeholder="e.g., LinkedIn"
                                            />
                                        </div>

                                        <div>
                                            <Label>Icon</Label>
                                            <select
                                                value={link.icon || 'linkedin'}
                                                onChange={(e) => updateSocialLink(index, 'icon', e.target.value)}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500"
                                            >
                                                {iconOptions.map((opt) => (
                                                    <option key={opt.value} value={opt.value}>{opt.label}</option>
                                                ))}
                                            </select>
                                        </div>

                                        <div>
                                            <Label>URL</Label>
                                            <Input
                                                value={link.url || ''}
                                                onChange={(e) => updateSocialLink(index, 'url', e.target.value)}
                                                placeholder="https://linkedin.com/..."
                                            />
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button (Fixed at Bottom) */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 p-4 -mx-4 -mb-4">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Contact Section
                </button>
            </div>
        </div>
    );
}

function GenericBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [viewMode, setViewMode] = useState('form'); // 'form' or 'json'

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    return (
        <div className="space-y-4">
            <div className="flex space-x-2 mb-4">
                <button
                    type="button"
                    onClick={() => setViewMode('form')}
                    className={`px-3 py-1 rounded ${viewMode === 'form' ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-700'}`}
                >
                    Form View
                </button>
                <button
                    type="button"
                    onClick={() => setViewMode('json')}
                    className={`px-3 py-1 rounded ${viewMode === 'json' ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-700'}`}
                >
                    JSON View
                </button>
            </div>

            {viewMode === 'json' ? (
                <div>
                    <Label>Custom Content (JSON)</Label>
                    <Textarea
                        value={JSON.stringify(content, null, 2)}
                        onChange={(e) => {
                            try {
                                const parsed = JSON.parse(e.target.value);
                                setContent(parsed);
                            } catch (error) {
                                // Invalid JSON, don't update
                            }
                        }}
                        rows={15}
                        className="font-mono text-sm"
                    />
                </div>
            ) : (
                <div className="space-y-4">
                    {Object.keys(content).map((key) => {
                        // Use TinyMCE for text-related fields or long strings
                        const useRichText = typeof content[key] === 'string' && 
                            (content[key].length > 50 || 
                             ['description', 'content', 'text', 'message', 'subtitle', 'summary', 'details'].includes(key.toLowerCase()));
                        
                        return (
                            <div key={key}>
                                <Label htmlFor={key} className="capitalize">{key.replace(/_/g, ' ')}</Label>
                                {useRichText ? (
                                    <RichTextEditor
                                        value={content[key]}
                                        onChange={(value) => setContent({...content, [key]: value})}
                                        height={250}
                                    />
                                ) : typeof content[key] === 'object' ? (
                                    <Textarea
                                        id={key}
                                        value={JSON.stringify(content[key], null, 2)}
                                        onChange={(e) => {
                                            try {
                                                const parsed = JSON.parse(e.target.value);
                                                setContent({...content, [key]: parsed});
                                            } catch (error) {
                                                // Invalid JSON
                                            }
                                        }}
                                        rows={5}
                                        className="font-mono text-sm"
                                    />
                                ) : (
                                    <Input
                                        id={key}
                                        value={content[key] || ''}
                                        onChange={(e) => setContent({...content, [key]: e.target.value})}
                                    />
                                )}
                            </div>
                        );
                    })}
                </div>
            )}
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

// Service Block Editors with TinyMCE Support

function ServiceHeroBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [keyPoints, setKeyPoints] = useState(content.keyPoints || []);
    const [trustSignals, setTrustSignals] = useState(content.trustSignals || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            keyPoints,
            trustSignals
        });
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'keyPoints', label: 'Key Points', icon: '✓' },
        { id: 'cta', label: 'Call to Action', icon: '🔘' },
        { id: 'trust', label: 'Trust Signals', icon: '⭐' },
    ];

    // Key Points management
    const addKeyPoint = () => {
        setKeyPoints([...keyPoints, '']);
    };

    const updateKeyPoint = (index, value) => {
        const updated = [...keyPoints];
        updated[index] = value;
        setKeyPoints(updated);
    };

    const removeKeyPoint = (index) => {
        setKeyPoints(keyPoints.filter((_, i) => i !== index));
    };

    const moveKeyPointUp = (index) => {
        if (index === 0) return;
        const updated = [...keyPoints];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setKeyPoints(updated);
    };

    const moveKeyPointDown = (index) => {
        if (index === keyPoints.length - 1) return;
        const updated = [...keyPoints];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setKeyPoints(updated);
    };

    // Trust Signals management
    const addTrustSignal = () => {
        setTrustSignals([...trustSignals, '']);
    };

    const updateTrustSignal = (index, value) => {
        const updated = [...trustSignals];
        updated[index] = value;
        setTrustSignals(updated);
    };

    const removeTrustSignal = (index) => {
        setTrustSignals(trustSignals.filter((_, i) => i !== index));
    };

    const moveTrustSignalUp = (index) => {
        if (index === 0) return;
        const updated = [...trustSignals];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setTrustSignals(updated);
    };

    const moveTrustSignalDown = (index) => {
        if (index === trustSignals.length - 1) return;
        const updated = [...trustSignals];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setTrustSignals(updated);
    };

    return (
        <div className="space-y-6">
            {/* Tab Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Hero Content</h3>
                    
                    <div>
                        <Label htmlFor="title">Service Title *</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Professional Essay Writing Service | Expert Custom Essay Help"
                            className="text-base"
                        />
                        <p className="text-xs text-gray-500 mt-1">This appears as the main headline on the service page</p>
                    </div>

                    <div>
                        <Label htmlFor="description">Service Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={250}
                            placeholder="Enter compelling service description with rich text formatting..."
                        />
                        <p className="text-xs text-gray-500 mt-1">Use bold, italic, links, and formatting to create engaging content</p>
                    </div>
                </div>
            )}

            {/* Key Points Tab */}
            {activeTab === 'keyPoints' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Key Points</h3>
                        <Button
                            type="button"
                            onClick={addKeyPoint}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Key Point
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Bullet points highlighting key service features (displayed with checkmarks)
                    </p>

                    {keyPoints.length === 0 ? (
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <p className="text-gray-500 dark:text-gray-400 mb-4">No key points yet</p>
                            <Button
                                type="button"
                                onClick={addKeyPoint}
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                            >
                                Add Your First Key Point
                            </Button>
                        </div>
                    ) : (
                        <div className="space-y-3">
                            {keyPoints.map((point, index) => (
                                <div key={index} className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-start gap-3">
                                        <span className="flex-shrink-0 w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center font-semibold text-sm">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-2">
                                            <Textarea
                                                value={point}
                                                onChange={(e) => updateKeyPoint(index, e.target.value)}
                                                placeholder={`Key point ${index + 1}...`}
                                                rows={2}
                                                className="w-full"
                                            />
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveKeyPointUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveKeyPointDown(index)}
                                                disabled={index === keyPoints.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeKeyPoint(index)}
                                                className="p-1 text-red-500 hover:text-red-700"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* CTA Tab */}
            {activeTab === 'cta' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Call to Action Button</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Primary CTA button configuration (secondary "Place Order" button is automatic)
                    </p>

                    <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                        <div className="space-y-4">
                            <div>
                                <Label htmlFor="ctaText">Primary Button Text</Label>
                                <Input
                                    id="ctaText"
                                    value={content.ctaText || ''}
                                    onChange={(e) => setContent({...content, ctaText: e.target.value})}
                                    placeholder="e.g., Get Professional Essay Help Now"
                                />
                            </div>

                            <div>
                                <Label htmlFor="ctaLink">Primary Button Link</Label>
                                <Input
                                    id="ctaLink"
                                    value={content.ctaLink || ''}
                                    onChange={(e) => setContent({...content, ctaLink: e.target.value})}
                                    placeholder="/contact"
                                />
                                <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                                    Common links: /contact, /place-order, /pricing
                                </p>
                            </div>

                            {/* Button Preview */}
                            <div className="pt-4 border-t border-blue-200 dark:border-blue-800">
                                <Label>Button Preview:</Label>
                                <div className="mt-2 flex gap-3">
                                    <div className="bg-white text-blue-800 px-6 py-3 rounded-md font-medium shadow-md">
                                        {content.ctaText || 'Get Started'}
                                    </div>
                                    <div className="border-2 border-white text-gray-700 px-6 py-3 rounded-md font-medium">
                                        Place Order
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Trust Signals Tab */}
            {activeTab === 'trust' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Trust Signals</h3>
                        <Button
                            type="button"
                            onClick={addTrustSignal}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Trust Signal
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Trust badges displayed above the hero headline (e.g., ratings, customer count, support availability)
                    </p>

                    {trustSignals.length === 0 ? (
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <p className="text-gray-500 dark:text-gray-400 mb-4">No trust signals yet</p>
                            <Button
                                type="button"
                                onClick={addTrustSignal}
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                            >
                                Add Your First Trust Signal
                            </Button>
                        </div>
                    ) : (
                        <div className="space-y-3">
                            {trustSignals.map((signal, index) => (
                                <div key={index} className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center gap-3">
                                        <span className="flex-shrink-0 w-8 h-8 bg-yellow-500 text-white rounded-full flex items-center justify-center font-semibold text-sm">
                                            {index + 1}
                                        </span>
                                        <Input
                                            value={signal}
                                            onChange={(e) => updateTrustSignal(index, e.target.value)}
                                            placeholder={`e.g., ${index === 0 ? '10,000+ satisfied students' : index === 1 ? '4.8/5 average rating' : '24/7 customer support'}`}
                                            className="flex-1"
                                        />
                                        <div className="flex gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move left"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveTrustSignalDown(index)}
                                                disabled={index === trustSignals.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move right"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeTrustSignal(index)}
                                                className="p-1 text-red-500 hover:text-red-700"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button - Sticky */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 pt-4 mt-6">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Service Hero
                </button>
            </div>
        </div>
    );
}

function ServiceOverviewBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [features, setFeatures] = useState(content.features || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            features
        });
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'features', label: 'Features', icon: '✨' },
    ];

    const iconOptions = [
        { value: 'Award', label: '🏆 Award' },
        { value: 'BookOpen', label: '📖 Book' },
        { value: 'Search', label: '🔍 Search' },
        { value: 'FileText', label: '📄 File' },
        { value: 'Users', label: '👥 Users' },
        { value: 'Target', label: '🎯 Target' },
        { value: 'Shield', label: '🛡️ Shield' },
        { value: 'Star', label: '⭐ Star' },
        { value: 'CheckCircle', label: '✓ Check' },
        { value: 'Zap', label: '⚡ Zap' },
        { value: 'Heart', label: '❤️ Heart' },
        { value: 'Clock', label: '⏰ Clock' },
    ];

    // Features management
    const addFeature = () => {
        setFeatures([...features, { icon: 'Award', title: '', description: '' }]);
    };

    const updateFeature = (index, field, value) => {
        const updated = [...features];
        updated[index][field] = value;
        setFeatures(updated);
    };

    const removeFeature = (index) => {
        setFeatures(features.filter((_, i) => i !== index));
    };

    const moveFeatureUp = (index) => {
        if (index === 0) return;
        const updated = [...features];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setFeatures(updated);
    };

    const moveFeatureDown = (index) => {
        if (index === features.length - 1) return;
        const updated = [...features];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setFeatures(updated);
    };

    return (
        <div className="space-y-6">
            {/* Tab Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Overview Section</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Our Comprehensive Essay Writing Service"
                        />
                    </div>

                    <div>
                        <Label htmlFor="description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={200}
                            placeholder="Enter overview description with rich text formatting..."
                        />
                        <p className="text-xs text-gray-500 mt-1">Brief introduction to the service features</p>
                    </div>
                </div>
            )}

            {/* Features Tab */}
            {activeTab === 'features' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Features</h3>
                        <Button
                            type="button"
                            onClick={addFeature}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Feature
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Key features displayed in a grid format with icons
                    </p>

                    {features.length === 0 ? (
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <p className="text-gray-500 dark:text-gray-400 mb-4">No features yet</p>
                            <Button
                                type="button"
                                onClick={addFeature}
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                            >
                                Add Your First Feature
                            </Button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {features.map((feature, index) => (
                                <div key={index} className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-start gap-3">
                                        <span className="flex-shrink-0 w-8 h-8 bg-blue-600 text-white rounded-full flex items-center justify-center font-semibold text-sm">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div>
                                                <Label>Icon</Label>
                                                <select
                                                    value={feature.icon || 'Award'}
                                                    onChange={(e) => updateFeature(index, 'icon', e.target.value)}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                >
                                                    {iconOptions.map((opt) => (
                                                        <option key={opt.value} value={opt.value}>
                                                            {opt.label}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                            <div>
                                                <Label>Feature Title</Label>
                                                <Input
                                                    value={feature.title || ''}
                                                    onChange={(e) => updateFeature(index, 'title', e.target.value)}
                                                    placeholder="e.g., Professional Essay Writers"
                                                />
                                            </div>
                                            <div>
                                                <Label>Feature Description</Label>
                                                <Textarea
                                                    value={feature.description || ''}
                                                    onChange={(e) => updateFeature(index, 'description', e.target.value)}
                                                    placeholder="Describe this feature..."
                                                    rows={3}
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveFeatureUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveFeatureDown(index)}
                                                disabled={index === features.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeFeature(index)}
                                                className="p-1 text-red-500 hover:text-red-700"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button - Sticky */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 pt-4 mt-6">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Service Overview
                </button>
            </div>
        </div>
    );
}

function ServiceProcessBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [steps, setSteps] = useState(content.steps || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        // Auto-number steps before saving
        const numberedSteps = steps.map((step, index) => ({
            ...step,
            number: index + 1
        }));
        onUpdate(block.id, {
            ...content,
            steps: numberedSteps
        });
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'steps', label: 'Process Steps', icon: '🔢' },
    ];

    const iconOptions = [
        { value: 'Target', label: '🎯 Target' },
        { value: 'Search', label: '🔍 Search' },
        { value: 'Edit', label: '✏️ Edit' },
        { value: 'FileText', label: '📄 File' },
        { value: 'BookOpen', label: '📖 Book' },
        { value: 'CheckCircle', label: '✓ Check' },
        { value: 'Award', label: '🏆 Award' },
        { value: 'Users', label: '👥 Users' },
        { value: 'Shield', label: '🛡️ Shield' },
        { value: 'Star', label: '⭐ Star' },
        { value: 'Zap', label: '⚡ Zap' },
        { value: 'Clock', label: '⏰ Clock' },
    ];

    // Steps management
    const addStep = () => {
        setSteps([...steps, { number: steps.length + 1, icon: 'Target', title: '', description: '' }]);
    };

    const updateStep = (index, field, value) => {
        const updated = [...steps];
        updated[index][field] = value;
        setSteps(updated);
    };

    const removeStep = (index) => {
        setSteps(steps.filter((_, i) => i !== index));
    };

    const moveStepUp = (index) => {
        if (index === 0) return;
        const updated = [...steps];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setSteps(updated);
    };

    const moveStepDown = (index) => {
        if (index === steps.length - 1) return;
        const updated = [...steps];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setSteps(updated);
    };

    return (
        <div className="space-y-6">
            {/* Tab Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Process Section</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Our Professional Essay Writing Process"
                        />
                    </div>

                    <div>
                        <Label htmlFor="description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={150}
                            placeholder="Enter process description with rich text formatting..."
                        />
                        <p className="text-xs text-gray-500 mt-1">Brief introduction to the process steps</p>
                    </div>
                </div>
            )}

            {/* Steps Tab */}
            {activeTab === 'steps' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Process Steps</h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400">Steps are automatically numbered (1, 2, 3...)</p>
                        </div>
                        <Button
                            type="button"
                            onClick={addStep}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Step
                        </Button>
                    </div>

                    {steps.length === 0 ? (
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <p className="text-gray-500 dark:text-gray-400 mb-4">No process steps yet</p>
                            <Button
                                type="button"
                                onClick={addStep}
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                            >
                                Add Your First Step
                            </Button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {steps.map((step, index) => (
                                <div key={index} className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-start gap-3">
                                        <div className="flex-shrink-0">
                                            <div className="w-10 h-10 bg-gradient-to-br from-blue-600 to-cyan-500 text-white rounded-full flex items-center justify-center font-bold text-lg shadow-lg">
                                                {index + 1}
                                            </div>
                                            <p className="text-xs text-center mt-1 text-gray-500">Step</p>
                                        </div>
                                        <div className="flex-1 space-y-3">
                                            <div>
                                                <Label>Icon</Label>
                                                <select
                                                    value={step.icon || 'Target'}
                                                    onChange={(e) => updateStep(index, 'icon', e.target.value)}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                >
                                                    {iconOptions.map((opt) => (
                                                        <option key={opt.value} value={opt.value}>
                                                            {opt.label}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                            <div>
                                                <Label>Step Title</Label>
                                                <Input
                                                    value={step.title || ''}
                                                    onChange={(e) => updateStep(index, 'title', e.target.value)}
                                                    placeholder={`e.g., Step ${index + 1}: Topic Analysis & Planning`}
                                                />
                                            </div>
                                            <div>
                                                <Label>Step Description</Label>
                                                <Textarea
                                                    value={step.description || ''}
                                                    onChange={(e) => updateStep(index, 'description', e.target.value)}
                                                    placeholder="Describe what happens in this step..."
                                                    rows={3}
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveStepUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveStepDown(index)}
                                                disabled={index === steps.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeStep(index)}
                                                className="p-1 text-red-500 hover:text-red-700"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* Save Button - Sticky */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 pt-4 mt-6">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Service Process
                </button>
            </div>
        </div>
    );
}

function ServiceBenefitsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [benefits, setBenefits] = useState(content.benefits || []);
    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        onUpdate(block.id, {
            ...content,
            benefits
        });
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'benefits', label: 'Benefits', icon: '✨' },
        { id: 'cta', label: 'Call to Action', icon: '🔘' },
    ];

    const iconOptions = [
        { value: 'Award', label: '🏆 Award' },
        { value: 'Shield', label: '🛡️ Shield' },
        { value: 'BookOpen', label: '📖 Book' },
        { value: 'CheckCircle', label: '✓ Check' },
        { value: 'Star', label: '⭐ Star' },
        { value: 'Target', label: '🎯 Target' },
        { value: 'Users', label: '👥 Users' },
        { value: 'Clock', label: '⏰ Clock' },
        { value: 'Zap', label: '⚡ Zap' },
        { value: 'Heart', label: '❤️ Heart' },
        { value: 'FileText', label: '📄 File' },
        { value: 'Search', label: '🔍 Search' },
    ];

    // Benefits management
    const addBenefit = () => {
        setBenefits([...benefits, { icon: 'Award', title: '', description: '' }]);
    };

    const updateBenefit = (index, field, value) => {
        const updated = [...benefits];
        updated[index][field] = value;
        setBenefits(updated);
    };

    const removeBenefit = (index) => {
        setBenefits(benefits.filter((_, i) => i !== index));
    };

    const moveBenefitUp = (index) => {
        if (index === 0) return;
        const updated = [...benefits];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setBenefits(updated);
    };

    const moveBenefitDown = (index) => {
        if (index === benefits.length - 1) return;
        const updated = [...benefits];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setBenefits(updated);
    };

    return (
        <div className="space-y-6">
            {/* Tab Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Main Content Tab */}
            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Benefits Section</h3>
                    
                    <div>
                        <Label htmlFor="title">Section Title</Label>
                        <Input
                            id="title"
                            value={content.title || ''}
                            onChange={(e) => setContent({...content, title: e.target.value})}
                            placeholder="e.g., Why Choose Our Professional Essay Writing Service"
                        />
                    </div>

                    <div>
                        <Label htmlFor="description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description || ''}
                            onChange={(value) => setContent({...content, description: value})}
                            height={150}
                            placeholder="Enter benefits description with rich text formatting..."
                        />
                        <p className="text-xs text-gray-500 mt-1">Brief introduction to service benefits</p>
                    </div>
                </div>
            )}

            {/* Benefits Tab */}
            {activeTab === 'benefits' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Service Benefits</h3>
                        <Button
                            type="button"
                            onClick={addBenefit}
                            className="bg-green-600 hover:bg-green-700 text-white"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Benefit
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Key benefits displayed with icons and descriptions
                    </p>

                    {benefits.length === 0 ? (
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <p className="text-gray-500 dark:text-gray-400 mb-4">No benefits yet</p>
                            <Button
                                type="button"
                                onClick={addBenefit}
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                            >
                                Add Your First Benefit
                            </Button>
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {benefits.map((benefit, index) => (
                                <div key={index} className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div className="flex items-start gap-3">
                                        <span className="flex-shrink-0 w-8 h-8 bg-green-600 text-white rounded-full flex items-center justify-center font-semibold text-sm">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div>
                                                <Label>Icon</Label>
                                                <select
                                                    value={benefit.icon || 'Award'}
                                                    onChange={(e) => updateBenefit(index, 'icon', e.target.value)}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                                                >
                                                    {iconOptions.map((opt) => (
                                                        <option key={opt.value} value={opt.value}>
                                                            {opt.label}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                            <div>
                                                <Label>Benefit Title</Label>
                                                <Input
                                                    value={benefit.title || ''}
                                                    onChange={(e) => updateBenefit(index, 'title', e.target.value)}
                                                    placeholder="e.g., Subject-Specific Essay Writers"
                                                />
                                            </div>
                                            <div>
                                                <Label>Benefit Description</Label>
                                                <Textarea
                                                    value={benefit.description || ''}
                                                    onChange={(e) => updateBenefit(index, 'description', e.target.value)}
                                                    placeholder="Describe this benefit..."
                                                    rows={3}
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveBenefitUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveBenefitDown(index)}
                                                disabled={index === benefits.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => removeBenefit(index)}
                                                className="p-1 text-red-500 hover:text-red-700"
                                                title="Remove"
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            {/* CTA Tab */}
            {activeTab === 'cta' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Call to Action</h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                        Optional CTA button at the end of benefits section
                    </p>

                    <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 space-y-4">
                        <div>
                            <Label htmlFor="ctaText">CTA Button Text</Label>
                            <Input
                                id="ctaText"
                                value={content.ctaText || ''}
                                onChange={(e) => setContent({...content, ctaText: e.target.value})}
                                placeholder="e.g., Order Professional Essay Now"
                            />
                        </div>
                        <div>
                            <Label htmlFor="ctaLink">CTA Button Link</Label>
                            <Input
                                id="ctaLink"
                                value={content.ctaLink || ''}
                                onChange={(e) => setContent({...content, ctaLink: e.target.value})}
                                placeholder="/place-order"
                            />
                        </div>
                    </div>
                </div>
            )}

            {/* Save Button - Sticky */}
            <div className="sticky bottom-0 bg-white dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 pt-4 mt-6">
                <button
                    type="button"
                    onClick={handleSave}
                    className="w-full inline-flex items-center justify-center px-6 py-3 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors shadow-lg"
                >
                    <Save className="h-5 w-5 mr-2" />
                    Save Service Benefits
                </button>
            </div>
        </div>
    );
}

function ServiceTestimonialsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div>
                <Label>Testimonials (JSON Array)</Label>
                <Textarea
                    value={JSON.stringify(content.testimonials || [], null, 2)}
                    onChange={(e) => {
                        try {
                            const parsed = JSON.parse(e.target.value);
                            setContent({...content, testimonials: parsed});
                        } catch (error) {
                            // Invalid JSON
                        }
                    }}
                    rows={12}
                    className="font-mono text-sm"
                    placeholder='[{"text": "Quote...", "author": "Name", "role": "Position", "rating": 5}]'
                />
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function ServiceFAQsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [faqs, setFaqs] = useState(content.faqs || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, faqs });
    };

    const addFaq = () => {
        setFaqs([...faqs, { question: '', answer: '' }]);
    };

    const removeFaq = (index) => {
        setFaqs(faqs.filter((_, i) => i !== index));
    };

    const updateFaq = (index, field, value) => {
        const updated = [...faqs];
        updated[index] = { ...updated[index], [field]: value };
        setFaqs(updated);
    };

    const moveFaqUp = (index) => {
        if (index === 0) return;
        const updated = [...faqs];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setFaqs(updated);
    };

    const moveFaqDown = (index) => {
        if (index === faqs.length - 1) return;
        const updated = [...faqs];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setFaqs(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>FAQs ({faqs.length})</Label>
                    <button
                        type="button"
                        onClick={addFaq}
                        className="inline-flex items-center px-3 py-1.5 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add FAQ
                    </button>
                </div>

                <div className="space-y-6">
                    {faqs.map((faq, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">FAQ #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveFaqUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveFaqDown(index)}
                                        disabled={index === faqs.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeFaq(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove FAQ"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div>
                                    <Label htmlFor={`question-${index}`}>Question</Label>
                                    <Input
                                        id={`question-${index}`}
                                        value={faq.question || ''}
                                        onChange={(e) => updateFaq(index, 'question', e.target.value)}
                                        placeholder="Enter the FAQ question..."
                                    />
                                </div>

                                <div>
                                    <Label htmlFor={`answer-${index}`}>Answer (with TinyMCE)</Label>
                                    <RichTextEditor
                                        value={faq.answer || ''}
                                        onChange={(value) => updateFaq(index, 'answer', value)}
                                        height={200}
                                    />
                                </div>
                            </div>
                        </div>
                    ))}

                    {faqs.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No FAQs yet. Click "Add FAQ" to get started.
                        </div>
                    )}
                </div>
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}

function ServiceCTABlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [saving, setSaving] = useState(false);

    const handleSave = () => {
        console.log('ServiceCTABlockEditor: Saving block', block.id, content);
        setSaving(true);
        onUpdate(block.id, content);
        setTimeout(() => {
            setSaving(false);
        }, 500);
    };

    return (
        <div className="space-y-4">
            <div>
                <Label htmlFor="title">Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="grid grid-cols-2 gap-4">
                <div>
                    <Label htmlFor="primaryButtonText">Primary Button Text</Label>
                    <Input
                        id="primaryButtonText"
                        value={content.primaryButtonText || ''}
                        onChange={(e) => setContent({...content, primaryButtonText: e.target.value})}
                    />
                </div>
                <div>
                    <Label htmlFor="primaryButtonLink">Primary Button Link</Label>
                    <Input
                        id="primaryButtonLink"
                        value={content.primaryButtonLink || ''}
                        onChange={(e) => setContent({...content, primaryButtonLink: e.target.value})}
                    />
                </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
                <div>
                    <Label htmlFor="secondaryButtonText">Secondary Button Text</Label>
                    <Input
                        id="secondaryButtonText"
                        value={content.secondaryButtonText || ''}
                        onChange={(e) => setContent({...content, secondaryButtonText: e.target.value})}
                    />
                </div>
                <div>
                    <Label htmlFor="secondaryButtonLink">Secondary Button Link</Label>
                    <Input
                        id="secondaryButtonLink"
                        value={content.secondaryButtonLink || ''}
                        onChange={(e) => setContent({...content, secondaryButtonLink: e.target.value})}
                    />
                </div>
            </div>

            <div>
                <Label htmlFor="bgColor">Background Color</Label>
                <Input
                    id="bgColor"
                    value={content.bgColor || ''}
                    onChange={(e) => setContent({...content, bgColor: e.target.value})}
                    placeholder="blue, gray, etc."
                />
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                disabled={saving}
                className={`inline-flex items-center px-4 py-2 ${saving ? 'bg-gray-400' : 'bg-blue-600'} text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors ${saving ? 'cursor-not-allowed' : ''}`}
            >
                <Save className="h-4 w-4 mr-2" />
                {saving ? 'Saving...' : 'Save Changes'}
            </button>
        </div>
    );
}

function InternalLinksBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [links, setLinks] = useState(() => {
        if (!Array.isArray(content.links)) {
            return [];
        }
        return content.links.map((link) => ({
            title: link?.title || '',
            url: link?.url || '',
            description: link?.description_html || link?.description || '',
            is_active: link?.is_active !== undefined ? link.is_active : true,
        }));
    });

    const handleSave = () => {
        const payload = {
            ...content,
            links: links.map((link) => ({
                title: link.title || '',
                url: link.url || '',
                description: link.description || '',
                description_html: link.description || '',
                is_active: link.is_active !== undefined ? link.is_active : true,
            })),
        };
        onUpdate(block.id, payload);
    };

    const addLink = () => {
        setLinks([
            ...links,
            { title: '', url: '', description: '', is_active: true },
        ]);
    };

    const updateLink = (index, partial) => {
        setLinks((current) => {
            const clone = [...current];
            clone[index] = {
                ...clone[index],
                ...partial,
            };
            return clone;
        });
    };

    const removeLink = (index) => {
        setLinks((current) => current.filter((_, itemIndex) => itemIndex !== index));
    };

    const moveLinkUp = (index) => {
        if (index === 0) return;
        setLinks((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveLinkDown = (index) => {
        if (index === links.length - 1) return;
        setLinks((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="space-y-4">
                <div>
                    <Label htmlFor="internal-links-title">Section Title</Label>
                    <Input
                        id="internal-links-title"
                        value={content.title || ''}
                        onChange={(e) => setContent({...content, title: e.target.value})}
                        placeholder="e.g., Related Services"
                    />
                </div>

                <div>
                    <Label htmlFor="internal-links-description">Description (Rich Text)</Label>
                    <RichTextEditor
                        value={content.description || ''}
                        onChange={(value) => setContent({...content, description: value})}
                        height={160}
                        placeholder="Optional supporting copy for the links section."
                    />
                </div>
            </div>

            <div className="border-t border-gray-200 pt-4 dark:border-gray-700">
                <div className="mb-4 flex items-center justify-between">
                    <div>
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Internal Links</h3>
                        <p className="text-sm text-gray-600 dark:text-gray-400">
                            Manage internal links to other pages or sections.
                        </p>
                    </div>
                    <Button type="button" onClick={addLink} className="bg-green-600 text-white hover:bg-green-700">
                        <Plus className="mr-2 h-4 w-4" />
                        Add Link
                    </Button>
                </div>

                {links.length === 0 ? (
                    <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                        No links configured yet. Click "Add Link" to get started.
                    </div>
                ) : (
                    <div className="space-y-4">
                        {links.map((link, index) => (
                            <div
                                key={`internal-link-${index}`}
                                className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                            >
                                <div className="flex items-start gap-3">
                                    <span className="flex h-9 w-9 shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm font-semibold text-white">
                                        {index + 1}
                                    </span>
                                    <div className="flex-1 space-y-3">
                                        <div className="flex items-center justify-between gap-4">
                                            <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                Link Details
                                            </Label>
                                            <div className="flex items-center gap-3">
                                                <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                    <input
                                                        type="checkbox"
                                                        className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                        checked={link.is_active}
                                                        onChange={(event) => updateLink(index, { is_active: event.target.checked })}
                                                    />
                                                    Active
                                                </label>
                                                <button
                                                    type="button"
                                                    onClick={() => removeLink(index)}
                                                    className="p-1 text-red-500 hover:text-red-600"
                                                    title="Remove link"
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </button>
                                            </div>
                                        </div>

                                        <Input
                                            value={link.title}
                                            onChange={(event) => updateLink(index, { title: event.target.value })}
                                            placeholder="e.g., Essay Writing Service"
                                        />

                                        <Input
                                            value={link.url}
                                            onChange={(event) => updateLink(index, { url: event.target.value })}
                                            placeholder="/services/essay-writing"
                                        />

                                        <RichTextEditor
                                            value={link.description}
                                            onChange={(value) => updateLink(index, { description: value })}
                                            height={140}
                                            placeholder="Optional description for this link (rich text supported)."
                                        />
                                    </div>
                                    <div className="flex flex-col gap-1">
                                        <button
                                            type="button"
                                            onClick={() => moveLinkUp(index)}
                                            disabled={index === 0}
                                            className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                            title="Move up"
                                        >
                                            <ChevronUp className="h-5 w-5" />
                                        </button>
                                        <button
                                            type="button"
                                            onClick={() => moveLinkDown(index)}
                                            disabled={index === links.length - 1}
                                            className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                            title="Move down"
                                        >
                                            <ChevronDown className="h-5 w-5" />
                                        </button>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>

            <div className="sticky bottom-0 mt-6 border-t border-gray-200 bg-white pt-4 dark:border-gray-700 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Internal Links
                </button>
            </div>
        </div>
    );
}

function ServicesBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [services, setServices] = useState(content.services || []);

    const handleSave = () => {
        onUpdate(block.id, {...content, services});
    };

    const addService = () => {
        setServices([...services, { title: '', description: '', icon: '', link: '', color: 'blue', features: [] }]);
    };

    const updateService = (index, field, value) => {
        const updated = [...services];
        updated[index][field] = value;
        setServices(updated);
    };

    const addFeature = (serviceIndex) => {
        const updated = [...services];
        if (!updated[serviceIndex].features) {
            updated[serviceIndex].features = [];
        }
        updated[serviceIndex].features.push('');
        setServices(updated);
    };

    const updateFeature = (serviceIndex, featureIndex, value) => {
        const updated = [...services];
        updated[serviceIndex].features[featureIndex] = value;
        setServices(updated);
    };

    const removeFeature = (serviceIndex, featureIndex) => {
        const updated = [...services];
        updated[serviceIndex].features = updated[serviceIndex].features.filter((_, i) => i !== featureIndex);
        setServices(updated);
    };

    const removeService = (index) => {
        setServices(services.filter((_, i) => i !== index));
    };

    const moveServiceUp = (index) => {
        if (index === 0) return;
        const updated = [...services];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setServices(updated);
    };

    const moveServiceDown = (index) => {
        if (index === services.length - 1) return;
        const updated = [...services];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setServices(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="subtitle">Section Subtitle</Label>
                <Input
                    id="subtitle"
                    value={content.subtitle || ''}
                    onChange={(e) => setContent({...content, subtitle: e.target.value})}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Services ({services.length})</Label>
                    <button
                        type="button"
                        onClick={addService}
                        className="inline-flex items-center px-3 py-1.5 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition-colors"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Service
                    </button>
                </div>

                <div className="space-y-6">
                    {services.map((service, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Service #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveServiceUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveServiceDown(index)}
                                        disabled={index === services.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeService(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove Service"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                    <div>
                                        <Label htmlFor={`title-${index}`}>Service Title</Label>
                                        <Input
                                            id={`title-${index}`}
                                            value={service.title || ''}
                                            onChange={(e) => updateService(index, 'title', e.target.value)}
                                            placeholder="Enter service title..."
                                        />
                                    </div>

                                    <div>
                                        <Label htmlFor={`icon-${index}`}>Icon</Label>
                                        <Input
                                            id={`icon-${index}`}
                                            value={service.icon || ''}
                                            onChange={(e) => updateService(index, 'icon', e.target.value)}
                                            placeholder="file-text, search, code, etc."
                                        />
                                        <p className="text-xs text-gray-500 mt-1">Available: file-text, search, graduation-cap, clipboard-list, bar-chart-3, book-open, target, code</p>
                                    </div>
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                    <div>
                                        <Label htmlFor={`color-${index}`}>Color</Label>
                                        <select
                                            id={`color-${index}`}
                                            value={service.color || 'blue'}
                                            onChange={(e) => updateService(index, 'color', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                        >
                                            <option value="blue">Blue</option>
                                            <option value="green">Green</option>
                                            <option value="purple">Purple</option>
                                            <option value="orange">Orange</option>
                                            <option value="indigo">Indigo</option>
                                            <option value="teal">Teal</option>
                                            <option value="pink">Pink</option>
                                            <option value="red">Red</option>
                                        </select>
                                    </div>

                                    <div>
                                        <Label htmlFor={`link-${index}`}>Link (Optional)</Label>
                                        <Input
                                            id={`link-${index}`}
                                            value={service.link || ''}
                                            onChange={(e) => updateService(index, 'link', e.target.value)}
                                            placeholder="/service-page"
                                        />
                                    </div>
                                </div>

                                <div>
                                    <Label htmlFor={`description-${index}`}>Description</Label>
                                    <RichTextEditor
                                        value={service.description || ''}
                                        onChange={(value) => updateService(index, 'description', value)}
                                        height={200}
                                    />
                                </div>

                                <div>
                                    <div className="flex justify-between items-center mb-2">
                                        <Label>Features (Optional)</Label>
                                        <button
                                            type="button"
                                            onClick={() => addFeature(index)}
                                            className="inline-flex items-center px-2 py-1 text-xs bg-blue-600 text-white rounded hover:bg-blue-700"
                                        >
                                            <Plus className="h-3 w-3 mr-1" />
                                            Add Feature
                                        </button>
                                    </div>
                                    <div className="space-y-2">
                                        {service.features && service.features.map((feature, featureIndex) => (
                                            <div key={featureIndex} className="flex gap-2">
                                                <Input
                                                    value={feature}
                                                    onChange={(e) => updateFeature(index, featureIndex, e.target.value)}
                                                    placeholder={`Feature ${featureIndex + 1}`}
                                                    className="flex-1"
                                                />
                                                <button
                                                    type="button"
                                                    onClick={() => removeFeature(index, featureIndex)}
                                                    className="p-2 text-red-500 hover:text-red-700"
                                                    title="Remove Feature"
                                                >
                                                    <X className="h-4 w-4" />
                                                </button>
                                            </div>
                                        ))}
                                        {(!service.features || service.features.length === 0) && (
                                            <p className="text-sm text-gray-500 italic">No features added yet</p>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    ))}

                    {services.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No services yet. Click "Add Service" to get started.
                        </div>
                    )}
                </div>
            </div>
            
            <button
                type="button"
                onClick={handleSave}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </button>
        </div>
    );
}


