import { useMemo, useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

export default function ServiceCTABlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
        primaryActionText: ensureString(initialContent.primaryActionText, 'Get Started'),
        primaryActionLink: ensureString(initialContent.primaryActionLink, '/place-order'),
        secondaryActionText: ensureString(initialContent.secondaryActionText),
        secondaryActionLink: ensureString(initialContent.secondaryActionLink),
        showSecondary: ensureBoolean(initialContent.showSecondary ?? !!initialContent.secondaryActionText, false),
    });

    const handleSave = () => {
        const payload = {
            ...content,
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: content.description || '',
            primaryActionText: ensureString(content.primaryActionText, 'Get Started'),
            primaryActionLink: ensureString(content.primaryActionLink, '/place-order'),
            secondaryActionText: ensureString(content.secondaryActionText),
            secondaryActionLink: ensureString(content.secondaryActionLink),
            showSecondary: ensureBoolean(content.showSecondary, false),
        };

        onUpdate(block.id, payload);
    };

    return (
        <div className="space-y-6">
            <div className="space-y-4">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Call to Action Block</h3>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="service-cta-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-cta-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Ready to boost your grades?"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-cta-title">CTA Title</Label>
                        <Input
                            id="service-cta-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Hire a Professional Essay Writer Today"
                        />
                    </div>
                </div>

                <div>
                    <Label htmlFor="service-cta-description">Description (Rich Text)</Label>
                    <RichTextEditor
                        value={content.description}
                        onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                        height={180}
                        placeholder="Add supporting copy that motivates visitors to take action."
                    />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="service-cta-primaryText">Primary Button Text</Label>
                        <Input
                            id="service-cta-primaryText"
                            value={content.primaryActionText}
                            onChange={(event) => setContent((prev) => ({ ...prev, primaryActionText: event.target.value }))}
                            placeholder="Get Started"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-cta-primaryLink">Primary Button Link</Label>
                        <Input
                            id="service-cta-primaryLink"
                            value={content.primaryActionLink}
                            onChange={(event) => setContent((prev) => ({ ...prev, primaryActionLink: event.target.value }))}
                            placeholder="/place-order"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-cta-secondaryText">Secondary Button Text</Label>
                        <Input
                            id="service-cta-secondaryText"
                            value={content.secondaryActionText}
                            onChange={(event) => setContent((prev) => ({ ...prev, secondaryActionText: event.target.value, showSecondary: true }))}
                            placeholder="Talk to Support"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-cta-secondaryLink">Secondary Button Link</Label>
                        <Input
                            id="service-cta-secondaryLink"
                            value={content.secondaryActionLink}
                            onChange={(event) => setContent((prev) => ({ ...prev, secondaryActionLink: event.target.value, showSecondary: true }))}
                            placeholder="/contact"
                        />
                    </div>
                </div>

                <label className="flex items-center gap-2 text-sm text-gray-600 dark:text-gray-300">
                    <input
                        type="checkbox"
                        className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        checked={content.showSecondary}
                        onChange={(event) => setContent((prev) => ({ ...prev, showSecondary: event.target.checked }))}
                    />
                    Display secondary button when copy/link provided
                </label>
            </div>

            <div className="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-800 dark:bg-blue-900/20">
                <Label>Preview</Label>
                <div className="mt-3 flex flex-wrap gap-3">
                    <span className="rounded-md bg-blue-600 px-5 py-2.5 font-medium text-white shadow">
                        {content.primaryActionText || 'Get Started'}
                    </span>
                    {content.showSecondary && content.secondaryActionText ? (
                        <span className="rounded-md border border-blue-600 px-5 py-2.5 font-medium text-blue-600">
                            {content.secondaryActionText}
                        </span>
                    ) : null}
                </div>
            </div>

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service CTA
                </button>
            </div>
        </div>
    );
}
