import { useMemo, useState, useEffect } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, ChevronUp, ChevronDown, Trash2, Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

export default function ServiceFAQsBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
    });

    const [faqs, setFaqs] = useState(() => {
        if (!Array.isArray(initialContent.faqs)) {
            return [];
        }

        return initialContent.faqs.map((faq) => {
            // Handle both legacy format (just answer) and new format (answer_html)
            const answerValue = faq?.answer_html || faq?.answer || '';
            return {
                question: ensureString(faq?.question),
                answer: ensureRichText(answerValue),
                is_active: ensureBoolean(faq?.is_active, true), // Default to true if not set
            };
        });
    });

    // Sync state when block content changes (e.g., after save or external update)
    useEffect(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return;
        }

        const blockContent = block.content;
        
        // Update content state
        setContent({
            eyebrow: ensureString(blockContent.eyebrow),
            title: ensureString(blockContent.title),
            description: ensureRichText(blockContent.description_html || blockContent.description || ''),
        });

        // Update FAQs state
        if (Array.isArray(blockContent.faqs)) {
            setFaqs(
                blockContent.faqs.map((faq) => {
                    // Handle both legacy format (just answer) and new format (answer_html)
                    const answerValue = faq?.answer_html || faq?.answer || '';
                    return {
                        question: ensureString(faq?.question),
                        answer: ensureRichText(answerValue),
                        is_active: ensureBoolean(faq?.is_active, true), // Default to true if not set
                    };
                })
            );
        } else {
            setFaqs([]);
        }
    }, [block?.content, block?.id]);

    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        const descriptionHtml = content.description || '';
        const payload = {
            ...content,
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: descriptionHtml,
            description_html: descriptionHtml, // Ensure description_html is saved
            faqs: faqs.map((faq) => {
                const answerHtml = faq.answer || '';
                return {
                    question: ensureString(faq.question),
                    answer: answerHtml,
                    answer_html: answerHtml,
                    is_active: ensureBoolean(faq.is_active),
                };
            }),
        };

        onUpdate(block.id, payload);
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'faqs', label: 'FAQs', icon: '❓' },
    ];

    const addFaq = () => {
        setFaqs((current) => [
            ...current,
            { question: '', answer: '', is_active: true },
        ]);
    };

    const updateFaq = (index, partial) => {
        setFaqs((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeFaq = (index) => {
        setFaqs((current) => current.filter((_, idx) => idx !== index));
    };

    const moveFaqUp = (index) => {
        if (index === 0) return;
        setFaqs((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveFaqDown = (index) => {
        if (index === faqs.length - 1) return;
        setFaqs((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap gap-2 border-b border-gray-200 pb-3 dark:border-gray-700">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 text-gray-700 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">FAQs Section</h3>

                    <div>
                        <Label htmlFor="service-faqs-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-faqs-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Answers before you order"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-faqs-title">Section Title</Label>
                        <Input
                            id="service-faqs-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="Frequently Asked Questions"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-faqs-description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description}
                            onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                            height={180}
                            placeholder="Provide context or instructions for the FAQ section."
                        />
                    </div>
                </div>
            )}

            {activeTab === 'faqs' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">FAQ Items</h3>
                        <Button type="button" onClick={addFaq} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add FAQ
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">FAQs display in an accordion layout.</p>

                    {faqs.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No FAQs yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {faqs.map((faq, index) => (
                                <div
                                    key={`service-faq-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-orange-500 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Question & Answer
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={faq.is_active}
                                                            onChange={(event) => updateFaq(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeFaq(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove FAQ"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Question</Label>
                                                <Input
                                                    value={faq.question}
                                                    onChange={(event) => updateFaq(index, { question: event.target.value })}
                                                    placeholder="Can I request revisions?"
                                                />
                                            </div>
                                            <div>
                                                <Label>Answer (Rich Text)</Label>
                                                <RichTextEditor
                                                    value={faq.answer}
                                                    onChange={(value) => updateFaq(index, { answer: value })}
                                                    height={160}
                                                    placeholder="Provide a helpful answer. Use bullets for steps/promises."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveFaqUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveFaqDown(index)}
                                                disabled={index === faqs.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service FAQs
                </button>
            </div>
        </div>
    );
}
