import { useMemo, useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { ChevronDown, ChevronUp, Plus, Save, Trash2 } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

export default function ServiceLinksBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
        linkCategory: ensureString(initialContent.linkCategory, 'related') || 'related',
    });

    const [links, setLinks] = useState(() => {
        if (!Array.isArray(initialContent.links)) {
            return [];
        }

        return initialContent.links.map((link) => ({
            title: link?.title || '',
            url: link?.url || '',
            description: ensureRichText(link?.description_html || link?.description || ''),
            is_active: ensureBoolean(link?.is_active),
        }));
    });

    const handleSave = () => {
        const payload = {
            ...content,
            description: content.description || '',
            description_html: content.description || '',
            links: links.map((link) => ({
                ...link,
                description: link.description || '',
                description_html: link.description || '',
            })),
        };

        onUpdate(block.id, payload);
    };

    const addLink = () => {
        setLinks([
            ...links,
            { title: '', url: '', description: '', is_active: true },
        ]);
    };

    const updateLink = (index, partial) => {
        setLinks((current) => {
            const clone = [...current];
            clone[index] = {
                ...clone[index],
                ...partial,
            };
            return clone;
        });
    };

    const removeLink = (index) => {
        setLinks((current) => current.filter((_, itemIndex) => itemIndex !== index));
    };

    const moveLinkUp = (index) => {
        if (index === 0) return;
        setLinks((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveLinkDown = (index) => {
        if (index === links.length - 1) return;
        setLinks((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <Label htmlFor="service-links-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-links-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Explore more services"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-links-title">Block Title</Label>
                        <Input
                            id="service-links-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="e.g., Related Writing Services"
                        />
                    </div>
                </div>

                <div>
                    <Label htmlFor="service-links-description">Description (Rich Text)</Label>
                    <RichTextEditor
                        value={content.description}
                        onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                        height={160}
                        placeholder="Optional supporting copy for the links section."
                    />
                </div>

                <div>
                    <Label htmlFor="service-links-category">Link Category</Label>
                    <select
                        id="service-links-category"
                        value={content.linkCategory}
                        onChange={(event) => setContent((prev) => ({ ...prev, linkCategory: event.target.value }))}
                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-700 dark:bg-gray-800 dark:text-gray-100"
                    >
                        <option value="related">Related Services</option>
                        <option value="popular">Popular Services</option>
                        <option value="supplemental">Supplemental Services</option>
                        <option value="custom">Custom Category</option>
                    </select>
                </div>
            </div>

            <div className="border-t border-gray-200 pt-4 dark:border-gray-700">
                <div className="mb-4 flex items-center justify-between">
                    <div>
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Links</h3>
                        <p className="text-sm text-gray-600 dark:text-gray-400">
                            Manage outbound links for related or supplemental services.
                        </p>
                    </div>
                    <Button type="button" onClick={addLink} className="bg-green-600 text-white hover:bg-green-700">
                        <Plus className="mr-2 h-4 w-4" />
                        Add Link
                    </Button>
                </div>

                {links.length === 0 ? (
                    <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                        No links configured yet.
                    </div>
                ) : (
                    <div className="space-y-4">
                        {links.map((link, index) => (
                            <div
                                key={`service-link-${index}`}
                                className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                            >
                                <div className="flex items-start gap-3">
                                    <span className="flex h-9 w-9 shrink-0 items-center justify-center rounded-full bg-purple-600 text-sm font-semibold text-white">
                                        {index + 1}
                                    </span>
                                    <div className="flex-1 space-y-3">
                                        <div className="flex items-center justify-between gap-4">
                                            <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                Link Details
                                            </Label>
                                            <div className="flex items-center gap-3">
                                                <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                    <input
                                                        type="checkbox"
                                                        className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                        checked={link.is_active}
                                                        onChange={(event) => updateLink(index, { is_active: event.target.checked })}
                                                    />
                                                    Active
                                                </label>
                                                <button
                                                    type="button"
                                                    onClick={() => removeLink(index)}
                                                    className="p-1 text-red-500 hover:text-red-600"
                                                    title="Remove link"
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </button>
                                            </div>
                                        </div>

                                        <Input
                                            value={link.title}
                                            onChange={(event) => updateLink(index, { title: event.target.value })}
                                            placeholder="e.g., Dissertation writing service"
                                        />

                                        <Input
                                            value={link.url}
                                            onChange={(event) => updateLink(index, { url: event.target.value })}
                                            placeholder="/services/dissertation-writing"
                                        />

                                        <RichTextEditor
                                            value={link.description}
                                            onChange={(value) => updateLink(index, { description: value })}
                                            height={140}
                                            placeholder="Optional supporting copy for this link."
                                        />
                                    </div>
                                    <div className="flex flex-col gap-1">
                                        <button
                                            type="button"
                                            onClick={() => moveLinkUp(index)}
                                            disabled={index === 0}
                                            className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                            title="Move up"
                                        >
                                            <ChevronUp className="h-5 w-5" />
                                        </button>
                                        <button
                                            type="button"
                                            onClick={() => moveLinkDown(index)}
                                            disabled={index === links.length - 1}
                                            className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                            title="Move down"
                                        >
                                            <ChevronDown className="h-5 w-5" />
                                        </button>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>

            <div className="sticky bottom-0 mt-6 border-t border-gray-200 bg-white pt-4 dark:border-gray-700 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service Links
                </button>
            </div>
        </div>
    );
}
