import { useMemo, useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

const combineLegacySections = (sections = []) => {
    if (!Array.isArray(sections)) {
        return '';
    }

    const escapeHtml = (value) =>
        ensureString(value)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');

    return sections
        .filter((section) => ensureBoolean(section?.is_active) !== false)
        .map((section) => {
            const title = ensureString(section?.title).trim();
            const body = ensureRichText(section?.body_html || section?.body || '');

            if (!body) {
                return '';
            }

            const heading = title ? `<h3>${escapeHtml(title)}</h3>` : '';
            return `${heading}${body}`;
        })
        .filter(Boolean)
        .join('<br /><br />');
};

export default function ServiceLongformBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const legacyHtml = useMemo(
        () => combineLegacySections(initialContent.sections),
        [initialContent.sections]
    );

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(
            initialContent.description_html || initialContent.description || legacyHtml
        ),
        preferredHeight:
            typeof initialContent.preferredHeight === 'number' &&
            !Number.isNaN(initialContent.preferredHeight)
                ? initialContent.preferredHeight
                : 480,
    });

    const handleSave = () => {
        const html = ensureRichText(content.description);

        const payload = {
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: html,
            description_html: html,
            preferredHeight:
                typeof content.preferredHeight === 'number' &&
                !Number.isNaN(content.preferredHeight)
                    ? content.preferredHeight
                    : 480,
            sections: [],
        };

        onUpdate(block.id, payload);
    };

    return (
        <div className="space-y-6">
            <div className="space-y-4">
                <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                    <div>
                        <Label htmlFor="service-scrollable-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-scrollable-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., Deep Dive"
                        />
                    </div>
                    <div>
                        <Label htmlFor="service-scrollable-title">Section Title</Label>
                        <Input
                            id="service-scrollable-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="e.g., Why Students Trust Our Writers"
                        />
                    </div>
                </div>

                <div>
                    <Label htmlFor="service-scrollable-description">Scrollable Content (Rich Text)</Label>
                    <RichTextEditor
                        value={content.description}
                        onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                        height={320}
                        placeholder="Add long-form content that appears inside the scrollable panel."
                    />
                </div>

                <div>
                    <Label htmlFor="service-scrollable-height">Preferred Height (px)</Label>
                    <input
                        id="service-scrollable-height"
                        type="number"
                        min="280"
                        max="1200"
                        value={content.preferredHeight}
                        onChange={(event) =>
                            setContent((prev) => ({ ...prev, preferredHeight: Number(event.target.value) || 480 }))
                        }
                        className="w-32 rounded-md border border-gray-300 px-3 py-2 text-sm text-gray-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-700 dark:bg-gray-800 dark:text-gray-100"
                    />
                </div>
            </div>

            <div className="sticky bottom-0 border-t border-gray-200 bg-white pt-4 dark:border-gray-700 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Scrollable Section
                </button>
            </div>
        </div>
    );
}
