import { useMemo, useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, ChevronUp, ChevronDown, Trash2, Save } from 'lucide-react';
import { ensureBoolean, ensureRichText, ensureString } from './editorUtils';

export default function ServiceProcessBlockEditor({ block, onUpdate }) {
    const initialContent = useMemo(() => {
        if (!block?.content || typeof block.content !== 'object') {
            return {};
        }
        return block.content;
    }, [block]);

    const [content, setContent] = useState({
        eyebrow: ensureString(initialContent.eyebrow),
        title: ensureString(initialContent.title),
        description: ensureRichText(initialContent.description_html || initialContent.description || ''),
    });

    const [steps, setSteps] = useState(() => {
        if (!Array.isArray(initialContent.steps)) {
            return [];
        }

        return initialContent.steps.map((step) => ({
            icon: ensureString(step?.icon, 'Target') || 'Target',
            title: ensureString(step?.title),
            description: ensureRichText(step?.description_html || step?.description || ''),
            is_active: ensureBoolean(step?.is_active),
        }));
    });

    const [activeTab, setActiveTab] = useState('main');

    const handleSave = () => {
        const payload = {
            ...content,
            eyebrow: ensureString(content.eyebrow),
            title: ensureString(content.title),
            description: content.description || '',
            steps: steps.map((step, index) => ({
                position: index + 1,
                icon: ensureString(step.icon, 'Target') || 'Target',
                title: ensureString(step.title),
                description: step.description || '',
                is_active: ensureBoolean(step.is_active),
            })),
        };

        onUpdate(block.id, payload);
    };

    const tabs = [
        { id: 'main', label: 'Main Content', icon: '📝' },
        { id: 'steps', label: 'Process Steps', icon: '🔢' },
    ];

    const iconOptions = [
        { value: 'Target', label: '🎯 Target' },
        { value: 'Search', label: '🔍 Research' },
        { value: 'ClipboardList', label: '📋 Plan' },
        { value: 'PenTool', label: '✒️ Writing' },
        { value: 'Edit3', label: '📝 Draft' },
        { value: 'BookOpen', label: '📖 Review' },
        { value: 'CheckCircle', label: '✅ Approve' },
        { value: 'Send', label: '📦 Deliver' },
        { value: 'Clock', label: '⏰ Timing' },
        { value: 'Shield', label: '🛡️ Assurance' },
        { value: 'Star', label: '⭐ Quality' },
        { value: 'Users', label: '👥 Collaboration' },
    ];

    const addStep = () => {
        setSteps((current) => [
            ...current,
            { icon: 'Target', title: '', description: '', is_active: true },
        ]);
    };

    const updateStep = (index, partial) => {
        setSteps((current) => {
            const clone = [...current];
            clone[index] = { ...clone[index], ...partial };
            return clone;
        });
    };

    const removeStep = (index) => {
        setSteps((current) => current.filter((_, idx) => idx !== index));
    };

    const moveStepUp = (index) => {
        if (index === 0) return;
        setSteps((current) => {
            const clone = [...current];
            [clone[index - 1], clone[index]] = [clone[index], clone[index - 1]];
            return clone;
        });
    };

    const moveStepDown = (index) => {
        if (index === steps.length - 1) return;
        setSteps((current) => {
            const clone = [...current];
            [clone[index], clone[index + 1]] = [clone[index + 1], clone[index]];
            return clone;
        });
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap gap-2 border-b border-gray-200 pb-3 dark:border-gray-700">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 text-gray-700 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {activeTab === 'main' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Process Section</h3>

                    <div>
                        <Label htmlFor="service-process-eyebrow">Eyebrow</Label>
                        <Input
                            id="service-process-eyebrow"
                            value={content.eyebrow}
                            onChange={(event) => setContent((prev) => ({ ...prev, eyebrow: event.target.value }))}
                            placeholder="e.g., A simple four-step process"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-process-title">Section Title</Label>
                        <Input
                            id="service-process-title"
                            value={content.title}
                            onChange={(event) => setContent((prev) => ({ ...prev, title: event.target.value }))}
                            placeholder="How Our Essay Writing Process Works"
                        />
                    </div>

                    <div>
                        <Label htmlFor="service-process-description">Section Description (Rich Text)</Label>
                        <RichTextEditor
                            value={content.description}
                            onChange={(value) => setContent((prev) => ({ ...prev, description: value }))}
                            height={180}
                            placeholder="Share context for the process and set expectations."
                        />
                    </div>
                </div>
            )}

            {activeTab === 'steps' && (
                <div className="space-y-4">
                    <div className="flex items-center justify-between">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Process Steps</h3>
                        <Button type="button" onClick={addStep} className="bg-green-600 text-white hover:bg-green-700">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Step
                        </Button>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400">Steps are displayed with sequential numbering.</p>

                    {steps.length === 0 ? (
                        <div className="rounded-lg border-2 border-dashed border-gray-300 p-8 text-center text-sm text-gray-500 dark:border-gray-600 dark:text-gray-400">
                            No steps yet.
                        </div>
                    ) : (
                        <div className="space-y-4">
                            {steps.map((step, index) => (
                                <div
                                    key={`service-process-step-${index}`}
                                    className="rounded-lg border border-gray-200 bg-gray-50 p-4 shadow-sm dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <div className="flex items-start gap-3">
                                        <span className="flex h-9 w-9 shrink-0 items-center justify-center rounded-full bg-indigo-600 text-sm font-semibold text-white">
                                            {index + 1}
                                        </span>
                                        <div className="flex-1 space-y-3">
                                            <div className="flex items-center justify-between gap-3">
                                                <Label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                                                    Step Details
                                                </Label>
                                                <div className="flex items-center gap-3">
                                                    <label className="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-300">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                            checked={step.is_active}
                                                            onChange={(event) => updateStep(index, { is_active: event.target.checked })}
                                                        />
                                                        Active
                                                    </label>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeStep(index)}
                                                        className="p-1 text-red-500 hover:text-red-600"
                                                        title="Remove step"
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </button>
                                                </div>
                                            </div>
                                            <div>
                                                <Label>Icon</Label>
                                                <select
                                                    value={step.icon || 'Target'}
                                                    onChange={(event) => updateStep(index, { icon: event.target.value })}
                                                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                                                >
                                                    {iconOptions.map((opt) => (
                                                        <option key={opt.value} value={opt.value}>
                                                            {opt.label}
                                                        </option>
                                                    ))}
                                                </select>
                                            </div>
                                            <div>
                                                <Label>Step Title</Label>
                                                <Input
                                                    value={step.title}
                                                    onChange={(event) => updateStep(index, { title: event.target.value })}
                                                    placeholder={`Step ${index + 1} title`}
                                                />
                                            </div>
                                            <div>
                                                <Label>Step Description</Label>
                                                <RichTextEditor
                                                    value={step.description}
                                                    onChange={(value) => updateStep(index, { description: value })}
                                                    height={160}
                                                    placeholder="Explain what happens during this phase."
                                                />
                                            </div>
                                        </div>
                                        <div className="flex flex-col gap-1">
                                            <button
                                                type="button"
                                                onClick={() => moveStepUp(index)}
                                                disabled={index === 0}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move up"
                                            >
                                                <ChevronUp className="h-5 w-5" />
                                            </button>
                                            <button
                                                type="button"
                                                onClick={() => moveStepDown(index)}
                                                disabled={index === steps.length - 1}
                                                className="p-1 text-gray-500 hover:text-blue-600 disabled:cursor-not-allowed disabled:opacity-30"
                                                title="Move down"
                                            >
                                                <ChevronDown className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            )}

            <div className="sticky bottom-0 bg-white pt-4 dark:bg-gray-900">
                <button
                    type="button"
                    onClick={handleSave}
                    className="flex w-full items-center justify-center rounded-lg bg-blue-600 px-6 py-3 font-medium text-white shadow hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                    <Save className="mr-2 h-5 w-5" />
                    Save Service Process
                </button>
            </div>
        </div>
    );
}
