import React, { useState, useEffect } from 'react';
import { Head, Link, router, usePage } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { toast } from 'react-hot-toast';
import { 
    Plus, 
    Edit, 
    Trash2,
    Eye,
    EyeOff,
    AlertCircle,
    Info,
    Link as LinkIcon,
    Mail,
    Phone,
    FileText
} from 'lucide-react';

export default function FooterIndex({ settings }) {
    const [deleting, setDeleting] = useState(null);
    const { flash } = usePage().props;

    // Handle flash messages
    useEffect(() => {
        if (flash?.success) {
            toast.success(flash.success, { 
                duration: 4000,
                icon: '✅'
            });
        }
        if (flash?.error) {
            toast.error(flash.error, { 
                duration: 4000,
                icon: '❌'
            });
        }
    }, [flash]);

    const handleDelete = (id) => {
        if (confirm('Are you sure you want to delete this footer setting?')) {
            setDeleting(id);
            router.delete(route('admin.cms.footer.destroy', id), {
                onSuccess: () => {
                    toast.success('Footer setting deleted successfully!', {
                        icon: '🗑️',
                        duration: 3000
                    });
                },
                onError: () => {
                    toast.error('Failed to delete footer setting', {
                        icon: '❌',
                        duration: 4000
                    });
                },
                onFinish: () => setDeleting(null),
            });
        }
    };

    const handleToggleActive = (id) => {
        router.post(route('admin.cms.footer.toggle-active', id), {
            preserveScroll: true,
            onSuccess: () => {
                toast.success('Status updated successfully!', {
                    icon: '👁️',
                    duration: 2000
                });
            },
            onError: () => {
                toast.error('Failed to update status', {
                    icon: '❌',
                    duration: 3000
                });
            }
        });
    };

    const getSectionIcon = (sectionType) => {
        switch (sectionType) {
            case 'company_info':
                return <Info className="h-5 w-5" />;
            case 'service_link':
            case 'quick_link':
                return <LinkIcon className="h-5 w-5" />;
            case 'contact_info':
                return <Phone className="h-5 w-5" />;
            case 'legal_link':
                return <FileText className="h-5 w-5" />;
            default:
                return <FileText className="h-5 w-5" />;
        }
    };

    const getSectionTitle = (sectionType) => {
        const titles = {
            'company_info': 'Company Information',
            'service_link': 'Service Links',
            'quick_link': 'Quick Links',
            'contact_info': 'Contact Information',
            'legal_link': 'Legal Links',
        };
        return titles[sectionType] || sectionType;
    };

    const renderSettingPreview = (setting) => {
        const data = setting.data;
        
        switch (setting.section_type) {
            case 'company_info':
                return (
                    <div className="text-sm">
                        {data.title && <p className="font-semibold">{data.title}</p>}
                        {data.description && (
                            <p className="text-gray-600 dark:text-gray-400 mt-1 line-clamp-2" 
                               dangerouslySetInnerHTML={{ __html: data.description }} />
                        )}
                        {data.text && <p className="text-gray-600 dark:text-gray-400">{data.text}</p>}
                    </div>
                );
            case 'service_link':
            case 'quick_link':
            case 'legal_link':
                return (
                    <div className="text-sm">
                        <p className="font-semibold">{data.name}</p>
                        <p className="text-gray-600 dark:text-gray-400 flex items-center gap-1">
                            <LinkIcon className="h-3 w-3" />
                            {data.url}
                        </p>
                    </div>
                );
            case 'contact_info':
                return (
                    <div className="text-sm space-y-1">
                        {data.email && (
                            <p className="flex items-center gap-2">
                                <Mail className="h-4 w-4 text-blue-500" />
                                {data.email}
                            </p>
                        )}
                        {data.phone && (
                            <p className="flex items-center gap-2">
                                <Phone className="h-4 w-4 text-blue-500" />
                                {data.phone}
                            </p>
                        )}
                    </div>
                );
            default:
                return <p className="text-sm text-gray-600 dark:text-gray-400">No preview available</p>;
        }
    };

    return (
        <AdminLayout>
            <Head title="Footer Management" />

            <div className="space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Footer Management
                        </h1>
                        <p className="mt-2 text-gray-600 dark:text-gray-400">
                            Manage all footer content across your website
                        </p>
                    </div>
                    <Link href={route('admin.cms.footer.create')}>
                        <Button className="flex items-center gap-2">
                            <Plus className="h-4 w-4" />
                            Add Footer Item
                        </Button>
                    </Link>
                </div>

                {/* Info Alert */}
                <Card className="bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-800">
                    <CardContent className="p-4">
                        <div className="flex items-start gap-3">
                            <AlertCircle className="h-5 w-5 text-blue-600 dark:text-blue-400 mt-0.5" />
                            <div>
                                <p className="text-sm font-medium text-blue-900 dark:text-blue-200">
                                    Footer is CMS-Driven
                                </p>
                                <p className="text-sm text-blue-700 dark:text-blue-300 mt-1">
                                    Changes made here will immediately reflect on your website footer. 
                                    The cache is automatically cleared when updates are made.
                                </p>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Footer Sections */}
                <div className="space-y-6">
                    {Object.entries(settings).map(([sectionType, sectionSettings]) => (
                        <Card key={sectionType}>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-3">
                                    <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-lg text-blue-600 dark:text-blue-400">
                                        {getSectionIcon(sectionType)}
                                    </div>
                                    <div>
                                        <h3 className="text-lg font-semibold">
                                            {getSectionTitle(sectionType)}
                                        </h3>
                                        <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                            {sectionSettings.length} {sectionSettings.length === 1 ? 'item' : 'items'}
                                        </p>
                                    </div>
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3">
                                    {sectionSettings.map((setting) => (
                                        <div
                                            key={setting.id}
                                            className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-800/50 rounded-lg border border-gray-200 dark:border-gray-700"
                                        >
                                            <div className="flex items-start gap-4 flex-1">
                                                <div className="flex items-center gap-2">
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        #{setting.order}
                                                    </span>
                                                </div>
                                                <div className="flex-1">
                                                    {renderSettingPreview(setting)}
                                                </div>
                                            </div>

                                            <div className="flex items-center gap-2">
                                                <Badge
                                                    variant={setting.is_active ? 'success' : 'secondary'}
                                                    className="cursor-pointer"
                                                    onClick={() => handleToggleActive(setting.id)}
                                                >
                                                    {setting.is_active ? (
                                                        <>
                                                            <Eye className="h-3 w-3 mr-1" />
                                                            Active
                                                        </>
                                                    ) : (
                                                        <>
                                                            <EyeOff className="h-3 w-3 mr-1" />
                                                            Inactive
                                                        </>
                                                    )}
                                                </Badge>

                                                <Link href={route('admin.cms.footer.edit', setting.id)}>
                                                    <Button variant="outline" size="sm">
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                </Link>

                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() => handleDelete(setting.id)}
                                                    disabled={deleting === setting.id}
                                                    className="text-red-600 hover:text-red-700 hover:bg-red-50 dark:text-red-400 dark:hover:bg-red-900/20"
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </div>
                                    ))}

                                    {sectionSettings.length === 0 && (
                                        <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                                            <p>No items in this section</p>
                                        </div>
                                    )}
                                </div>
                            </CardContent>
                        </Card>
                    ))}

                    {Object.keys(settings).length === 0 && (
                        <Card>
                            <CardContent className="py-12">
                                <div className="text-center">
                                    <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                                        No Footer Settings Yet
                                    </h3>
                                    <p className="text-gray-600 dark:text-gray-400 mb-6">
                                        Get started by creating your first footer setting
                                    </p>
                                    <Link href={route('admin.cms.footer.create')}>
                                        <Button>
                                            <Plus className="h-4 w-4 mr-2" />
                                            Add Footer Item
                                        </Button>
                                    </Link>
                                </div>
                            </CardContent>
                        </Card>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}

