import React, { useEffect, useState } from 'react';
import { Head, useForm } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/Components/ui/Select';
import { ArrowLeft, Save, Eye } from 'lucide-react';

export default function CreatePage({ pageTypes, templates, parentPages }) {
    const { data, setData, post, processing, errors } = useForm({
        title: '',
        slug: '',
        page_type: 'service',
        status: 'draft',
        template: 'service',
        parent_id: null,
        seo_priority: 5,
        published_at: null,
        use_template: true,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post('/admin/cms/pages');
    };

    const generateSlug = (title) => {
        return title
            .toLowerCase()
            .replace(/[^a-z0-9\s-]/g, '')
            .replace(/\s+/g, '-')
            .replace(/-+/g, '-')
            .trim('-');
    };

    const handleTitleChange = (e) => {
        const title = e.target.value;
        setData('title', title);
        if (!data.slug) {
            setData('slug', generateSlug(title));
        }
    };

    useEffect(() => {
        if (data.page_type !== 'service' && data.use_template) {
            setData('use_template', false);
        }
    }, [data.page_type]);

    useEffect(() => {
        if (data.page_type === 'service' && data.use_template && data.template !== 'service') {
            setData('template', 'service');
        }
    }, [data.page_type, data.use_template, data.template]);

    return (
        <AdminLayout>
            <Head title="Create New Page" />
            <div className="space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                            Create New Page
                        </h1>
                        <p className="text-gray-600 dark:text-gray-400">
                            Add a new page to your content management system
                        </p>
                    </div>
                    <div className="flex space-x-3">
                        <Button
                            variant="outline"
                            onClick={() => window.history.back()}
                        >
                            <ArrowLeft className="w-4 h-4 mr-2" />
                            Back
                        </Button>
                        <Button
                            type="submit"
                            form="page-form"
                            disabled={processing}
                        >
                            <Save className="w-4 h-4 mr-2" />
                            {processing ? 'Creating...' : 'Create Page'}
                        </Button>
                    </div>
                </div>

                {/* Form */}
                <form id="page-form" onSubmit={handleSubmit} className="space-y-6">
                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        {/* Main Content */}
                        <div className="lg:col-span-2 space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Page Details</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div>
                                        <Label htmlFor="title">Page Title *</Label>
                                        <Input
                                            id="title"
                                            type="text"
                                            value={data.title}
                                            onChange={handleTitleChange}
                                            placeholder="Enter page title"
                                            className={errors.title ? 'border-red-500' : ''}
                                        />
                                        {errors.title && (
                                            <p className="text-red-500 text-sm mt-1">{errors.title}</p>
                                        )}
                                    </div>

                                    <div>
                                        <Label htmlFor="slug">URL Slug *</Label>
                                        <Input
                                            id="slug"
                                            type="text"
                                            value={data.slug}
                                            onChange={(e) => setData('slug', e.target.value)}
                                            placeholder="page-url-slug"
                                            className={errors.slug ? 'border-red-500' : ''}
                                        />
                                        {errors.slug && (
                                            <p className="text-red-500 text-sm mt-1">{errors.slug}</p>
                                        )}
                                        <p className="text-gray-500 text-sm mt-1">
                                            URL: /{data.slug || 'page-url-slug'}
                                        </p>
                                    </div>

                                    <div>
                                        <Label htmlFor="page_type">Page Type *</Label>
                                        <Select
                                            value={data.page_type}
                                            onValueChange={(value) => setData('page_type', value)}
                                        >
                                            <SelectTrigger className={errors.page_type ? 'border-red-500' : ''}>
                                                <SelectValue placeholder="Select page type" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                {pageTypes?.map((type) => (
                                                    <SelectItem key={type.value} value={type.value}>
                                                        {type.label}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                        {errors.page_type && (
                                            <p className="text-red-500 text-sm mt-1">{errors.page_type}</p>
                                        )}
                                    </div>

                                    {data.page_type === 'service' && (
                                        <div className="flex items-start space-x-3 rounded-md border border-blue-100 bg-blue-50 p-3 dark:border-blue-900/60 dark:bg-blue-950/40">
                                            <input
                                                id="use_template"
                                                type="checkbox"
                                                checked={data.use_template}
                                                onChange={(event) => setData('use_template', event.target.checked)}
                                                className="mt-1 h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                            />
                                            <div>
                                                <Label htmlFor="use_template" className="font-medium text-blue-900 dark:text-blue-200">
                                                    Scaffold canonical service layout
                                                </Label>
                                                <p className="mt-1 text-xs text-blue-800 dark:text-blue-300">
                                                    Generates the standardized <code>service_*</code> block stack with TipTap editors, matching the service-page workflow.
                                                </p>
                                            </div>
                                        </div>
                                    )}

                                    <div>
                                        <Label htmlFor="template">Template</Label>
                                        <Select
                                            value={data.template}
                                            onValueChange={(value) => setData('template', value)}
                                            disabled={data.page_type === 'service' && data.use_template}
                                        >
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select template" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                {templates?.map((template) => (
                                                    <SelectItem key={template.value} value={template.value}>
                                                        {template.label}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                    </div>
                                </CardContent>
                            </Card>
                        </div>

                        {/* Sidebar */}
                        <div className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Publishing</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div>
                                        <Label htmlFor="status">Status</Label>
                                        <Select
                                            value={data.status}
                                            onValueChange={(value) => setData('status', value)}
                                        >
                                            <SelectTrigger>
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="draft">Draft</SelectItem>
                                                <SelectItem value="published">Published</SelectItem>
                                                <SelectItem value="archived">Archived</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>

                                    <div>
                                        <Label htmlFor="parent_id">Parent Page</Label>
                                        <Select
                                            value={data.parent_id || ''}
                                            onValueChange={(value) => setData('parent_id', value || null)}
                                        >
                                            <SelectTrigger>
                                                <SelectValue placeholder="No parent page" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="">No parent page</SelectItem>
                                                {parentPages?.map((page) => (
                                                    <SelectItem key={page.id} value={page.id.toString()}>
                                                        {page.title}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                    </div>

                                    <div>
                                        <Label htmlFor="seo_priority">SEO Priority</Label>
                                        <Select
                                            value={data.seo_priority.toString()}
                                            onValueChange={(value) => setData('seo_priority', parseInt(value))}
                                        >
                                            <SelectTrigger>
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="1">1 - Highest</SelectItem>
                                                <SelectItem value="2">2 - High</SelectItem>
                                                <SelectItem value="3">3 - Medium-High</SelectItem>
                                                <SelectItem value="4">4 - Medium</SelectItem>
                                                <SelectItem value="5">5 - Default</SelectItem>
                                                <SelectItem value="6">6 - Low</SelectItem>
                                                <SelectItem value="7">7 - Lowest</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Preview</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="text-sm text-gray-600 dark:text-gray-400 space-y-2">
                                        <p><strong>Title:</strong> {data.title || 'Untitled Page'}</p>
                                        <p><strong>URL:</strong> /{data.slug || 'page-url-slug'}</p>
                                        <p><strong>Type:</strong> {data.page_type}</p>
                                        <p><strong>Status:</strong> {data.status}</p>
                                    </div>
                                    <Button
                                        type="button"
                                        variant="outline"
                                        size="sm"
                                        className="w-full mt-4"
                                        disabled={!data.slug}
                                    >
                                        <Eye className="w-4 h-4 mr-2" />
                                        Preview Page
                                    </Button>
                                </CardContent>
                            </Card>
                        </div>
                    </div>
                </form>
            </div>
        </AdminLayout>
    );
}


