import React from 'react';
import { Head, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Badge } from '@/Components/ui/Badge';
import { 
    ArrowLeft, 
    Edit, 
    Copy, 
    Trash2, 
    Eye, 
    Calendar,
    FileText,
    Search,
    Image,
    Link as LinkIcon,
    MoreVertical
} from 'lucide-react';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/Components/ui/dropdown-menu';

export default function ShowPage({ page }) {
    const getStatusColor = (status) => {
        switch (status) {
            case 'published': return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300';
            case 'draft': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300';
            case 'archived': return 'bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300';
            default: return 'bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300';
        }
    };

    const formatDate = (date) => {
        if (!date) return 'Not set';
        return new Date(date).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    return (
        <AdminLayout>
            <Head title={`${page.title} - Page Details`} />
            <div className="space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                        <Button
                            variant="outline"
                            onClick={() => window.history.back()}
                        >
                            <ArrowLeft className="w-4 h-4 mr-2" />
                            Back
                        </Button>
                        <div>
                            <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                {page.title}
                            </h1>
                            <p className="text-gray-600 dark:text-gray-400">
                                /{page.slug}
                            </p>
                        </div>
                    </div>
                    <div className="flex items-center space-x-3">
                        <Badge className={getStatusColor(page.status)}>
                            {page.status}
                        </Badge>
                        <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                                <Button variant="outline" size="sm">
                                    <MoreVertical className="w-4 h-4" />
                                </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                                <Link href={`/services/${page.slug}`} target="_blank">
                                    <DropdownMenuItem>
                                        <Eye className="w-4 h-4 mr-2" />
                                        View Page
                                    </DropdownMenuItem>
                                </Link>
                                <Link href={`/admin/cms/pages/${page.id}/edit`}>
                                    <DropdownMenuItem>
                                        <Edit className="w-4 h-4 mr-2" />
                                        Edit Page
                                    </DropdownMenuItem>
                                </Link>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem>
                                    <Copy className="w-4 h-4 mr-2" />
                                    Duplicate Page
                                </DropdownMenuItem>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem className="text-red-600 dark:text-red-400">
                                    <Trash2 className="w-4 h-4 mr-2" />
                                    Delete Page
                                </DropdownMenuItem>
                            </DropdownMenuContent>
                        </DropdownMenu>
                    </div>
                </div>

                {/* Page Information */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Main Content */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Basic Information */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <FileText className="w-5 h-5 mr-2" />
                                    Page Information
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Page Type
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white capitalize">
                                            {page.page_type?.replace('_', ' ')}
                                        </p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Template
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {page.template || 'Default'}
                                        </p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            SEO Priority
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {page.seo_priority || 5}
                                        </p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Parent Page
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white">
                                            {page.parent?.title || 'None'}
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Content Blocks */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center justify-between">
                                    <span className="flex items-center">
                                        <FileText className="w-5 h-5 mr-2" />
                                        Content Blocks
                                    </span>
                                    <Badge variant="secondary">
                                        {page.content_blocks?.length || 0} blocks
                                    </Badge>
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                {page.content_blocks && page.content_blocks.length > 0 ? (
                                    <div className="space-y-3">
                                        {page.content_blocks.map((block, index) => (
                                            <div key={block.id} className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                                                <div className="flex items-center space-x-3">
                                                    <div className="w-8 h-8 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center">
                                                        <span className="text-sm font-medium text-blue-600 dark:text-blue-400">
                                                            {block.order}
                                                        </span>
                                                    </div>
                                                    <div>
                                                        <p className="font-medium text-gray-900 dark:text-white capitalize">
                                                            {block.block_type?.replace('_', ' ')}
                                                        </p>
                                                        <p className="text-sm text-gray-500 dark:text-gray-400">
                                                            {block.block_key}
                                                        </p>
                                                    </div>
                                                </div>
                                                <Badge variant={block.is_active ? "default" : "secondary"}>
                                                    {block.is_active ? 'Active' : 'Inactive'}
                                                </Badge>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <p className="text-gray-500 dark:text-gray-400 text-center py-4">
                                        No content blocks found
                                    </p>
                                )}
                            </CardContent>
                        </Card>
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Publishing Status */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <Calendar className="w-5 h-5 mr-2" />
                                    Publishing
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div>
                                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Status
                                    </label>
                                    <div className="mt-1">
                                        <Badge className={getStatusColor(page.status)}>
                                            {page.status}
                                        </Badge>
                                    </div>
                                </div>
                                <div>
                                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Published At
                                    </label>
                                    <p className="text-sm text-gray-900 dark:text-white">
                                        {formatDate(page.published_at)}
                                    </p>
                                </div>
                                <div>
                                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Created At
                                    </label>
                                    <p className="text-sm text-gray-900 dark:text-white">
                                        {formatDate(page.created_at)}
                                    </p>
                                </div>
                                <div>
                                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Updated At
                                    </label>
                                    <p className="text-sm text-gray-900 dark:text-white">
                                        {formatDate(page.updated_at)}
                                    </p>
                                </div>
                            </CardContent>
                        </Card>

                        {/* SEO Information */}
                        {page.seo && (
                            <Card>
                                <CardHeader>
                                    <CardTitle className="flex items-center">
                                        <Search className="w-5 h-5 mr-2" />
                                        SEO Information
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-3">
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Meta Title
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white truncate">
                                            {page.seo.meta_title || 'Not set'}
                                        </p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Meta Description
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white line-clamp-2">
                                            {page.seo.meta_description || 'Not set'}
                                        </p>
                                    </div>
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Canonical URL
                                        </label>
                                        <p className="text-sm text-gray-900 dark:text-white truncate">
                                            {page.seo.canonical_url || 'Not set'}
                                        </p>
                                    </div>
                                </CardContent>
                            </Card>
                        )}

                        {/* Media & Relationships */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center">
                                    <Image className="w-5 h-5 mr-2" />
                                    Media & Links
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Media Files
                                    </span>
                                    <Badge variant="secondary">
                                        {page.media?.length || 0}
                                    </Badge>
                                </div>
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Related Pages
                                    </span>
                                    <Badge variant="secondary">
                                        {page.relationships?.length || 0}
                                    </Badge>
                                </div>
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Child Pages
                                    </span>
                                    <Badge variant="secondary">
                                        {page.children?.length || 0}
                                    </Badge>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Actions */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Quick Actions</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <Link href={`/admin/cms/pages/${page.id}/edit`} className="w-full">
                                    <Button className="w-full">
                                        <Edit className="w-4 h-4 mr-2" />
                                        Edit Page
                                    </Button>
                                </Link>
                                <Link href={`/services/${page.slug}`} target="_blank" className="w-full">
                                    <Button variant="outline" className="w-full">
                                        <Eye className="w-4 h-4 mr-2" />
                                        View Page
                                    </Button>
                                </Link>
                                {page.seo && (
                                    <Link href={`/admin/cms/seo/${page.id}`} className="w-full">
                                        <Button variant="outline" className="w-full">
                                            <Search className="w-4 h-4 mr-2" />
                                            Edit SEO
                                        </Button>
                                    </Link>
                                )}
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}


