import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/Components/ui/Card';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import { Badge } from '@/Components/ui/Badge';
import { 
    Star, 
    Search, 
    Share, 
    Eye, 
    AlertCircle,
    CheckCircle,
    Info,
    RefreshCw,
    ExternalLink,
    Upload,
    X,
    Image as ImageIcon
} from 'lucide-react';

export default function SEOEditor({ seoData, onUpdate, pageData }) {
    const [seo, setSeo] = useState({
        meta_title: '',
        meta_description: '',
        meta_keywords: '',
        canonical_url: '',
        robots: 'index, follow',
        language: 'en',
        og_title: '',
        og_description: '',
        og_image: '',
        og_image_alt: '',
        og_type: 'website',
        og_url: '',
        og_site_name: 'Academic Scribe',
        og_locale: 'en_US',
        twitter_card: 'summary_large_image',
        twitter_site: '@academicscribe',
        twitter_creator: '@academicscribe',
        twitter_title: '',
        twitter_description: '',
        twitter_image: '',
        twitter_image_alt: '',
        structured_data: null,
        ...seoData
    });

    // Separate state for the textarea raw value to allow invalid JSON while typing
    const [structuredDataText, setStructuredDataText] = useState(() => {
        if (!seoData?.structured_data) return '';
        return typeof seoData.structured_data === 'string' 
            ? seoData.structured_data
            : JSON.stringify(seoData.structured_data, null, 2);
    });

    const [suggestions, setSuggestions] = useState([]);
    const [isGenerating, setIsGenerating] = useState(false);
    const [isUploadingImage, setIsUploadingImage] = useState(false);
    const [jsonError, setJsonError] = useState(null);

    useEffect(() => {
        generateSuggestions();
    }, [seo, pageData]);

    const handleUpdate = (field, value) => {
        console.log('SEOEditor handleUpdate:', { field, value });
        const updated = { ...seo, [field]: value };
        setSeo(updated);
        onUpdate(updated);
        console.log('SEOEditor updated state:', updated[field]);
    };

    const handleImageUpload = async (e) => {
        const file = e.target.files[0];
        if (!file) return;

        // Validate file type
        const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
        if (!validTypes.includes(file.type)) {
            alert('Please upload a valid image file (JPEG, PNG, GIF, or WebP)');
            return;
        }

        // Validate file size (5MB max)
        if (file.size > 5 * 1024 * 1024) {
            alert('Image size must be less than 5MB');
            return;
        }

        setIsUploadingImage(true);
        const formData = new FormData();
        formData.append('og_image', file);

        try {
            const response = await fetch('/admin/cms/seo/upload-image', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                },
                body: formData,
            });

            const result = await response.json();

            if (result.success) {
                handleUpdate('og_image', result.image_url);
                // Also update Twitter image if it's empty or same as OG image
                if (!seo.twitter_image || seo.twitter_image === seo.og_image) {
                    handleUpdate('twitter_image', result.image_url);
                }
            } else {
                alert('Failed to upload image: ' + result.message);
            }
        } catch (error) {
            console.error('Upload error:', error);
            alert('Failed to upload image. Please try again.');
        } finally {
            setIsUploadingImage(false);
            // Reset file input
            e.target.value = '';
        }
    };

    const generateSuggestions = () => {
        const newSuggestions = [];

        // Meta title suggestions
        if (!seo.meta_title) {
            newSuggestions.push({
                type: 'warning',
                message: 'Meta title is missing. This is crucial for SEO.',
                field: 'meta_title'
            });
        } else if (seo.meta_title.length < 30) {
            newSuggestions.push({
                type: 'info',
                message: 'Meta title could be longer (30-60 characters recommended).',
                field: 'meta_title'
            });
        } else if (seo.meta_title.length > 60) {
            newSuggestions.push({
                type: 'error',
                message: 'Meta title is too long (60 characters maximum).',
                field: 'meta_title'
            });
        }

        // Meta description suggestions
        if (!seo.meta_description) {
            newSuggestions.push({
                type: 'warning',
                message: 'Meta description is missing. This helps with click-through rates.',
                field: 'meta_description'
            });
        } else if (seo.meta_description.length < 120) {
            newSuggestions.push({
                type: 'info',
                message: 'Meta description could be longer (120-160 characters recommended).',
                field: 'meta_description'
            });
        } else if (seo.meta_description.length > 160) {
            newSuggestions.push({
                type: 'error',
                message: 'Meta description is too long (160 characters maximum).',
                field: 'meta_description'
            });
        }

        // Open Graph suggestions
        if (!seo.og_image) {
            newSuggestions.push({
                type: 'warning',
                message: 'Open Graph image is missing. This affects social media sharing.',
                field: 'og_image'
            });
        }

        // Structured data suggestions
        if (!seo.structured_data) {
            newSuggestions.push({
                type: 'info',
                message: 'Structured data is missing. This helps search engines understand your content.',
                field: 'structured_data'
            });
        }

        setSuggestions(newSuggestions);
    };

    const autoGenerateSEO = async () => {
        setIsGenerating(true);
        
        try {
            const response = await fetch(`/admin/cms/seo/${pageData.slug}/generate`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                },
                body: JSON.stringify({
                    title: pageData.title,
                    page_type: pageData.page_type,
                    content: pageData.content || ''
                })
            });

            const data = await response.json();
            
            if (data.success) {
                const updated = { ...seo, ...data.seo_data };
                setSeo(updated);
                onUpdate(updated);
                alert('SEO data generated successfully!');
            } else {
                alert('Failed to generate SEO data: ' + data.message);
            }
        } catch (error) {
            alert('Error generating SEO data: ' + error.message);
        } finally {
            setIsGenerating(false);
        }
    };

    const getSuggestionIcon = (type) => {
        switch (type) {
            case 'error': return <AlertCircle className="h-4 w-4 text-red-500" />;
            case 'warning': return <AlertCircle className="h-4 w-4 text-yellow-500" />;
            case 'info': return <Info className="h-4 w-4 text-blue-500" />;
            default: return <CheckCircle className="h-4 w-4 text-green-500" />;
        }
    };

    const getSuggestionColor = (type) => {
        switch (type) {
            case 'error': return 'border-red-200 bg-red-50';
            case 'warning': return 'border-yellow-200 bg-yellow-50';
            case 'info': return 'border-blue-200 bg-blue-50';
            default: return 'border-green-200 bg-green-50';
        }
    };

    return (
        <div className="space-y-6">
            {/* SEO Suggestions */}
            {suggestions.length > 0 && (
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center space-x-2">
                            <Star className="h-5 w-5" />
                            <span>SEO Suggestions</span>
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="space-y-3">
                            {suggestions.map((suggestion, index) => (
                                <div
                                    key={index}
                                    className={`flex items-start space-x-3 p-3 border rounded-lg ${getSuggestionColor(suggestion.type)}`}
                                >
                                    {getSuggestionIcon(suggestion.type)}
                                    <div className="flex-1">
                                        <p className="text-sm">{suggestion.message}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </CardContent>
                </Card>
            )}

            {/* Basic SEO */}
            <Card>
                <CardHeader>
                    <div className="flex justify-between items-center">
                        <CardTitle className="flex items-center space-x-2">
                            <Search className="h-5 w-5" />
                            <span>Basic SEO</span>
                        </CardTitle>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={autoGenerateSEO}
                            disabled={isGenerating}
                        >
                            <RefreshCw className={`h-4 w-4 mr-2 ${isGenerating ? 'animate-spin' : ''}`} />
                            {isGenerating ? 'Generating...' : 'Auto Generate'}
                        </Button>
                    </div>
                </CardHeader>
                <CardContent className="space-y-4">
                    <div>
                        <Label htmlFor="meta_title">Meta Title</Label>
                        <Input
                            id="meta_title"
                            value={seo.meta_title || ''}
                            onChange={(e) => handleUpdate('meta_title', e.target.value)}
                            placeholder="Enter meta title (30-60 characters)"
                        />
                        <div className="flex justify-between items-center mt-1">
                            <span className="text-xs text-gray-500">
                                {seo.meta_title.length} characters
                            </span>
                            <div className="flex space-x-1">
                                {seo.meta_title.length >= 30 && seo.meta_title.length <= 60 && (
                                    <Badge variant="default" className="text-xs">Good</Badge>
                                )}
                                {seo.meta_title.length < 30 && (
                                    <Badge variant="secondary" className="text-xs">Too Short</Badge>
                                )}
                                {seo.meta_title.length > 60 && (
                                    <Badge variant="destructive" className="text-xs">Too Long</Badge>
                                )}
                            </div>
                        </div>
                    </div>

                    <div>
                        <Label htmlFor="meta_description">Meta Description</Label>
                        <Textarea
                            id="meta_description"
                            value={seo.meta_description || ''}
                            onChange={(e) => handleUpdate('meta_description', e.target.value)}
                            placeholder="Enter meta description (120-160 characters)"
                            rows={3}
                        />
                        <div className="flex justify-between items-center mt-1">
                            <span className="text-xs text-gray-500">
                                {seo.meta_description.length} characters
                            </span>
                            <div className="flex space-x-1">
                                {seo.meta_description.length >= 120 && seo.meta_description.length <= 160 && (
                                    <Badge variant="default" className="text-xs">Good</Badge>
                                )}
                                {seo.meta_description.length < 120 && (
                                    <Badge variant="secondary" className="text-xs">Too Short</Badge>
                                )}
                                {seo.meta_description.length > 160 && (
                                    <Badge variant="destructive" className="text-xs">Too Long</Badge>
                                )}
                            </div>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="meta_keywords">Meta Keywords</Label>
                            <Input
                                id="meta_keywords"
                                value={seo.meta_keywords || ''}
                                onChange={(e) => handleUpdate('meta_keywords', e.target.value)}
                                placeholder="keyword1, keyword2, keyword3"
                            />
                        </div>
                        <div>
                            <Label htmlFor="canonical_url">Canonical URL</Label>
                            <Input
                                id="canonical_url"
                                value={seo.canonical_url || ''}
                                onChange={(e) => handleUpdate('canonical_url', e.target.value)}
                                placeholder="https://example.com/page"
                            />
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="robots">Robots</Label>
                            <select
                                id="robots"
                                value={seo.robots}
                                onChange={(e) => handleUpdate('robots', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="index, follow">Index, Follow</option>
                                <option value="index, nofollow">Index, No Follow</option>
                                <option value="noindex, follow">No Index, Follow</option>
                                <option value="noindex, nofollow">No Index, No Follow</option>
                            </select>
                        </div>
                        <div>
                            <Label htmlFor="language">Language</Label>
                            <select
                                id="language"
                                value={seo.language}
                                onChange={(e) => handleUpdate('language', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="en">English</option>
                                <option value="es">Spanish</option>
                                <option value="fr">French</option>
                                <option value="de">German</option>
                            </select>
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Open Graph */}
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                        <Share className="h-5 w-5" />
                        <span>Open Graph (Social Media)</span>
                    </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="og_title">OG Title</Label>
                            <Input
                                id="og_title"
                                value={seo.og_title || ''}
                                onChange={(e) => handleUpdate('og_title', e.target.value)}
                                placeholder="Leave empty to use meta title"
                            />
                        </div>
                        <div>
                            <Label htmlFor="og_type">OG Type</Label>
                            <select
                                id="og_type"
                                value={seo.og_type}
                                onChange={(e) => handleUpdate('og_type', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="website">Website</option>
                                <option value="article">Article</option>
                                <option value="product">Product</option>
                                <option value="service">Service</option>
                            </select>
                        </div>
                    </div>

                    <div>
                        <Label htmlFor="og_description">OG Description</Label>
                        <Textarea
                            id="og_description"
                            value={seo.og_description || ''}
                            onChange={(e) => handleUpdate('og_description', e.target.value)}
                            placeholder="Leave empty to use meta description"
                            rows={3}
                        />
                    </div>

                    <div>
                        <Label htmlFor="og_image">OG Image (Used for Twitter too)</Label>
                        <div className="space-y-3">
                            {/* Image Upload */}
                            <div className="flex items-center gap-3">
                                <label className="flex items-center justify-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                                    <Upload className="h-4 w-4 mr-2" />
                                    <span className="text-sm">{isUploadingImage ? 'Uploading...' : 'Upload Image'}</span>
                                    <input
                                        type="file"
                                        accept="image/jpeg,image/png,image/jpg,image/gif,image/webp"
                                        className="hidden"
                                        onChange={handleImageUpload}
                                        disabled={isUploadingImage}
                                    />
                                </label>
                                {seo.og_image && (
                                    <Button
                                        type="button"
                                        variant="outline"
                                        size="sm"
                                        onClick={() => handleUpdate('og_image', '')}
                                        className="text-red-600 hover:text-red-700"
                                    >
                                        <X className="h-4 w-4 mr-1" />
                                        Clear
                                    </Button>
                                )}
                            </div>
                            
                            {/* Image Preview */}
                            {seo.og_image && (
                                <div className="relative border border-gray-300 dark:border-gray-600 rounded-md p-3 bg-gray-50 dark:bg-gray-800">
                                    <img
                                        src={seo.og_image.startsWith('http') ? seo.og_image : `${window.location.origin}${seo.og_image}`}
                                        alt="OG Image Preview"
                                        className="max-w-full h-auto max-h-48 rounded-md"
                                        onError={(e) => {
                                            e.target.style.display = 'none';
                                        }}
                                    />
                                </div>
                            )}
                            
                            {/* URL Input (fallback) */}
                            <div>
                                <Input
                                    id="og_image"
                                    value={seo.og_image || ''}
                                    onChange={(e) => handleUpdate('og_image', e.target.value)}
                                    placeholder="Or enter image URL: https://example.com/image.jpg"
                                />
                                <p className="text-xs text-gray-500 mt-1">
                                    Recommended: 1200x630px. This image will be used for both Open Graph and Twitter Card.
                                </p>
                            </div>
                        </div>
                    </div>
                    <div>
                        <Label htmlFor="og_image_alt">OG Image Alt Text</Label>
                        <Input
                            id="og_image_alt"
                            value={seo.og_image_alt || ''}
                            onChange={(e) => handleUpdate('og_image_alt', e.target.value)}
                            placeholder="Describe the OG image for accessibility"
                        />
                    </div>
                    <div>
                        <Label htmlFor="og_url">OG URL</Label>
                        <Input
                            id="og_url"
                            value={seo.og_url || ''}
                            onChange={(e) => handleUpdate('og_url', e.target.value)}
                            placeholder="Leave empty to use canonical URL"
                        />
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="og_site_name">OG Site Name</Label>
                            <Input
                                id="og_site_name"
                                value={seo.og_site_name || ''}
                                onChange={(e) => handleUpdate('og_site_name', e.target.value)}
                            />
                        </div>
                        <div>
                            <Label htmlFor="og_locale">OG Locale</Label>
                            <select
                                id="og_locale"
                                value={seo.og_locale}
                                onChange={(e) => handleUpdate('og_locale', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="en_US">English (US)</option>
                                <option value="en_GB">English (UK)</option>
                                <option value="es_ES">Spanish</option>
                                <option value="fr_FR">French</option>
                                <option value="de_DE">German</option>
                            </select>
                        </div>
                    </div>
                </CardContent>
            </Card>

            {/* Twitter Cards */}
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                        <ExternalLink className="h-5 w-5" />
                        <span>Twitter Cards</span>
                    </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="twitter_card">Twitter Card Type</Label>
                            <select
                                id="twitter_card"
                                value={seo.twitter_card}
                                onChange={(e) => handleUpdate('twitter_card', e.target.value)}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            >
                                <option value="summary">Summary</option>
                                <option value="summary_large_image">Summary Large Image</option>
                                <option value="app">App</option>
                                <option value="player">Player</option>
                            </select>
                        </div>
                        <div>
                            <Label htmlFor="twitter_site">Twitter Site</Label>
                            <Input
                                id="twitter_site"
                                value={seo.twitter_site || ''}
                                onChange={(e) => handleUpdate('twitter_site', e.target.value)}
                                placeholder="@username"
                            />
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="twitter_creator">Twitter Creator</Label>
                            <Input
                                id="twitter_creator"
                                value={seo.twitter_creator || ''}
                                onChange={(e) => handleUpdate('twitter_creator', e.target.value)}
                                placeholder="@username"
                            />
                        </div>
                        <div>
                            <Label htmlFor="twitter_image">Twitter Image URL</Label>
                            <Input
                                id="twitter_image"
                                value={seo.twitter_image || ''}
                                onChange={(e) => handleUpdate('twitter_image', e.target.value)}
                                placeholder="Leave empty to use OG image"
                            />
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <Label htmlFor="twitter_title">Twitter Title</Label>
                            <Input
                                id="twitter_title"
                                value={seo.twitter_title || ''}
                                onChange={(e) => handleUpdate('twitter_title', e.target.value)}
                                placeholder="Leave empty to use OG title"
                            />
                        </div>
                        <div>
                            <Label htmlFor="twitter_image_alt">Twitter Image Alt Text</Label>
                            <Input
                                id="twitter_image_alt"
                                value={seo.twitter_image_alt || ''}
                                onChange={(e) => handleUpdate('twitter_image_alt', e.target.value)}
                                placeholder="Describe the image"
                            />
                        </div>
                    </div>

                    <div>
                        <Label htmlFor="twitter_description">Twitter Description</Label>
                        <Textarea
                            id="twitter_description"
                            value={seo.twitter_description || ''}
                            onChange={(e) => handleUpdate('twitter_description', e.target.value)}
                            placeholder="Leave empty to use OG description"
                            rows={3}
                        />
                    </div>
                </CardContent>
            </Card>

            {/* Structured Data Editor */}
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                        <Eye className="h-5 w-5" />
                        <span>Structured Data (JSON-LD Schema)</span>
                    </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                    <div>
                        <Label htmlFor="structured_data">
                            Structured Data JSON
                            <span className="ml-2 text-xs text-gray-500">
                                (Enter valid JSON or leave empty)
                            </span>
                        </Label>
                        <Textarea
                            id="structured_data"
                            value={structuredDataText}
                            onChange={(e) => {
                                const value = e.target.value;
                                setStructuredDataText(value);
                                
                                // Try to parse and validate
                                if (value.trim() === '') {
                                    setJsonError(null);
                                    handleUpdate('structured_data', null);
                                } else {
                                    try {
                                        const parsed = JSON.parse(value);
                                        setJsonError(null);
                                        handleUpdate('structured_data', parsed);
                                    } catch (error) {
                                        setJsonError(error.message);
                                        // Don't update the parent state with invalid JSON
                                        // This prevents submitting invalid data
                                    }
                                }
                            }}
                            placeholder={`{
  "@context": "https://schema.org",
  "@type": "Organization",
  "name": "Academic Scribe",
  "url": "https://academicscribe.com"
}`}
                            rows={15}
                            className={`font-mono text-sm ${jsonError ? 'border-red-500' : ''}`}
                        />
                        {jsonError && (
                            <div className="mt-2 text-sm text-red-600 bg-red-50 p-2 rounded">
                                <strong>JSON Error:</strong> {jsonError}
                            </div>
                        )}
                        <div className="flex items-start space-x-2 mt-2">
                            <Info className="h-4 w-4 text-blue-500 mt-0.5 flex-shrink-0" />
                            <div className="text-xs text-gray-600">
                                <p className="mb-2">
                                    Add Schema.org structured data to help search engines understand your content better.
                                    This can enable rich snippets in search results.
                                </p>
                                <p>
                                    <strong>Common schemas:</strong> Organization, Service, FAQPage, Article, BreadcrumbList
                                </p>
                                <a 
                                    href="https://schema.org/docs/schemas.html" 
                                    target="_blank" 
                                    rel="noopener noreferrer"
                                    className="text-blue-600 hover:underline inline-flex items-center mt-1"
                                >
                                    View Schema.org Documentation
                                    <ExternalLink className="h-3 w-3 ml-1" />
                                </a>
                            </div>
                        </div>
                    </div>

                    {/* Validation Status */}
                    {seo.structured_data && (
                        <div className="mt-3">
                            {(() => {
                                try {
                                    const parsed = typeof seo.structured_data === 'string' 
                                        ? JSON.parse(seo.structured_data)
                                        : seo.structured_data;
                                    
                                    return (
                                        <div className="flex items-center space-x-2 text-green-600 bg-green-50 border border-green-200 rounded-lg p-3">
                                            <CheckCircle className="h-4 w-4" />
                                            <span className="text-sm font-medium">Valid JSON Schema</span>
                                        </div>
                                    );
                                } catch (error) {
                                    return (
                                        <div className="flex items-center space-x-2 text-red-600 bg-red-50 border border-red-200 rounded-lg p-3">
                                            <AlertCircle className="h-4 w-4" />
                                            <span className="text-sm font-medium">Invalid JSON: {error.message}</span>
                                        </div>
                                    );
                                }
                            })()}
                        </div>
                    )}

                    {/* Preview */}
                    {seo.structured_data && (() => {
                        try {
                            const parsed = typeof seo.structured_data === 'string' 
                                ? JSON.parse(seo.structured_data)
                                : seo.structured_data;
                            
                            return (
                                <div className="mt-4">
                                    <Label>Formatted Preview</Label>
                                    <div className="bg-gray-50 p-4 rounded-lg border border-gray-200 mt-2">
                                        <pre className="text-xs overflow-x-auto text-gray-700">
                                            {JSON.stringify(parsed, null, 2)}
                                        </pre>
                                    </div>
                                </div>
                            );
                        } catch {
                            return null;
                        }
                    })()}
                </CardContent>
            </Card>
        </div>
    );
}
