import React, { useState, useEffect } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Save,
    ArrowLeft,
    Tag,
    Calendar,
    DollarSign,
    Percent,
    Hash,
    Users,
    Clock,
    Info,
    HelpCircle,
    AlertCircle,
    RefreshCw,
    Award,
    Gift,
    Zap,
    User,
    ShoppingBag,
    Package,
    CalendarDays,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import Tooltip from "@/Components/ui/Tooltip";
import { RadioGroup } from "@/Components/ui/RadioGroup";
import { formatDateForInput } from "@/lib/utils";

const Create = ({ couponTypes, loyaltyTiers, seasonalTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        coupon_code: "",
        discount_type: "percentage",
        discount_amount: "",
        coupon_type: "flash",
        loyalty_tier_id: "",
        seasonal_type_id: "",
        start_date: "",
        expiry_date: "",
        max_uses: "",
        per_user_limit: "1",
        min_order_amount: "",
        min_pages: "",
        is_first_order_only: false,
        user_specific: false,
        referrer_user_id: "",
        is_referral: false,
        is_active: true,
    });

    const [isGeneratingCode, setIsGeneratingCode] = useState(false);
    const [showAdvancedOptions, setShowAdvancedOptions] = useState(false);

    const generateRandomCode = async () => {
        setIsGeneratingCode(true);
        try {
            const response = await fetch(route("admin.coupons.generate-code"));
            const result = await response.json();

            if (result.code) {
                setData("coupon_code", result.code);
            }
        } catch (error) {
            console.error("Error generating coupon code:", error);
        } finally {
            setIsGeneratingCode(false);
        }
    };

    const onSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.store"), {
            onSuccess: () => {
                reset();
            },
        });
    };

    // Define icons for coupon types
    const couponTypeIcons = {
        flash: <Zap className="w-5 h-5" />,
        first_order: <ShoppingBag className="w-5 h-5" />,
        referral: <Users className="w-5 h-5" />,
        loyalty: <Award className="w-5 h-5" />,
        anniversary: <Gift className="w-5 h-5" />,
        seasonal: <CalendarDays className="w-5 h-5" />,
        volume: <Package className="w-5 h-5" />,
    };

    return (
        <AdminLayout>
            <Head title="Create Coupon" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-3 md:space-y-0 mb-6">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                Create New Coupon
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create a discount coupon for your customers
                            </p>
                        </div>
                        <div>
                            <Link
                                href={route("admin.coupons.index")}
                                className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md font-semibold text-xs text-gray-700 dark:text-gray-300 uppercase tracking-widest shadow-sm bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800 transition"
                            >
                                <ArrowLeft className="w-4 h-4 mr-2" />
                                Back to Coupons
                            </Link>
                        </div>
                    </div>

                    <form onSubmit={onSubmit}>
                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
                            <div className="p-6 border-b border-gray-200 dark:border-gray-700">
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                    Coupon Information
                                </h3>
                                <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                    Basic information about the coupon
                                </p>
                            </div>

                            <div className="p-6 space-y-6">
                                {/* Coupon Code */}
                                <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                    <div>
                                        <label
                                            htmlFor="coupon_code"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                        >
                                            Coupon Code
                                            <span className="text-red-500 ml-1">
                                                *
                                            </span>
                                        </label>
                                        <div className="mt-1 flex rounded-md shadow-sm">
                                            <div className="relative flex items-stretch flex-grow">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <Tag
                                                        className="h-5 w-5 text-gray-400"
                                                        aria-hidden="true"
                                                    />
                                                </div>
                                                <input
                                                    type="text"
                                                    id="coupon_code"
                                                    name="coupon_code"
                                                    value={data.coupon_code}
                                                    onChange={(e) =>
                                                        setData(
                                                            "coupon_code",
                                                            e.target.value.toUpperCase()
                                                        )
                                                    }
                                                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                    placeholder="SUMMER2023"
                                                />
                                            </div>
                                            <button
                                                type="button"
                                                onClick={generateRandomCode}
                                                className="ml-3 inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 shadow-sm text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800"
                                                disabled={isGeneratingCode}
                                            >
                                                {isGeneratingCode ? (
                                                    <RefreshCw className="h-4 w-4 animate-spin" />
                                                ) : (
                                                    <Hash className="h-4 w-4 mr-2" />
                                                )}
                                                Generate
                                            </button>
                                        </div>
                                        {errors.coupon_code && (
                                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                {errors.coupon_code}
                                            </p>
                                        )}
                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                            Code should be uppercase and unique.
                                            Use letters and numbers only.
                                        </p>
                                    </div>

                                    {/* Discount Type and Amount */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Discount Type & Amount
                                            <span className="text-red-500 ml-1">
                                                *
                                            </span>
                                        </label>
                                        <div className="mt-1 flex rounded-md shadow-sm">
                                            <div className="relative flex-1 flex items-stretch">
                                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    {data.discount_type ===
                                                    "percentage" ? (
                                                        <Percent
                                                            className="h-5 w-5 text-gray-400"
                                                            aria-hidden="true"
                                                        />
                                                    ) : (
                                                        <DollarSign
                                                            className="h-5 w-5 text-gray-400"
                                                            aria-hidden="true"
                                                        />
                                                    )}
                                                </div>
                                                <input
                                                    type="number"
                                                    id="discount_amount"
                                                    name="discount_amount"
                                                    min="0"
                                                    step={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "1"
                                                            : "0.01"
                                                    }
                                                    max={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "100"
                                                            : ""
                                                    }
                                                    value={data.discount_amount}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_amount",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-l-md"
                                                    placeholder={
                                                        data.discount_type ===
                                                        "percentage"
                                                            ? "15"
                                                            : "10.00"
                                                    }
                                                />
                                            </div>
                                            <div className="flex-shrink-0">
                                                <select
                                                    id="discount_type"
                                                    name="discount_type"
                                                    value={data.discount_type}
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_type",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="focus:ring-blue-500 focus:border-blue-500 h-full py-0 pl-3 pr-7 border-l-0 border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white bg-transparent text-gray-500 sm:text-sm rounded-r-md"
                                                >
                                                    <option value="percentage">
                                                        Percentage (%)
                                                    </option>
                                                    <option value="fixed">
                                                        Fixed Amount ($)
                                                    </option>
                                                </select>
                                            </div>
                                        </div>
                                        {errors.discount_amount && (
                                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                {errors.discount_amount}
                                            </p>
                                        )}
                                        {errors.discount_type && (
                                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                {errors.discount_type}
                                            </p>
                                        )}
                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                            {data.discount_type === "percentage"
                                                ? "Enter a percentage between 1-100%"
                                                : "Enter a fixed amount in dollars"}
                                        </p>
                                    </div>
                                </div>

                                {/* Coupon Type */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Coupon Type
                                        <span className="text-red-500 ml-1">
                                            *
                                        </span>
                                    </label>
                                    <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-3">
                                        {Object.entries(couponTypes).map(
                                            ([value, label]) => (
                                                <div
                                                    key={value}
                                                    onClick={() =>
                                                        setData(
                                                            "coupon_type",
                                                            value
                                                        )
                                                    }
                                                    className={`cursor-pointer relative flex items-center p-4 rounded-lg border-2 transition-colors ${
                                                        data.coupon_type ===
                                                        value
                                                            ? "border-blue-500 bg-blue-50 dark:border-blue-400 dark:bg-blue-900/30"
                                                            : "border-gray-200 bg-white hover:border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:hover:border-gray-600"
                                                    }`}
                                                >
                                                    <div
                                                        className={`mr-3 ${
                                                            data.coupon_type ===
                                                            value
                                                                ? "text-blue-600 dark:text-blue-400"
                                                                : "text-gray-500 dark:text-gray-400"
                                                        }`}
                                                    >
                                                        {couponTypeIcons[value]}
                                                    </div>
                                                    <div>
                                                        <span
                                                            className={`text-sm font-medium ${
                                                                data.coupon_type ===
                                                                value
                                                                    ? "text-blue-700 dark:text-blue-300"
                                                                    : "text-gray-700 dark:text-gray-300"
                                                            }`}
                                                        >
                                                            {label}
                                                        </span>
                                                    </div>
                                                </div>
                                            )
                                        )}
                                    </div>
                                    {errors.coupon_type && (
                                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                            {errors.coupon_type}
                                        </p>
                                    )}
                                </div>

                                {/* Conditional Fields Based on Coupon Type */}
                                {data.coupon_type === "loyalty" && (
                                    <div>
                                        <label
                                            htmlFor="loyalty_tier_id"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                        >
                                            Loyalty Tier
                                            <span className="text-red-500 ml-1">
                                                *
                                            </span>
                                        </label>
                                        <select
                                            id="loyalty_tier_id"
                                            name="loyalty_tier_id"
                                            value={data.loyalty_tier_id}
                                            onChange={(e) =>
                                                setData(
                                                    "loyalty_tier_id",
                                                    e.target.value
                                                )
                                            }
                                            className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                        >
                                            <option value="">
                                                Select a loyalty tier
                                            </option>
                                            {loyaltyTiers.map((tier) => (
                                                <option
                                                    key={tier.id}
                                                    value={tier.id}
                                                >
                                                    {tier.name} (Orders:{" "}
                                                    {tier.min_orders}+)
                                                </option>
                                            ))}
                                        </select>
                                        {errors.loyalty_tier_id && (
                                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                {errors.loyalty_tier_id}
                                            </p>
                                        )}
                                    </div>
                                )}

                                {data.coupon_type === "seasonal" && (
                                    <div>
                                        <label
                                            htmlFor="seasonal_type_id"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                        >
                                            Seasonal Event
                                            <span className="text-red-500 ml-1">
                                                *
                                            </span>
                                        </label>
                                        <select
                                            id="seasonal_type_id"
                                            name="seasonal_type_id"
                                            value={data.seasonal_type_id}
                                            onChange={(e) =>
                                                setData(
                                                    "seasonal_type_id",
                                                    e.target.value
                                                )
                                            }
                                            className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                        >
                                            <option value="">
                                                Select a seasonal event
                                            </option>
                                            {seasonalTypes.map((type) => (
                                                <option
                                                    key={type.id}
                                                    value={type.id}
                                                >
                                                    {type.name}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.seasonal_type_id && (
                                            <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                {errors.seasonal_type_id}
                                            </p>
                                        )}
                                    </div>
                                )}

                                {data.coupon_type === "referral" && (
                                    <div>
                                        <div className="flex items-start mb-4">
                                            <div className="flex items-center h-5">
                                                <input
                                                    id="is_referral"
                                                    name="is_referral"
                                                    type="checkbox"
                                                    checked={data.is_referral}
                                                    onChange={(e) =>
                                                        setData(
                                                            "is_referral",
                                                            e.target.checked
                                                        )
                                                    }
                                                    className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 dark:border-gray-600 dark:bg-gray-700 rounded"
                                                />
                                            </div>
                                            <div className="ml-3 text-sm">
                                                <label
                                                    htmlFor="is_referral"
                                                    className="font-medium text-gray-700 dark:text-gray-300"
                                                >
                                                    Is Referral Reward
                                                </label>
                                                <p className="text-gray-500 dark:text-gray-400">
                                                    Mark this coupon as a reward
                                                    for referring new customers
                                                </p>
                                            </div>
                                        </div>

                                        <div className="flex items-start mb-4">
                                            <div className="flex items-center h-5">
                                                <input
                                                    id="user_specific"
                                                    name="user_specific"
                                                    type="checkbox"
                                                    checked={data.user_specific}
                                                    onChange={(e) =>
                                                        setData(
                                                            "user_specific",
                                                            e.target.checked
                                                        )
                                                    }
                                                    className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 dark:border-gray-600 dark:bg-gray-700 rounded"
                                                />
                                            </div>
                                            <div className="ml-3 text-sm">
                                                <label
                                                    htmlFor="user_specific"
                                                    className="font-medium text-gray-700 dark:text-gray-300"
                                                >
                                                    User Specific
                                                </label>
                                                <p className="text-gray-500 dark:text-gray-400">
                                                    This coupon is specific to a
                                                    particular user
                                                </p>
                                            </div>
                                        </div>

                                        {data.user_specific && (
                                            <div>
                                                <label
                                                    htmlFor="referrer_user_id"
                                                    className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                                >
                                                    Referrer User ID
                                                    <span className="text-red-500 ml-1">
                                                        *
                                                    </span>
                                                </label>
                                                <div className="mt-1 flex rounded-md shadow-sm">
                                                    <div className="relative flex items-stretch flex-grow">
                                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                            <User
                                                                className="h-5 w-5 text-gray-400"
                                                                aria-hidden="true"
                                                            />
                                                        </div>
                                                        <input
                                                            type="text"
                                                            id="referrer_user_id"
                                                            name="referrer_user_id"
                                                            value={
                                                                data.referrer_user_id
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "referrer_user_id",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-10 sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                            placeholder="User ID"
                                                        />
                                                    </div>
                                                </div>
                                                {errors.referrer_user_id && (
                                                    <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                        {
                                                            errors.referrer_user_id
                                                        }
                                                    </p>
                                                )}
                                            </div>
                                        )}
                                    </div>
                                )}

                                {/* Date Restrictions */}
                                <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white flex items-center">
                                        <Calendar className="w-5 h-5 mr-2 text-gray-500 dark:text-gray-400" />
                                        Date Restrictions
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-600 dark:text-gray-400 mb-4">
                                        Set the validity period for this coupon
                                    </p>

                                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                        <div>
                                            <label
                                                htmlFor="start_date"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Start Date
                                            </label>
                                            <div className="mt-1">
                                                <input
                                                    type="datetime-local"
                                                    name="start_date"
                                                    id="start_date"
                                                    value={data.start_date}
                                                    onChange={(e) =>
                                                        setData(
                                                            "start_date",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                />
                                            </div>
                                            {errors.start_date && (
                                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave blank to activate
                                                immediately
                                            </p>
                                        </div>

                                        <div>
                                            <label
                                                htmlFor="expiry_date"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Expiry Date
                                            </label>
                                            <div className="mt-1">
                                                <input
                                                    type="datetime-local"
                                                    name="expiry_date"
                                                    id="expiry_date"
                                                    value={data.expiry_date}
                                                    onChange={(e) =>
                                                        setData(
                                                            "expiry_date",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                />
                                            </div>
                                            {errors.expiry_date && (
                                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave blank for no expiration
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                {/* Usage Limits */}
                                <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-white flex items-center">
                                        <Clock className="w-5 h-5 mr-2 text-gray-500 dark:text-gray-400" />
                                        Usage Limits
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-600 dark:text-gray-400 mb-4">
                                        Set limits on how this coupon can be
                                        used
                                    </p>

                                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                        <div>
                                            <label
                                                htmlFor="max_uses"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Maximum Uses
                                            </label>
                                            <div className="mt-1">
                                                <input
                                                    type="number"
                                                    name="max_uses"
                                                    id="max_uses"
                                                    min="0"
                                                    value={data.max_uses}
                                                    onChange={(e) =>
                                                        setData(
                                                            "max_uses",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                    placeholder="Unlimited"
                                                />
                                            </div>
                                            {errors.max_uses && (
                                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave blank for unlimited uses
                                            </p>
                                        </div>

                                        <div>
                                            <label
                                                htmlFor="per_user_limit"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Uses Per User
                                            </label>
                                            <div className="mt-1">
                                                <input
                                                    type="number"
                                                    name="per_user_limit"
                                                    id="per_user_limit"
                                                    min="0"
                                                    value={data.per_user_limit}
                                                    onChange={(e) =>
                                                        setData(
                                                            "per_user_limit",
                                                            e.target.value
                                                        )
                                                    }
                                                    className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                    placeholder="1"
                                                />
                                            </div>
                                            {errors.per_user_limit && (
                                                <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                    {errors.per_user_limit}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                How many times a single user can
                                                use this coupon
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                {/* Advanced Options */}
                                <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                    <button
                                        type="button"
                                        className="text-sm font-medium text-blue-600 dark:text-blue-400 flex items-center"
                                        onClick={() =>
                                            setShowAdvancedOptions(
                                                !showAdvancedOptions
                                            )
                                        }
                                    >
                                        <Info className="w-5 h-5 mr-2" />
                                        {showAdvancedOptions
                                            ? "Hide Advanced Options"
                                            : "Show Advanced Options"}
                                    </button>

                                    {showAdvancedOptions && (
                                        <div className="mt-4 space-y-6">
                                            <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                                                <div>
                                                    <label
                                                        htmlFor="min_order_amount"
                                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                                    >
                                                        Minimum Order Amount
                                                    </label>
                                                    <div className="mt-1 relative rounded-md shadow-sm">
                                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                            <span className="text-gray-500 dark:text-gray-400 sm:text-sm">
                                                                $
                                                            </span>
                                                        </div>
                                                        <input
                                                            type="number"
                                                            name="min_order_amount"
                                                            id="min_order_amount"
                                                            min="0"
                                                            step="0.01"
                                                            value={
                                                                data.min_order_amount
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "min_order_amount",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                            placeholder="0.00"
                                                        />
                                                    </div>
                                                    {errors.min_order_amount && (
                                                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                            {
                                                                errors.min_order_amount
                                                            }
                                                        </p>
                                                    )}
                                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                        Minimum order amount
                                                        required to use this
                                                        coupon
                                                    </p>
                                                </div>

                                                <div>
                                                    <label
                                                        htmlFor="min_pages"
                                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                                    >
                                                        Minimum Pages
                                                    </label>
                                                    <div className="mt-1">
                                                        <input
                                                            type="number"
                                                            name="min_pages"
                                                            id="min_pages"
                                                            min="0"
                                                            value={
                                                                data.min_pages
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "min_pages",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-md"
                                                            placeholder="0"
                                                        />
                                                    </div>
                                                    {errors.min_pages && (
                                                        <p className="mt-2 text-sm text-red-600 dark:text-red-400">
                                                            {errors.min_pages}
                                                        </p>
                                                    )}
                                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                        Minimum number of pages
                                                        required in the order
                                                    </p>
                                                </div>
                                            </div>

                                            <div className="flex items-start">
                                                <div className="flex items-center h-5">
                                                    <input
                                                        id="is_first_order_only"
                                                        name="is_first_order_only"
                                                        type="checkbox"
                                                        checked={
                                                            data.is_first_order_only
                                                        }
                                                        onChange={(e) =>
                                                            setData(
                                                                "is_first_order_only",
                                                                e.target.checked
                                                            )
                                                        }
                                                        className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 dark:border-gray-600 dark:bg-gray-700 rounded"
                                                    />
                                                </div>
                                                <div className="ml-3 text-sm">
                                                    <label
                                                        htmlFor="is_first_order_only"
                                                        className="font-medium text-gray-700 dark:text-gray-300"
                                                    >
                                                        First Order Only
                                                    </label>
                                                    <p className="text-gray-500 dark:text-gray-400">
                                                        This coupon can only be
                                                        used by customers making
                                                        their first purchase
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </div>

                                {/* Status */}
                                <div className="border-t border-gray-200 dark:border-gray-700 pt-6">
                                    <div className="flex items-start">
                                        <div className="flex items-center h-5">
                                            <input
                                                id="is_active"
                                                name="is_active"
                                                type="checkbox"
                                                checked={data.is_active}
                                                onChange={(e) =>
                                                    setData(
                                                        "is_active",
                                                        e.target.checked
                                                    )
                                                }
                                                className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 dark:border-gray-600 dark:bg-gray-700 rounded"
                                            />
                                        </div>
                                        <div className="ml-3 text-sm">
                                            <label
                                                htmlFor="is_active"
                                                className="font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Active
                                            </label>
                                            <p className="text-gray-500 dark:text-gray-400">
                                                Enable or disable this coupon.
                                                Inactive coupons cannot be used.
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Form Actions */}
                            <div className="px-6 py-4 bg-gray-50 dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 flex items-center justify-end space-x-3">
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md font-semibold text-xs text-gray-700 dark:text-gray-300 uppercase tracking-widest shadow-sm bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800 transition"
                                >
                                    Cancel
                                </Link>
                                <Button
                                    type="submit"
                                    className="inline-flex items-center"
                                    disabled={processing}
                                >
                                    {processing ? (
                                        <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                                    ) : (
                                        <Save className="h-4 w-4 mr-2" />
                                    )}
                                    Save Coupon
                                </Button>
                            </div>
                        </div>
                    </form>

                    {/* Help Section */}
                    <div className="mt-6 bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                        <div className="px-6 py-4">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white flex items-center">
                                <HelpCircle className="w-5 h-5 mr-2 text-blue-500" />
                                Tips for Creating Effective Coupons
                            </h3>
                            <div className="mt-4 text-sm text-gray-600 dark:text-gray-400 space-y-3">
                                <p>
                                    <strong>Flash Sales:</strong> Create
                                    time-sensitive promotions with short expiry
                                    dates to create urgency.
                                </p>
                                <p>
                                    <strong>First Order Coupons:</strong> Offer
                                    attractive discounts to new customers to
                                    encourage the first purchase.
                                </p>
                                <p>
                                    <strong>Loyalty Rewards:</strong> Reward
                                    your most loyal customers with exclusive
                                    discounts based on their order count.
                                </p>
                                <p>
                                    <strong>Seasonal Promotions:</strong> Align
                                    promotions with holidays and seasons to
                                    boost sales during specific periods.
                                </p>
                                <p>
                                    <strong>Volume Discounts:</strong> Encourage
                                    larger orders by offering discounts based on
                                    the order size.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Create;
