import React, { useState } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/Select";

import {
    Award,
    ArrowLeft,
    Calendar,
    Tag,
    Info,
    Plus,
    Minus,
    DollarSign,
    Percent,
} from "lucide-react";

import Alert from "@/Components/ui/Alert";
import { Badge } from "@/Components/ui/Badge";

const GenerateLoyalty = ({ loyaltyTiers, discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        tier_discounts: {},
        prefix: "LOYALTY",
        start_date: null,
        expiry_date: null,
        max_uses: "",
    });

    const [showAllTiers, setShowAllTiers] = useState(false);

    // Initialize tier discounts
    const initializeTierDiscounts = () => {
        const initialDiscounts = {};
        loyaltyTiers.forEach((tier) => {
            initialDiscounts[tier.id] = "";
        });
        setData("tier_discounts", initialDiscounts);
    };

    // Initialize tier discounts on component mount
    React.useEffect(() => {
        initializeTierDiscounts();
    }, [loyaltyTiers]);

    const handleTierDiscountChange = (tierId, value) => {
        const newDiscounts = { ...data.tier_discounts };
        newDiscounts[tierId] = value;
        setData("tier_discounts", newDiscounts);
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.coupons.generate.loyalty"));
    };

    // Helper function to apply the same discount to all tiers
    const applyToAllTiers = (value) => {
        const newDiscounts = { ...data.tier_discounts };
        loyaltyTiers.forEach((tier) => {
            newDiscounts[tier.id] = value;
        });
        setData("tier_discounts", newDiscounts);
    };

    // Filter active tiers for display
    const displayTiers = showAllTiers ? loyaltyTiers : loyaltyTiers.slice(0, 3);

    return (
        <AdminLayout>
            <Head title="Generate Loyalty Coupons" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Award className="w-6 h-6 mr-2 text-blue-500" />
                                Generate Loyalty Coupons
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create coupon codes for your loyalty tier
                                members
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <CardTitle>
                                        Loyalty Tier Discounts
                                    </CardTitle>
                                    <CardDescription>
                                        Configure discount amount for each
                                        loyalty tier
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                        </div>

                                        {/* Apply to All Tiers */}
                                        <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-800/50 rounded-md">
                                            <div className="flex items-center justify-between">
                                                <Label htmlFor="apply_all">
                                                    Apply same discount to all
                                                    tiers
                                                </Label>
                                                <div className="flex items-center">
                                                    <Input
                                                        id="apply_all"
                                                        type="number"
                                                        className="w-24 mr-2"
                                                        placeholder="Value"
                                                        min="0"
                                                        step={
                                                            data.discount_type ===
                                                            "percentage"
                                                                ? "0.1"
                                                                : "0.01"
                                                        }
                                                    />
                                                    <Button
                                                        type="button"
                                                        onClick={(e) =>
                                                            applyToAllTiers(
                                                                e.target
                                                                    .previousSibling
                                                                    .value
                                                            )
                                                        }
                                                        size="sm"
                                                    >
                                                        Apply
                                                    </Button>
                                                </div>
                                            </div>
                                        </div>

                                        {/* Tier Specific Discounts */}
                                        <div className="space-y-6">
                                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                                {displayTiers.map((tier) => (
                                                    <div
                                                        key={tier.id}
                                                        className="p-4 border border-gray-200 dark:border-gray-700 rounded-md"
                                                    >
                                                        <div className="flex justify-between items-start mb-3">
                                                            <div>
                                                                <h4 className="font-medium text-gray-900 dark:text-white flex items-center">
                                                                    <Award className="h-4 w-4 mr-1 text-blue-500" />
                                                                    {tier.name}
                                                                </h4>
                                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                    Min.{" "}
                                                                    {
                                                                        tier.min_orders
                                                                    }{" "}
                                                                    orders
                                                                </p>
                                                            </div>
                                                            <Badge variant="outline">
                                                                {tier.is_active
                                                                    ? "Active"
                                                                    : "Inactive"}
                                                            </Badge>
                                                        </div>
                                                        <div className="mt-2">
                                                            <Label
                                                                htmlFor={`tier_${tier.id}`}
                                                                className="text-sm"
                                                            >
                                                                Discount Amount
                                                            </Label>
                                                            <div className="mt-1 relative">
                                                                <Input
                                                                    id={`tier_${tier.id}`}
                                                                    type="number"
                                                                    min="0"
                                                                    step={
                                                                        data.discount_type ===
                                                                        "percentage"
                                                                            ? "0.1"
                                                                            : "0.01"
                                                                    }
                                                                    value={
                                                                        data
                                                                            .tier_discounts[
                                                                            tier
                                                                                .id
                                                                        ] || ""
                                                                    }
                                                                    onChange={(
                                                                        e
                                                                    ) =>
                                                                        handleTierDiscountChange(
                                                                            tier.id,
                                                                            e
                                                                                .target
                                                                                .value
                                                                        )
                                                                    }
                                                                    className="pr-8"
                                                                    placeholder={
                                                                        data.discount_type ===
                                                                        "percentage"
                                                                            ? "15.0"
                                                                            : "10.00"
                                                                    }
                                                                />
                                                                <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                                                    {data.discount_type ===
                                                                    "percentage" ? (
                                                                        <Percent className="h-4 w-4 text-gray-400" />
                                                                    ) : (
                                                                        <DollarSign className="h-4 w-4 text-gray-400" />
                                                                    )}
                                                                </div>
                                                            </div>
                                                            {errors[
                                                                `tier_discounts.${tier.id}`
                                                            ] && (
                                                                <p className="mt-1 text-sm text-red-600">
                                                                    {
                                                                        errors[
                                                                            `tier_discounts.${tier.id}`
                                                                        ]
                                                                    }
                                                                </p>
                                                            )}
                                                        </div>
                                                    </div>
                                                ))}
                                            </div>

                                            {/* Show/Hide Tiers Toggle */}
                                            {loyaltyTiers.length > 3 && (
                                                <div className="flex justify-center">
                                                    <Button
                                                        type="button"
                                                        variant="outline"
                                                        size="sm"
                                                        onClick={() =>
                                                            setShowAllTiers(
                                                                !showAllTiers
                                                            )
                                                        }
                                                    >
                                                        {showAllTiers ? (
                                                            <>
                                                                <Minus className="w-4 h-4 mr-2" />
                                                                Show Less
                                                            </>
                                                        ) : (
                                                            <>
                                                                <Plus className="w-4 h-4 mr-2" />
                                                                Show All{" "}
                                                                {
                                                                    loyaltyTiers.length
                                                                }{" "}
                                                                Tiers
                                                            </>
                                                        )}
                                                    </Button>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="LOYALTY"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Optional prefix for generated
                                                codes (max 10 chars)
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="start_date">
                                                Valid From
                                            </Label>
                                            {/* <DatePicker
                                                id="start_date"
                                                className="mt-1"
                                                date={data.start_date}
                                                onSelect={(date) =>
                                                    setData("start_date", date)
                                                }
                                                placeholder="Select start date..."
                                            /> */}

                                            <Input
                                                id="start_date"
                                                type="date"
                                                className="mt-1"
                                                value={data.start_date}
                                                onChange={(e) =>
                                                    setData(
                                                        "start_date",
                                                        e.target.value
                                                    )
                                                }
                                            />
                                            {errors.start_date && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="expiry_date">
                                                Expiry Date
                                            </Label>

                                            <Input
                                                id="expiry_date"
                                                type="date"
                                                className="mt-1"
                                                value={data.expiry_date}
                                                onChange={(e) =>
                                                    setData(
                                                        "expiry_date",
                                                        e.target.value
                                                    )
                                                }
                                            />
                                            {errors.expiry_date && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="max_uses">
                                                Maximum Uses
                                            </Label>
                                            <Input
                                                id="max_uses"
                                                type="number"
                                                className="mt-1"
                                                min="1"
                                                value={data.max_uses}
                                                onChange={(e) =>
                                                    setData(
                                                        "max_uses",
                                                        e.target.value
                                                    )
                                                }
                                                placeholder="Unlimited"
                                            />
                                            {errors.max_uses && (
                                                <p className="mt-1 text-sm text-red-600">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Leave empty for unlimited uses
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button type="submit" disabled={processing}>
                                        {processing
                                            ? "Generating..."
                                            : "Generate Coupons"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="How Loyalty Coupons Work"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Loyalty coupons are automatically available
                                    to users who meet the tier's order
                                    requirements. These coupons will:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        Be automatically suggested to eligible
                                        users when placing orders
                                    </li>
                                    <li>
                                        Only be usable by customers with the
                                        required number of previous orders
                                    </li>
                                    <li>
                                        Include the tier name as part of the
                                        coupon code for easy identification
                                    </li>
                                </ul>
                            </div>
                        </Alert>
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateLoyalty;
