import React, { useState } from "react";
import { Head, useForm, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Button } from "@/Components/ui/Button";
import { DatePicker } from "@/Components/ui/DatePicker";
import {
    ArrowLeft,
    Calendar,
    Info,
    DollarSign,
    Percent,
    Plus,
    Minus,
    BarChart,
    Layers,
    FileText,
    AlertTriangle,
} from "lucide-react";
import { Separator } from "@/Components/ui/Separator";
import Alert from "@/Components/ui/Alert";
import { Badge } from "@/Components/ui/Badge";

const GenerateVolume = ({ discountTypes }) => {
    const { data, setData, post, processing, errors, reset } = useForm({
        discount_type: "percentage",
        volume_thresholds: [
            { pages: "5", discount: "10" },
            { pages: "10", discount: "15" },
            { pages: "20", discount: "20" },
        ],
        prefix: "VOL",
        start_date: null,
        expiry_date: null,
        max_uses: "",
    });

    const [validationErrors, setValidationErrors] = useState({});

    const handleSubmit = (e) => {
        e.preventDefault();

        // Local validation
        const errors = {};
        const pageValues = new Set();

        // Check for duplicate page thresholds
        data.volume_thresholds.forEach((threshold, index) => {
            if (pageValues.has(threshold.pages)) {
                errors[`volume_thresholds.${index}.pages`] =
                    "Duplicate page threshold";
            } else {
                pageValues.add(threshold.pages);
            }

            // Validate discount amounts
            if (
                data.discount_type === "percentage" &&
                parseFloat(threshold.discount) > 100
            ) {
                errors[`volume_thresholds.${index}.discount`] =
                    "Percentage cannot exceed 100%";
            }
        });

        setValidationErrors(errors);

        // Only proceed if no validation errors
        if (Object.keys(errors).length === 0) {
            post(route("admin.coupons.generate.volume"), {
                onSuccess: () => {
                    reset();
                },
            });
        }
    };

    const addThreshold = () => {
        const lastThreshold =
            data.volume_thresholds[data.volume_thresholds.length - 1];
        const newPages = lastThreshold ? parseInt(lastThreshold.pages) * 2 : 5;
        const newDiscount = lastThreshold
            ? Math.min(
                  parseFloat(lastThreshold.discount) + 5,
                  data.discount_type === "percentage" ? 50 : 50
              )
            : 10;

        setData("volume_thresholds", [
            ...data.volume_thresholds,
            { pages: newPages.toString(), discount: newDiscount.toString() },
        ]);
    };

    const removeThreshold = (index) => {
        const newThresholds = [...data.volume_thresholds];
        newThresholds.splice(index, 1);
        setData("volume_thresholds", newThresholds);
    };

    const updateThreshold = (index, field, value) => {
        const newThresholds = [...data.volume_thresholds];
        newThresholds[index][field] = value;
        setData("volume_thresholds", newThresholds);
    };

    return (
        <AdminLayout>
            <Head title="Generate Volume Discount Coupons" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Layers className="w-6 h-6 mr-2 text-cyan-500" />
                                Generate Volume Discount Coupons
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create coupons that offer larger discounts for
                                higher page counts
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.coupons.index")}
                                    className="inline-flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Coupons
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <form onSubmit={handleSubmit}>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <Card className="md:col-span-2">
                                <CardHeader>
                                    <div className="flex justify-between items-center">
                                        <div>
                                            <CardTitle>
                                                Volume Discount Thresholds
                                            </CardTitle>
                                            <CardDescription>
                                                Set up discount tiers based on
                                                order page count
                                            </CardDescription>
                                        </div>
                                        <Button
                                            type="button"
                                            variant="outline"
                                            size="sm"
                                            onClick={addThreshold}
                                        >
                                            <Plus className="w-4 h-4 mr-2" />
                                            Add Tier
                                        </Button>
                                    </div>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-6">
                                        {/* Discount Type Selection */}
                                        <div className="mb-6">
                                            <Label htmlFor="discount_type">
                                                Discount Type
                                            </Label>
                                            <div className="mt-1 flex gap-4">
                                                {Object.entries(
                                                    discountTypes
                                                ).map(([key, label]) => (
                                                    <Label
                                                        key={key}
                                                        htmlFor={`discount_type_${key}`}
                                                        className="flex items-center space-x-2 cursor-pointer"
                                                    >
                                                        <input
                                                            type="radio"
                                                            id={`discount_type_${key}`}
                                                            name="discount_type"
                                                            value={key}
                                                            checked={
                                                                data.discount_type ===
                                                                key
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "discount_type",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                                                        />
                                                        <span className="flex items-center">
                                                            {key ===
                                                            "percentage" ? (
                                                                <Percent className="h-4 w-4 mr-1 text-green-600" />
                                                            ) : (
                                                                <DollarSign className="h-4 w-4 mr-1 text-green-600" />
                                                            )}
                                                            {label}
                                                        </span>
                                                    </Label>
                                                ))}
                                            </div>
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                This applies to all volume
                                                discount tiers
                                            </p>
                                        </div>

                                        {/* Volume Threshold Table */}
                                        <div className="overflow-hidden border border-gray-200 dark:border-gray-700 sm:rounded-lg">
                                            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                <thead className="bg-gray-50 dark:bg-gray-800">
                                                    <tr>
                                                        <th
                                                            scope="col"
                                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider"
                                                        >
                                                            Minimum Pages
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider"
                                                        >
                                                            Discount Amount
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-6 py-3 text-right text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider"
                                                        >
                                                            Actions
                                                        </th>
                                                    </tr>
                                                </thead>
                                                <tbody className="bg-white dark:bg-gray-900 divide-y divide-gray-200 dark:divide-gray-700">
                                                    {data.volume_thresholds.map(
                                                        (threshold, index) => (
                                                            <tr
                                                                key={index}
                                                                className={
                                                                    index %
                                                                        2 ===
                                                                    0
                                                                        ? "bg-white dark:bg-gray-900"
                                                                        : "bg-gray-50 dark:bg-gray-800/50"
                                                                }
                                                            >
                                                                <td className="px-6 py-4 whitespace-nowrap">
                                                                    <div className="relative">
                                                                        <Input
                                                                            type="number"
                                                                            min="1"
                                                                            value={
                                                                                threshold.pages
                                                                            }
                                                                            onChange={(
                                                                                e
                                                                            ) =>
                                                                                updateThreshold(
                                                                                    index,
                                                                                    "pages",
                                                                                    e
                                                                                        .target
                                                                                        .value
                                                                                )
                                                                            }
                                                                            required
                                                                            className={
                                                                                validationErrors[
                                                                                    `volume_thresholds.${index}.pages`
                                                                                ]
                                                                                    ? "border-red-500"
                                                                                    : ""
                                                                            }
                                                                        />
                                                                        <div className="flex items-center absolute right-2 top-1/2 transform -translate-y-1/2 pointer-events-none">
                                                                            <FileText className="h-4 w-4 text-gray-400" />
                                                                        </div>
                                                                        {validationErrors[
                                                                            `volume_thresholds.${index}.pages`
                                                                        ] && (
                                                                            <p className="mt-1 text-xs text-red-600">
                                                                                {
                                                                                    validationErrors[
                                                                                        `volume_thresholds.${index}.pages`
                                                                                    ]
                                                                                }
                                                                            </p>
                                                                        )}
                                                                    </div>
                                                                </td>
                                                                <td className="px-6 py-4 whitespace-nowrap">
                                                                    <div className="relative">
                                                                        <Input
                                                                            type="number"
                                                                            min="0.01"
                                                                            step={
                                                                                data.discount_type ===
                                                                                "percentage"
                                                                                    ? "1"
                                                                                    : "0.01"
                                                                            }
                                                                            max={
                                                                                data.discount_type ===
                                                                                "percentage"
                                                                                    ? "100"
                                                                                    : undefined
                                                                            }
                                                                            value={
                                                                                threshold.discount
                                                                            }
                                                                            onChange={(
                                                                                e
                                                                            ) =>
                                                                                updateThreshold(
                                                                                    index,
                                                                                    "discount",
                                                                                    e
                                                                                        .target
                                                                                        .value
                                                                                )
                                                                            }
                                                                            required
                                                                            className={
                                                                                validationErrors[
                                                                                    `volume_thresholds.${index}.discount`
                                                                                ]
                                                                                    ? "border-red-500"
                                                                                    : ""
                                                                            }
                                                                        />
                                                                        <div className="flex items-center absolute right-2 top-1/2 transform -translate-y-1/2 pointer-events-none">
                                                                            {data.discount_type ===
                                                                            "percentage" ? (
                                                                                <Percent className="h-4 w-4 text-gray-400" />
                                                                            ) : (
                                                                                <DollarSign className="h-4 w-4 text-gray-400" />
                                                                            )}
                                                                        </div>
                                                                        {validationErrors[
                                                                            `volume_thresholds.${index}.discount`
                                                                        ] && (
                                                                            <p className="mt-1 text-xs text-red-600">
                                                                                {
                                                                                    validationErrors[
                                                                                        `volume_thresholds.${index}.discount`
                                                                                    ]
                                                                                }
                                                                            </p>
                                                                        )}
                                                                    </div>
                                                                </td>
                                                                <td className="px-6 py-4 whitespace-nowrap text-right">
                                                                    <Button
                                                                        type="button"
                                                                        variant="ghost"
                                                                        size="sm"
                                                                        disabled={
                                                                            data
                                                                                .volume_thresholds
                                                                                .length <=
                                                                            1
                                                                        }
                                                                        onClick={() =>
                                                                            removeThreshold(
                                                                                index
                                                                            )
                                                                        }
                                                                        className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                                                                    >
                                                                        <Minus className="w-4 h-4" />
                                                                    </Button>
                                                                </td>
                                                            </tr>
                                                        )
                                                    )}
                                                </tbody>
                                            </table>
                                        </div>

                                        {/* Preview */}
                                        <div className="mt-6">
                                            <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 flex items-center mb-3">
                                                <BarChart className="w-4 h-4 mr-2 text-blue-500" />
                                                Discount Preview
                                            </h3>
                                            <div className="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg">
                                                <div className="space-y-2">
                                                    {data.volume_thresholds.map(
                                                        (threshold, index) => (
                                                            <div
                                                                key={index}
                                                                className="flex justify-between items-center"
                                                            >
                                                                <span className="flex items-center">
                                                                    <FileText className="w-4 h-4 mr-2 text-gray-600 dark:text-gray-400" />
                                                                    <span className="text-sm">
                                                                        {
                                                                            threshold.pages
                                                                        }
                                                                        + pages
                                                                    </span>
                                                                    <Badge className="ml-2 bg-cyan-100 text-cyan-800 dark:bg-cyan-900/30 dark:text-cyan-300">
                                                                        Code:{" "}
                                                                        {
                                                                            data.prefix
                                                                        }
                                                                        {
                                                                            threshold.pages
                                                                        }
                                                                    </Badge>
                                                                </span>
                                                                <span className="font-medium text-green-600 dark:text-green-400">
                                                                    {data.discount_type ===
                                                                    "percentage"
                                                                        ? `${threshold.discount}% off`
                                                                        : `$${parseFloat(
                                                                              threshold.discount
                                                                          ).toFixed(
                                                                              2
                                                                          )} off`}
                                                                </span>
                                                            </div>
                                                        )
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Coupon Settings</CardTitle>
                                    <CardDescription>
                                        Configure general coupon parameters
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        <div>
                                            <Label htmlFor="prefix">
                                                Code Prefix
                                            </Label>
                                            <Input
                                                id="prefix"
                                                className="mt-1"
                                                value={data.prefix}
                                                onChange={(e) =>
                                                    setData(
                                                        "prefix",
                                                        e.target.value
                                                    )
                                                }
                                                maxLength={10}
                                                placeholder="VOL"
                                            />
                                            {errors.prefix && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.prefix}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                Each coupon code will be{" "}
                                                {data.prefix}[PAGE_COUNT]_XXXX
                                            </p>
                                        </div>

                                        <div>
                                            <Label htmlFor="start_date">
                                                Valid From
                                            </Label>
                                            <DatePicker
                                                id="start_date"
                                                className="mt-1"
                                                date={data.start_date}
                                                onSelect={(date) =>
                                                    setData("start_date", date)
                                                }
                                                placeholder="Select start date..."
                                            />
                                            {errors.start_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.start_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="expiry_date">
                                                Expiry Date
                                            </Label>
                                            <DatePicker
                                                id="expiry_date"
                                                className="mt-1"
                                                date={data.expiry_date}
                                                onSelect={(date) =>
                                                    setData("expiry_date", date)
                                                }
                                                placeholder="Select expiry date..."
                                            />
                                            {errors.expiry_date && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.expiry_date}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <Label htmlFor="max_uses">
                                                Maximum Uses Per Coupon
                                            </Label>
                                            <Input
                                                id="max_uses"
                                                type="number"
                                                className="mt-1"
                                                min="1"
                                                value={data.max_uses}
                                                onChange={(e) =>
                                                    setData(
                                                        "max_uses",
                                                        e.target.value
                                                    )
                                                }
                                                placeholder="Leave empty for unlimited"
                                            />
                                            {errors.max_uses && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.max_uses}
                                                </p>
                                            )}
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                This limit applies to each tier
                                                separately
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                                <CardFooter className="flex justify-end">
                                    <Button
                                        type="submit"
                                        disabled={
                                            processing ||
                                            data.volume_thresholds.length === 0
                                        }
                                    >
                                        {processing
                                            ? "Generating..."
                                            : "Generate Volume Coupons"}
                                    </Button>
                                </CardFooter>
                            </Card>
                        </div>

                        {/* Info alert */}
                        <Alert
                            variant="info"
                            title="How Volume Discounts Work"
                            className="mt-6"
                        >
                            <div className="text-sm">
                                <p className="mb-2">
                                    Volume discounts encourage customers to
                                    place larger orders by offering better rates
                                    for higher page counts:
                                </p>
                                <ul className="list-disc pl-5 space-y-1">
                                    <li>
                                        Each threshold creates a separate coupon
                                        code
                                    </li>
                                    <li>
                                        Customers will only be able to use
                                        coupons if their order meets or exceeds
                                        the page threshold
                                    </li>
                                    <li>
                                        The system will automatically suggest
                                        the highest applicable discount to
                                        customers during checkout
                                    </li>
                                    <li>
                                        Consider setting higher discounts for
                                        higher page counts to incentivize larger
                                        orders
                                    </li>
                                </ul>
                            </div>
                        </Alert>

                        {/* Warning if no thresholds */}
                        {data.volume_thresholds.length === 0 && (
                            <Alert
                                variant="warning"
                                title="No Volume Thresholds"
                                className="mt-3"
                                icon={AlertTriangle}
                            >
                                <p className="text-sm">
                                    You need to add at least one volume
                                    threshold to generate coupons.
                                </p>
                            </Alert>
                        )}
                    </form>
                </div>
            </div>
        </AdminLayout>
    );
};

export default GenerateVolume;
